<?php
namespace wbb\data\thread;
use wbb\data\post\Post;
use wbb\data\post\PostEditor;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\DatabaseObjectEditor;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\WCF;

/**
 * Provides functions to edit threads.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 * 
 * @method static	Thread		create(array $parameters = [])
 * @method		Thread		getDecoratedObject()
 * @mixin		Thread
 */
class ThreadEditor extends DatabaseObjectEditor {
	/**
	 * @inheritDoc
	 */
	protected static $baseClass = Thread::class;
	
	/**
	 * Adds the given post to this thread.
	 * 
	 * @param	Post	$post
	 */
	public function addPost(Post $post) {
		$data = ['attachments' => $this->attachments + $post->attachments];
		
		if (!$this->isDeleted && !$post->isDeleted) {
			$data['replies'] = $this->replies + 1;
		}
		
		if ($post->time > $this->lastPostTime || ($post->time == $this->lastPostTime && $post->postID > $this->lastPostID)) {
			$this->object->data['lastPostID'] = $data['lastPostID'] = $post->postID;
			$this->object->data['lastPoster'] = $data['lastPoster'] = $post->username;
			$this->object->data['lastPostTime'] = $data['lastPostTime'] = $post->time;
			$this->object->data['lastPosterID'] = $data['lastPosterID'] = $post->userID;
		}
		
		$this->update($data);
	}
	
	/**
	 * Enables this thread.
	 * 
	 * @param boolean $updateTime
	 */
	public function enable($updateTime = false) {
		$data = ['isDisabled' => 0];
		if ($updateTime) {
			$data['time'] = TIME_NOW;
		}
		
		$this->update($data);
	}
	
	/**
	 * Disables this thread.
	 */
	public function disable() {
		$this->update(['isDisabled' => 1]);
	}
	
	/**
	 * Updates the id of the first post of this thread.
	 */
	public function updateFirstPostID() {
		$sql = "SELECT		post.postID
			FROM		wbb".WCF_N."_post post
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = post.postID
					AND post.isDeleted = thread.isDeleted
					AND post.isDisabled = thread.isDisabled)
			WHERE		post.threadID = ?
			ORDER BY	post.time ASC, post.postID ASC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute([$this->threadID]);
		$row = $statement->fetchArray();
		
		$this->update(['firstPostID' => $row['postID']]);
		
		if ($this->firstPostID !== null) {
			PostEditor::addPostIDsToSearchIndex([$this->firstPostID, $row['postID']]);
		}
		else {
			PostEditor::addPostIDsToSearchIndex([$row['postID']]);
		}
	}
	
	/**
	 * Updates the last post of this thread.
	 */
	public function updateLastPost() {
		$sql = "SELECT		post.postID, post.time, post.userID, post.username
			FROM		wbb".WCF_N."_post post
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = post.threadID)
			WHERE		post.threadID = ?
					AND post.isDeleted = thread.isDeleted
					AND post.isDisabled = thread.isDisabled
			ORDER BY	post.time DESC, post.postID DESC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute([$this->threadID]);
		$row = $statement->fetchArray();
		
		// handle entirely deleted threads
		if ($row === false) {
			// get latest post's time to maintain thread sort order
			$sql = "SELECT		time
				FROM		wbb".WCF_N."_post
				WHERE		threadID = ?
				ORDER BY	time DESC, postID DESC";
			$statement = WCF::getDB()->prepareStatement($sql, 1);
			$statement->execute([
				$this->threadID
			]);
			$row = $statement->fetchArray();
			
			$row = [
				'postID' => null,
				'time' => $row['time'],
				'userID' => null,
				'username' => ''
			];
		}
		
		$this->update([
			'lastPostID' => $row['postID'],
			'lastPostTime' => $row['time'],
			'lastPosterID' => $row['userID'],
			'lastPoster' => $row['username']
		]);
	}
	
	/**
	 * Updates the board ids for an announcement.
	 * 
	 * @param	integer[]		$boardIDs
	 */
	public function updateAnnouncementBoards(array $boardIDs = []) {
		// remove old assigns
		$sql = "DELETE FROM	wbb".WCF_N."_thread_announcement
			WHERE		threadID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->threadID]);
		
		// assign new boards
		if (!empty($boardIDs)) {
			WCF::getDB()->beginTransaction();
			
			$sql = "INSERT INTO	wbb".WCF_N."_thread_announcement
						(boardID, threadID)
				VALUES		(?, ?)";
			$statement = WCF::getDB()->prepareStatement($sql);
			foreach ($boardIDs as $boardID) {
				$statement->execute([
					$boardID,
					$this->threadID
				]);
			}
			
			WCF::getDB()->commitTransaction();
		}
	}
	
	/**
	 * Sets the similar threads.
	 * 
	 * @param	integer[]		$similarThreadIDs
	 */
	public function setSimilarThreads(array $similarThreadIDs) {
		$sql = "INSERT IGNORE INTO	wbb".WCF_N."_thread_similar
						(threadID, similarThreadID)
			VALUES		        (?, ?)";
		$statement = WCF::getDB()->prepareStatement($sql);
		foreach ($similarThreadIDs as $similarThreadID) {
			$statement->execute([$this->threadID, $similarThreadID]);
		}
	}
	
	/**
	 * Rebuilds current thread, returns false if thread should be deleted
	 * but deletion was disabled by setting $disableDelete.
	 * 
	 * @param	boolean		$disableDelete
	 * @return	boolean
	 */
	public function rebuild($disableDelete = false) {
		// count number of associated posts
		$sql = "SELECT	COUNT(*) AS count
			FROM	wbb".WCF_N."_post
			WHERE	threadID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->threadID]);
		$row = $statement->fetchArray();
		
		// update thread
		if ($row['count']) {
			$updates = [];
			
			// update stats
			$sql = "SELECT	COUNT(*)
				FROM	wbb".WCF_N."_post
				WHERE	threadID = ?
					AND isDeleted = 0
					AND isDisabled = 0";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->threadID]);
			$posts = $statement->fetchColumn();
			$replies = ($posts - 1 > 0) ? $posts - 1 : 0;
			
			if ($this->replies != $replies) {
				$updates['replies'] = $replies;
			}
			
			if (!$posts) {
				// all posts are either disabled or deleted thus
				// check if the thread either needs to be disabled
				// or deleted
				$sql = "SELECT	COUNT(*)
					FROM	wbb".WCF_N."_post
					WHERE	threadID = ?
						AND isDisabled = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([
					$this->threadID,
					1
				]);
				$disabledPosts = $statement->fetchColumn();
				
				if ($disabledPosts) {
					$updates['isDisabled'] = 1;
				}
				else {
					// get deleteTime of the first post
					$sql = "SELECT		deleteTime
						FROM		wbb".WCF_N."_post
						WHERE		threadID = ?
								AND isDeleted = ?
						ORDER BY	time ASC, postID ASC";
					$statement = WCF::getDB()->prepareStatement($sql);
					$statement->execute([
						$this->threadID,
						1
					]);
					
					$updates['isDeleted'] = 1;
					$updates['deleteTime'] = $statement->fetchColumn();
				}
			}
			
			// calculate number of attachments
			$conditionBuilder = new PreparedStatementConditionBuilder();
			$conditionBuilder->add('post.threadID = ?', [$this->threadID]);
			$conditionBuilder->add('attachment.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.attachment.objectType', 'com.woltlab.wbb.post')->objectTypeID]);
			$conditionBuilder->add('post.isDeleted = ?', [0]);
			$conditionBuilder->add('post.isDisabled = ?', [0]);
			
			$sql = "SELECT		COUNT(*)
				FROM		wcf".WCF_N."_attachment attachment
				LEFT JOIN	wbb".WCF_N."_post post
				ON		(post.postID = attachment.objectID)
				".$conditionBuilder;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			$updates['attachments'] = $statement->fetchColumn();
			
			// calculate number of polls
			$conditionBuilder = new PreparedStatementConditionBuilder();
			$conditionBuilder->add('post.threadID = ?', [$this->threadID]);
			$conditionBuilder->add('poll.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.poll', 'com.woltlab.wbb.post')->objectTypeID]);
			$conditionBuilder->add('post.isDeleted = ?', [0]);
			$conditionBuilder->add('post.isDisabled = ?', [0]);
			
			$sql = "SELECT		COUNT(*)
				FROM		wcf".WCF_N."_poll poll
				LEFT JOIN	wbb".WCF_N."_post post
				ON		(post.postID = poll.objectID)
				".$conditionBuilder;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			$updates['polls'] = $statement->fetchColumn();
			
			// update data
			$this->update($updates);
			
			// update first and last post
			$this->updateFirstPostID();
			$this->updateLastPost();
		}
		else {
			// orphaned thread
			if ($disableDelete) {
				return false;
			}
			else {
				$threadAction = new ThreadAction([$this], 'delete');
				$threadAction->executeAction();
			}
		}
		
		return true;
	}
	
	/**
	 * Resets the similar threads of the given threads.
	 * 
	 * @param	integer[]		$threadIDs
	 */
	public static function resetSimilarThreads(array $threadIDs) {
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('threadID IN (?)', [$threadIDs]);
		$sql = "DELETE FROM	wbb".WCF_N."_thread_similar
			".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
	}
	
	/**
	 * Rebuilds thread data.
	 * 
	 * If a thread is marked as deleted or is disabled, counters like replies
	 * or attachments is set 0 since they only consider posts which are not
	 * deleted and not disabled.
	 * The first post of a thread (in the sense of a thread's firstPostID)
	 * always must have the same state as the thread itself. The same applies
	 * for the last post.
	 * 
	 * @param	integer[]		$threadIDs
	 */
	public static function rebuildThreadData(array $threadIDs) {
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('threadID IN (?)', [$threadIDs]);
		
		$sql = "SELECT          firstPostID, threadID 
			FROM            wbb".WCF_N."_thread thread 
			". $conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$oldFirstPostIDs = $statement->fetchMap('threadID', 'firstPostID');
		
		$sql = "UPDATE	wbb".WCF_N."_thread thread
			SET	firstPostID = COALESCE((
					SELECT		postID
					FROM		wbb".WCF_N."_post
					WHERE		threadID = thread.threadID
							AND isDisabled = thread.isDisabled
							AND isDeleted = thread.isDeleted
					ORDER BY	time ASC, postID ASC
					LIMIT		1
				), firstPostID),
				replies = IF(thread.isDeleted = 0 AND thread.isDisabled = 0, (
					SELECT		COUNT(*)
					FROM		wbb".WCF_N."_post
					WHERE		threadID = thread.threadID
							AND isDisabled = 0
							AND isDeleted = 0
				) - 1, 0),
				lastPostID = COALESCE((
					SELECT		postID
					FROM		wbb".WCF_N."_post
					WHERE		threadID = thread.threadID
							AND isDisabled = thread.isDisabled
							AND isDeleted = thread.isDeleted
					ORDER BY	time DESC, postID DESC
					LIMIT		1
				), lastPostID)
			".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		
		$sql = "SELECT          firstPostID, threadID 
			FROM            wbb".WCF_N."_thread thread 
			". $conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$newFirstPostIDs = $statement->fetchMap('threadID', 'firstPostID');
		
		PostEditor::addPostIDsToSearchIndex(array_merge($oldFirstPostIDs, $newFirstPostIDs));
		
		// update like counter
		$sql = "UPDATE	wbb".WCF_N."_thread thread
			SET	cumulativeLikes = COALESCE((
					SELECT	cumulativeLikes
					FROM	wbb".WCF_N."_post
					WHERE	postID = thread.firstPostID
				), 0)
			".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		
		// calculate number of attachments
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('post.threadID IN (?)', [$threadIDs]);
		$conditionBuilder->add('attachment.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.attachment.objectType', 'com.woltlab.wbb.post')->objectTypeID]);
		$conditionBuilder->add('post.isDeleted = ?', [0]);
		$conditionBuilder->add('post.isDisabled = ?', [0]);
		$sql = "SELECT		post.threadID, COUNT(*) AS attachments
			FROM		wcf".WCF_N."_attachment attachment
			LEFT JOIN	wbb".WCF_N."_post post
			ON		(post.postID = attachment.objectID)
			".$conditionBuilder."
			GROUP BY	post.threadID";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$attachments = $statement->fetchMap('threadID', 'attachments');
		
		// calculate number of polls
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('post.threadID IN (?)', [$threadIDs]);
		$conditionBuilder->add('poll.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.poll', 'com.woltlab.wbb.post')->objectTypeID]);
		$conditionBuilder->add('post.isDeleted = ?', [0]);
		$conditionBuilder->add('post.isDisabled = ?', [0]);
		
		$sql = "SELECT		post.threadID, COUNT(*) AS polls
			FROM		wcf".WCF_N."_poll poll
			LEFT JOIN	wbb".WCF_N."_post post
			ON		(post.postID = poll.objectID)
			".$conditionBuilder."
			GROUP BY	post.threadID";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$polls = $statement->fetchMap('threadID', 'polls');
		
		// update threads
		$sql = "UPDATE	wbb".WCF_N."_thread thread
			SET	attachments = ?,
				polls = ?
			WHERE	threadID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		foreach ($threadIDs as $threadID) {
			$statement->execute([
				isset($attachments[$threadID]) ? $attachments[$threadID] : 0,
				isset($polls[$threadID]) ? $polls[$threadID] : 0,
				$threadID
			]);
		}
	}
}
