<?php
namespace wbb\data\thread;
use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardEditor;
use wbb\data\board\ModeratorBoardNodeList;
use wbb\data\modification\log\BoardModificationLogList;
use wbb\data\post\Post;
use wbb\data\post\PostAction;
use wbb\data\post\PostEditor;
use wbb\data\post\PostList;
use wbb\data\post\SimplifiedViewablePostList;
use wbb\data\post\ThreadPostList;
use wbb\data\post\ViewablePostList;
use wbb\system\cache\runtime\PostRuntimeCache;
use wbb\system\label\object\ThreadLabelObjectHandler;
use wbb\system\log\modification\PostModificationLogHandler;
use wbb\system\log\modification\ThreadModificationLogHandler;
use wbb\system\thread\ThreadHandler;
use wbb\system\user\notification\object\ThreadUserNotificationObject;
use wcf\data\article\content\ArticleContent;
use wcf\data\article\content\ArticleContentEditor;
use wcf\data\article\Article;
use wcf\data\IPopoverAction;
use wcf\data\label\Label;
use wcf\data\modification\log\ModificationLogList;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\user\object\watch\UserObjectWatchList;
use wcf\data\AbstractDatabaseObjectAction;
use wcf\data\IClipboardAction;
use wcf\data\ISearchAction;
use wcf\data\IVisitableObjectAction;
use wcf\data\user\UserAction;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\html\input\HtmlInputProcessor;
use wcf\system\label\LabelHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\request\LinkHandler;
use wcf\system\tagging\TagEngine;
use wcf\system\user\activity\event\UserActivityEventHandler;
use wcf\system\user\activity\point\UserActivityPointHandler;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\user\object\watch\UserObjectWatchHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;
use wcf\util\ArrayUtil;
use wcf\util\StringUtil;

/**
 * Executes thread-related actions.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 * 
 * @method	ThreadEditor[]		getObjects()
 * @method	ThreadEditor		getSingleObject()
 */
class ThreadAction extends AbstractDatabaseObjectAction implements IClipboardAction, IPopoverAction, ISearchAction, IVisitableObjectAction {
	/**
	 * @inheritDoc
	 */
	protected $className = ThreadEditor::class;
	
	/**
	 * @inheritDoc
	 */
	protected $allowGuestAccess = [
		'countReplies',
		'getNewPosts',
		'getPopover',
		'getPostPreview',
		'getSimilarThreads',
		'markAsRead',
		'validateThreadFormValues',
	];
	
	/**
	 * active board object
	 * @var	Board
	 */
	public $board;
	
	/**
	 * thread object
	 * @var	Thread|ThreadEditor
	 */
	public $thread;
	
	/**
	 * list of thread data
	 * @var	mixed[][]
	 */
	public $threadData = [];
	
	/**
	 * thread editor object
	 * @var	ThreadEditor
	 */
	public $threadEditor;
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 * @return	Thread
	 */
	public function create() {
		// get board
		$board = isset($this->parameters['board']) ? $this->parameters['board'] : BoardCache::getInstance()->getBoard($this->parameters['data']['boardID']);
		
		// create thread
		$data = $this->parameters['data'];
		if (!isset($data['time'])) $data['time'] = TIME_NOW;
		if (!array_key_exists('userID', $data)) {
			$data['userID'] = WCF::getUser()->userID;
			$data['username'] = WCF::getUser()->username;
		}
		$data['lastPosterID'] = $data['userID'];
		$data['lastPoster'] = $data['username'];
		$data['lastPostTime'] = $data['time'];
		
		// count attachments
		if (isset($this->parameters['attachmentHandler']) && $this->parameters['attachmentHandler'] !== null) {
			$data['attachments'] = count($this->parameters['attachmentHandler']);
		}
		
		/** @var Thread $thread */
		$thread = call_user_func([$this->className, 'create'], $data);
		
		// create post
		$postData = $this->parameters['postData'];
		$postData['threadID'] = $thread->threadID;
		$postData['subject'] = $this->parameters['data']['topic'];
		$postData['time'] = $this->parameters['data']['time'];
		$postData['userID'] = $this->parameters['data']['userID'];
		$postData['username'] = $this->parameters['data']['username'];
		$postData['isDisabled'] = $thread->isDisabled;
		
		$postCreateParameters = [
			'data' => $postData,
			'thread' => $thread,
			'board' => $board,
			'isFirstPost' => true,
			'attachmentHandler' => isset($this->parameters['attachmentHandler']) ? $this->parameters['attachmentHandler'] : null,
			'htmlInputProcessor' => isset($this->parameters['htmlInputProcessor']) ? $this->parameters['htmlInputProcessor'] : null, 
			'optionHandler' => isset($this->parameters['optionHandler']) ? $this->parameters['optionHandler'] : null
		];
		if (isset($this->parameters['subscribeThread'])) $postCreateParameters['subscribeThread'] = $this->parameters['subscribeThread'];
		
		$postAction = new PostAction([], 'create', $postCreateParameters);
		$resultValues = $postAction->executeAction();
		
		// update first post
		$threadEditor = new ThreadEditor($thread);
		$threadEditor->update([
			'firstPostID' => $resultValues['returnValues']->postID,
			'lastPostID' => $resultValues['returnValues']->postID
		]);
		
		// reindex post with thread form values
		PostEditor::addPostIDsToSearchIndex([$resultValues['returnValues']->postID]);
		
		// handle announcements
		if (isset($this->parameters['announcementBoardIDs'])) {
			$threadEditor->updateAnnouncementBoards($this->parameters['announcementBoardIDs']);
		}
		
		// set language id (cannot be zero)
		$languageID = (!isset($this->parameters['data']['languageID']) || ($this->parameters['data']['languageID'] === null)) ? LanguageFactory::getInstance()->getDefaultLanguageID() : $this->parameters['data']['languageID'];
		
		// save tags
		if (!empty($this->parameters['tags'])) {
			TagEngine::getInstance()->addObjectTags('com.woltlab.wbb.thread', $thread->threadID, $this->parameters['tags'], $languageID);
		}
		
		// update similar threads
		if (WBB_THREAD_ENABLE_SIMILAR_THREADS) {
			$action = new ThreadAction([$threadEditor], 'updateSimilarThreads');
			$action->executeAction();
		}
		
		if (!$thread->isDisabled) {
			$action = new ThreadAction([$threadEditor->threadID], 'triggerPublication');
			$action->executeAction();
		}
		
		return $thread;
	}
	
	/**
	 * @inheritDoc
	 * 
	 * In bulk processing mode, updating `isDeleted` does not also automatically trash or restore
	 * the thread as it would happen in non-bulk processing mode.
	 */
	public function update() {
		$isBulkProcessing = $this->isBulkProcessing();
		$ignoreThreadModificationLogs = (isset($this->parameters['ignoreThreadModificationLogs']) && $this->parameters['ignoreThreadModificationLogs']);
		$ignoreCounterUpdate = (isset($this->parameters['ignoreCounterUpdate']) && $this->parameters['ignoreCounterUpdate']);
		
		$reason = '';
		if (isset($this->parameters['data']['reason'])) {
			$reason = $this->parameters['data']['reason'];
			unset($this->parameters['data']['reason']);
		}
		$tags = null;
		if (isset($this->parameters['data']['tags'])) {
			$tags = $this->parameters['data']['tags'];
			unset($this->parameters['data']['tags']);
		}
		
		// set delete time
		if (isset($this->parameters['data']['isDeleted'])) {
			$this->parameters['data']['deleteTime'] = isset($this->parameters['data']['isDeleted']) ? TIME_NOW : 0;
		}
		
		parent::update();
		
		// load current tags of threads if tags will be updated
		$threadTags = [];
		if ($tags !== null) {
			$threadIDs = [];
			foreach ($this->getObjects() as $thread) {
				$threadIDs[] = $thread->threadID;
			}
			
			$threadTags = TagEngine::getInstance()->getObjectsTags('com.woltlab.wbb.thread', $threadIDs);
		}
		
		$trashPostsThreadIDs = $rebuildLastPostBoardIDs = $restorePostsThreadIDs = $deleteTagsThreadIDs = [];
		
		/** @var ThreadEditor $thread */
		foreach ($this->getObjects() as $thread) {
			// handle announcements
			if (isset($this->parameters['announcementBoardIDs'])) {
				$thread->updateAnnouncementBoards($this->parameters['announcementBoardIDs']);
			}
			
			// enable
			if (!$isBulkProcessing) {
				if (isset($this->parameters['data']['isDisabled']) && !$this->parameters['data']['isDisabled']) {
					if (!$ignoreThreadModificationLogs) {
						ThreadModificationLogHandler::getInstance()->enable($thread->getDecoratedObject());
					}
					
					if (!$ignoreCounterUpdate) {
						$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($thread->boardID));
						$boardEditor->updateCounters(['threads' => 1, 'posts' => $thread->replies + 1]);
					}	
				}
				
				// isClosed
				if (isset($this->parameters['data']['isClosed']) && !$ignoreThreadModificationLogs) {
					if ($this->parameters['data']['isClosed']) {
						ThreadModificationLogHandler::getInstance()->close($thread->getDecoratedObject());
					}
					else {
						ThreadModificationLogHandler::getInstance()->open($thread->getDecoratedObject());
					}
				}
				
				// isDeleted
				if (isset($this->parameters['data']['isDeleted'])) {
					if ($this->parameters['data']['isDeleted']) {
						if (!$ignoreThreadModificationLogs) {
							ThreadModificationLogHandler::getInstance()->trash($thread->getDecoratedObject(), $reason);
						}
						$trashPostsThreadIDs[] = $thread->threadID;
					}
					else {
						if (!$ignoreThreadModificationLogs) {
							ThreadModificationLogHandler::getInstance()->restore($thread->getDecoratedObject());
						}
						$restorePostsThreadIDs[] = $thread->threadID;
					}
				}
			}
			
			// update topic
			if (isset($this->parameters['data']['topic'])) {
				$postAction = new PostAction([$thread->firstPostID], 'update', [
					'data' => [
						'subject' => $this->parameters['data']['topic']
					],
					'isBulkProcessing' => $isBulkProcessing
				]);
				$postAction->executeAction();
				
				// log updated topic
				if (!$isBulkProcessing && $thread->topic != $this->parameters['data']['topic'] && !$ignoreThreadModificationLogs) {
					ThreadModificationLogHandler::getInstance()->changeTopic($thread->getDecoratedObject(), $this->parameters['data']['topic']);
				}
			}
			
			// update tags
			if ($tags !== null) {
				// set language id (cannot be zero)
				$languageID = (!isset($this->parameters['data']['languageID']) || ($this->parameters['data']['languageID'] === null)) ? LanguageFactory::getInstance()->getDefaultLanguageID() : $this->parameters['data']['languageID'];
				TagEngine::getInstance()->addObjectTags('com.woltlab.wbb.thread', $thread->threadID, $tags, $languageID);
			}
			else if (!empty($threadTags[$thread->threadID])) {
				$deleteTagsThreadIDs[] = $thread->threadID;
			}
			
			if (!$isBulkProcessing) {
				// language was changed
				if (isset($this->parameters['data']['languageID']) && $this->parameters['data']['languageID'] != $thread->languageID) {
					$rebuildLastPostBoardIDs[] = $thread->boardID;
				}
				
				if (!$ignoreThreadModificationLogs) {
					if (isset($this->parameters['data']['isSticky']) && $thread->isSticky != $this->parameters['data']['isSticky']) {
						if ($this->parameters['data']['isSticky']) {
							ThreadModificationLogHandler::getInstance()->sticky($thread->getDecoratedObject());
						}
						else {
							ThreadModificationLogHandler::getInstance()->scrape($thread->getDecoratedObject());
						}
					}
					
					if (isset($this->parameters['data']['isAnnouncement']) && $thread->isAnnouncement != $this->parameters['data']['isAnnouncement']) {
						if ($this->parameters['data']['isAnnouncement']) {
							ThreadModificationLogHandler::getInstance()->setAsAnnouncement($thread->getDecoratedObject());
						}
						else {
							ThreadModificationLogHandler::getInstance()->unsetAsAnnouncement($thread->getDecoratedObject());
						}
					}
				}
			}
		}
		
		if (!empty($deleteTagsThreadIDs)) {
			TagEngine::getInstance()->deleteObjects('com.woltlab.wbb.thread', $deleteTagsThreadIDs);
		}
		
		if (!$isBulkProcessing) {
			if (!empty($trashPostsThreadIDs)) {
				$this->trashPosts($trashPostsThreadIDs);
			}
			
			if (!empty($restorePostsThreadIDs)) {
				$this->restorePosts($restorePostsThreadIDs);
			}
			
			if (!empty($rebuildLastPostBoardIDs)) {
				$rebuildLastPostBoardIDs = array_unique($rebuildLastPostBoardIDs);
				foreach ($rebuildLastPostBoardIDs as $boardID) {
					$board = BoardCache::getInstance()->getBoard($boardID);
					$boardEditor = new BoardEditor($board);
					$boardEditor->updateLastPost();
				}
				
				BoardEditor::resetDataCache();
			}
		}
	}
	
	/**
	 * Checks the permissions for the given thread ids.
	 * 
	 * @param	string[]	$permissions
	 * @throws	UserInputException
	 */
	protected function checkPermissions(array $permissions = ['canViewBoard', 'canEnterBoard']) {
		// read data
		if (empty($this->objects)) {
			$this->readObjects();
			
			if (empty($this->objects)) {
				throw new UserInputException('objectIDs');
			}
		}
		
		// check permissions
		foreach ($this->getObjects() as $thread) {
			BoardCache::getInstance()->getBoard($thread->boardID)->checkPermission($permissions);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function markAsRead() {
		if (empty($this->parameters['visitTime'])) {
			$this->parameters['visitTime'] = TIME_NOW;
		}
		
		if (!count($this->objects)) {
			$this->readObjects();
		}
		
		$threadIDs = [];
		foreach ($this->getObjects() as $thread) {
			$threadIDs[] = $thread->threadID;
			VisitTracker::getInstance()->trackObjectVisit('com.woltlab.wbb.thread', $thread->threadID, $this->parameters['visitTime']);
		}
		
		if (WCF::getUser()->userID) {
			// mark obsolete notifications as confirmed
			if (!empty($threadIDs)) {
				$events = [];
				foreach (UserNotificationHandler::getInstance()->getEvents('com.woltlab.wbb.post') as $event) {
					$events[$event->eventID] = $event->eventName;
				}
				
				$conditionBuilder = new PreparedStatementConditionBuilder();
				$conditionBuilder->add('notification.eventID IN (?)', [array_keys($events)]);
				$conditionBuilder->add('notification.objectID = post.postID');
				$conditionBuilder->add('notification.userID = ?', [WCF::getUser()->userID]);
				$conditionBuilder->add('post.threadID IN (?)', [$threadIDs]);
				$conditionBuilder->add('post.time <= ?', [$this->parameters['visitTime']]);
				
				$sql = "SELECT		post.postID, notification.eventID
					FROM		wbb".WCF_N."_post post,
							wcf".WCF_N."_user_notification notification
					".$conditionBuilder;
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($conditionBuilder->getParameters());
				$postIDs = [];
				while ($row = $statement->fetchArray()) {
					if (!isset($postIDs[$events[$row['eventID']]])) $postIDs[$events[$row['eventID']]] = [];
					$postIDs[$events[$row['eventID']]][] = $row['postID'];
				}
				
				if (!empty($postIDs)) {
					foreach ($postIDs as $event => $eventPostIDs) {
						UserNotificationHandler::getInstance()->markAsConfirmed($event, 'com.woltlab.wbb.post', [WCF::getUser()->userID], $eventPostIDs);
					}
				}
				
				if (MODULE_LIKE && isset($this->parameters['markLikeNotificationsAsConfirmed']) && $this->parameters['markLikeNotificationsAsConfirmed']) {
					$threadAction = new ThreadAction($threadIDs, 'markLikeNotificationsAsConfirmed');
					$threadAction->executeAction();
				}
				
				if (WBB_THREAD_ENABLE_MODERATION_NOTIFICATION && isset($this->parameters['markModerationNotificationsAsConfirmed']) && $this->parameters['markModerationNotificationsAsConfirmed']) {
					$threadAction = new ThreadAction($threadIDs, 'markModerationNotificationsAsConfirmed');
					$threadAction->executeAction();
				}
				
				// mark board notifications as read
				UserNotificationHandler::getInstance()->markAsConfirmed('thread', 'com.woltlab.wbb.thread', [ WCF::getUser()->userID ], $threadIDs);
			}
			
			// reset storage
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadThreads');
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedBoards');
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedThreads');
		}
	}
	
	/**
	 * Marks like notifications as confirmed.
	 */
	public function markLikeNotificationsAsConfirmed() {
		$events = [];
		foreach (UserNotificationHandler::getInstance()->getEvents('com.woltlab.wbb.likeablePost.notification') as $event) {
			$events[$event->eventID] = $event->eventName;
		}
		
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('notification.eventID IN (?)', [array_keys($events)]);
		$conditionBuilder->add('notification.baseObjectID = post.postID');
		$conditionBuilder->add('notification.userID = ?', [WCF::getUser()->userID]);
		$conditionBuilder->add('post.threadID IN (?)', [$this->objectIDs]);
		
		$sql = "SELECT	notification.objectID, notification.eventID
			FROM	wbb".WCF_N."_post post,
				wcf".WCF_N."_user_notification notification
				".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$objectIDs = [];
		while ($row = $statement->fetchArray()) {
			if (!isset($objectIDs[$events[$row['eventID']]])) $objectIDs[$events[$row['eventID']]] = [];
			$objectIDs[$events[$row['eventID']]][] = $row['objectID'];
		}
		
		if (!empty($objectIDs)) {
			foreach ($objectIDs as $event => $eventObjectIDs) {
				UserNotificationHandler::getInstance()->markAsConfirmed($event, 'com.woltlab.wbb.likeablePost.notification', [WCF::getUser()->userID], $eventObjectIDs);
			}
		}
	}
	
	/**
	 * Marks moderation notifications as confirmed.
	 */
	public function markModerationNotificationsAsConfirmed() {
		// init vars
		$eventIDToObjectType = [];
		$events = [];
		
		// load moderation notification events
		$eventIDToObjectType[UserNotificationHandler::getInstance()->getEvent('com.woltlab.wbb.moderation.thread', 'moderate')->eventID] = 'com.woltlab.wbb.moderation.thread';
		$events[] = UserNotificationHandler::getInstance()->getEvent('com.woltlab.wbb.moderation.thread', 'moderate')->eventID;
		
		$eventIDToObjectType[UserNotificationHandler::getInstance()->getEvent('com.woltlab.wbb.moderation.post', 'moderate')->eventID] = 'com.woltlab.wbb.moderation.post';
		$events[] = UserNotificationHandler::getInstance()->getEvent('com.woltlab.wbb.moderation.post', 'moderate')->eventID;
		
		// filter notifications
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('notification.eventID IN (?)', [$events]);
		$conditionBuilder->add('notification.objectID = modification_log.logID');
		$conditionBuilder->add('notification.userID = ?', [WCF::getUser()->userID]);
		$conditionBuilder->add('notification.confirmTime = ?', [0]);
		
		$threadConditionBuilder = new PreparedStatementConditionBuilder(false);
		$threadConditionBuilder->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.thread')]);
		$threadConditionBuilder->add("modification_log.objectID = ?", [$this->objectIDs]);
		
		$postConditionBuilder = new PreparedStatementConditionBuilder(false);
		$postConditionBuilder->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.post')]);
		$postConditionBuilder->add("modification_log.objectID IN (SELECT postID FROM wbb" . WCF_N . "_post WHERE threadID = ?)", [$this->objectIDs]);
		
		$modificationLogConditionBuilder = new PreparedStatementConditionBuilder(false, 'OR');
		$modificationLogConditionBuilder->add('(' . $threadConditionBuilder . ')', $threadConditionBuilder->getParameters());
		$modificationLogConditionBuilder->add('(' . $postConditionBuilder . ')', $postConditionBuilder->getParameters());
		
		$conditionBuilder->add('(' . $modificationLogConditionBuilder . ')', $modificationLogConditionBuilder->getParameters());
		
		$sql = "SELECT		modification_log.logID, notification.eventID
			FROM		wcf" . WCF_N . "_modification_log modification_log,
					wcf" . WCF_N . "_user_notification notification
			" . $conditionBuilder;
		
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		
		// mark notification as read
		$logIDs = [];
		while ($row = $statement->fetchArray()) {
			if (!isset($logIDs[$row['eventID']])) {
				$logIDs[$row['eventID']] = [];
			}
			
			$logIDs[$row['eventID']][] = $row['logID'];
		}
		
		if (!empty($logIDs)) {
			foreach ($logIDs as $eventID => $logID) {
				UserNotificationHandler::getInstance()->markAsConfirmed('moderate', $eventIDToObjectType[$eventID], [WCF::getUser()->userID], $logID);
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateMarkAsRead() {
		$this->checkPermissions();
	}
	
	/**
	 * Validates parameters to mark threads as done.
	 */
	public function validateMarkAsDone() {
		if (empty($this->objects)) {
			$this->readObjects();
			if (empty($this->objects)) {
				throw new UserInputException('objectIDs');
			}
		}
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->isDone) {
				throw new UserInputException('objectIDs');
			}
			else if (!$thread->canMarkAsDone()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Marks threads as done.
	 */
	public function markAsDone() {
		foreach ($this->getObjects() as $thread) {
			$thread->update(['isDone' => 1]);
		}
	}
	
	/**
	 * Marks threads as undone.
	 *
	 * @since	5.0
	 */
	public function markAsUndone() {
		foreach ($this->getObjects() as $thread) {
			$thread->update(['isDone' => 0]);
		}
	}
	
	/**
	 * Validates parameters to mark posts as best answer.
	 * 
	 * @since       5.2
	 */
	public function validateMarkAsBestAnswer() {
		$this->readInteger('threadID');
		$this->readInteger('postID');
		
		$this->thread = new Thread($this->parameters['threadID']);
		$post = PostRuntimeCache::getInstance()->getObject($this->parameters['postID']);
		$post->setThread($this->thread);
		
		if (!$this->thread->threadID) {
			throw new UserInputException('threadID');
		}
		
		if (!$this->thread->canMarkBestAnswer()) {
			throw new PermissionDeniedException();
		}
		
		if ($post->threadID !== $this->thread->threadID || $post->isFirstPost()) {
			throw new UserInputException('postID');
		}
	}
	
	/**
	 * Marks posts as best answer.
	 * 
	 * @since       5.2
	 */
	public function markAsBestAnswer() {
		$post = PostRuntimeCache::getInstance()->getObject($this->parameters['postID']);
		if ($post->userID) {
			(new UserAction([$post->userID], 'update', [
				'counters' => [
					'wbbBestAnswers' => 1,
				],
			]))->executeAction();
		}
		
		$threadEditor = new ThreadEditor($this->thread);
		$threadEditor->update([
			'bestAnswerPostID' => $this->parameters['postID']
		]);
	}
	
	/**
	 * Validates parameters to unmark posts as best answer.
	 * 
	 * @since       5.2
	 */
	public function validateUnmarkAsBestAnswer() {
		$this->validateMarkAsBestAnswer();
	}
	
	/**
	 * Unmarks posts as best answer.
	 * 
	 * @since       5.2
	 */
	public function unmarkAsBestAnswer() {
		$post = PostRuntimeCache::getInstance()->getObject($this->parameters['postID']);
		if ($post->userID) {
			(new UserAction([$post->userID], 'update', [
				'counters' => [
					'wbbBestAnswers' => -1,
				],
			]))->executeAction();
		}
		
		$threadEditor = new ThreadEditor($this->thread);
		$threadEditor->update([
			'bestAnswerPostID' => null
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateGetPopover() {
		$this->thread = $this->getSingleObject();
		$permissions = ['canViewBoard', 'canEnterBoard', 'canReadThread'];
		if ($this->thread->getMovedThread()) {
			// run all permission checks against the actual thread
			BoardCache::getInstance()->getBoard($this->thread->getMovedThread()->boardID)->checkPermission($permissions);
			
			if (!$this->thread->getMovedThread()->canRead()) {
				throw new PermissionDeniedException();
			}
		}
		
		// check if board may be entered and thread can be read
		$this->checkPermissions($permissions);
		
		if (!$this->thread->canRead()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getPopover() {
		$postList = new SimplifiedViewablePostList();
		if (isset($this->parameters['sortOrder']) && $this->parameters['sortOrder'] == 'DESC') {
			$postList->getConditionBuilder()->add("post.postID = ?", [$this->thread->lastPostID]);
		}
		else {
			$postList->getConditionBuilder()->add("post.postID = ?", [$this->thread->firstPostID]);
			
			// get labels
			$assignedLabels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$this->thread->threadID]);
			if (!empty($assignedLabels[$this->thread->threadID])) {
				WCF::getTPL()->assign([
					'labels' => $assignedLabels[$this->thread->threadID]
				]);
			}
		}
		
		$postList->readObjects();
		$post = $postList->getSingleObject();
		if ($post === null) {
			return [];
		}
		
		return [
			'template' => WCF::getTPL()->fetch('postPreview', 'wbb', [
				'post' => $post,
			]),
		];
	}
	
	/**
	 * Returns a preview of a post in a specific thread.
	 * 
	 * @return	string[]
	 * @deprecated  5.3     Use `getPopover()` instead.
	 */
	public function getPostPreview() {
		return $this->getPopover();
	}
	
	/**
	 * Validates the get post preview action.
	 * 
	 * @deprecated  5.3     Use `validateGetPopover()` instead.
	 */
	public function validateGetPostPreview() {
		$this->validateGetPopover();
	}
	
	/**
	 * @deprecated	5.0, remember to also remove the method from $allowGuestAccess
	 */
	public function countReplies() {
		foreach ($this->objectIDs as $objectID) {
			$thread = new Thread($objectID);
			return $thread->replies;
		}
		
		return 0;
	}
	
	/**
	 * Validates the count replies action.
	 * @deprecated	5.0
	 */
	public function validateCountReplies() {
		$this->checkPermissions(['canViewBoard', 'canEnterBoard', 'canReadThread']);
	}
	
	/**
	 * Validates permissions to edit a thread.
	 */
	public function validateBeginEdit() {
		if (!isset($this->parameters['data']['threadID'])) {
			throw new UserInputException('threadID');
		}
		
		$thread = new Thread($this->parameters['data']['threadID']);
		if (!$thread->threadID) {
			throw new UserInputException('threadID');
		}
		
		if (!$thread->canEdit()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * Returns the thread edit form components.
	 * 
	 * @return	string[]
	 */
	public function beginEdit() {
		return [
			'actionName' => 'beginEdit',
			'template' => ThreadHandler::getInstance()->beginEdit($this->parameters['data']['threadID']),
			'threadID' => $this->parameters['data']['threadID']
		];
	}
	
	/**
	 * Validates permissions to edit a thread.
	 */
	public function validateSaveEdit() {
		$this->validateBeginEdit();
		
		if (!isset($this->parameters['data']['values']) || !is_array($this->parameters['data']['values'])) {
			throw new UserInputException('values');
		}
	}
	
	/**
	 * Edits a thread.
	 * 
	 * @return	array
	 */
	public function saveEdit() {
		$returnValues = ThreadHandler::getInstance()->saveEdit($this->parameters['data']['threadID'], $this->parameters['data']['values']);
		
		$returnValues['actionName'] = 'saveEdit';
		return $returnValues;
	}
	
	/**
	 * Validates parameters to mark threads as done.
	 */
	public function validateDone() {
		if (!WBB_MODULE_THREAD_MARKING_AS_DONE) {
			throw new PermissionDeniedException();
		}
		
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->isDone) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canMarkAsDoneThread')) {
				if (!$thread->getBoard()->getPermission('canMarkAsDoneOwnThread')) {
					throw new PermissionDeniedException();
				}
				else if ($thread->userID != WCF::getUser()->userID) {
					throw new PermissionDeniedException();
				}
			}
		}
	}
	
	/**
	 * Marks threads as done.
	 * 
	 * @return	array
	 */
	public function done() {
		foreach ($this->getObjects() as $thread) {
			$thread->update([
				'isDone' => 1
			]);
			
			$this->addThreadData($thread->getDecoratedObject(), 'isDone', 1);
			ThreadModificationLogHandler::getInstance()->done($thread->getDecoratedObject());
		}
		
		$this->unmarkItems();
		
		return $this->getThreadData();
	}
	
	/**
	 * Validates parameters to mark threads as undone.
	 */
	public function validateUndone() {
		if (!WBB_MODULE_THREAD_MARKING_AS_DONE) {
			throw new PermissionDeniedException();
		}
		
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->isDone) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canMarkAsDoneThread')) {
				if (!$thread->getBoard()->getPermission('canMarkAsDoneOwnThread')) {
					throw new PermissionDeniedException();
				}
				else if ($thread->userID != WCF::getUser()->userID) {
					throw new PermissionDeniedException();
				}
			}
		}
	}
	
	/**
	 * Marks threads as undone.
	 */
	public function undone() {
		foreach ($this->getObjects() as $thread) {
			$thread->update([
				'isDone' => 0
			]);
			
			$this->addThreadData($thread->getDecoratedObject(), 'isDone', 0);
			ThreadModificationLogHandler::getInstance()->undone($thread->getDecoratedObject());
		}
		
		$this->unmarkItems();
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters for closing threads.
	 */
	public function validateClose() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->isClosed) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canCloseThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Closes given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function close() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => ['isClosed' => 1],
			'isBulkProcessing' => $isBulkProcessing,
			'ignoreThreadModificationLogs' => true
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			foreach ($this->getObjects() as $thread) {
				$this->addThreadData($thread->getDecoratedObject(), 'isClosed', 1);
				
				ThreadModificationLogHandler::getInstance()->close($thread->getDecoratedObject());
			}
			
			$this->unmarkItems();
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters for opening threads.
	 */
	public function validateOpen() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->isClosed) {
				throw new UserInputException('objectIDs');
			}
				
			if (!$thread->getBoard()->getModeratorPermission('canCloseThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Opens given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function open() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => ['isClosed' => 0],
			'isBulkProcessing' => $isBulkProcessing,
			'ignoreThreadModificationLogs' => true
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			foreach ($this->getObjects() as $thread) {
				$this->addThreadData($thread->getDecoratedObject(), 'isClosed', 0);
				
				ThreadModificationLogHandler::getInstance()->open($thread->getDecoratedObject());
			}
			
			$this->unmarkItems();
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validates parameters to pin threads.
	 */
	public function validateSticky() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->isSticky || $thread->isAnnouncement) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canPinThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Pins threads.
	 * 
	 * @return	mixed[][]
	 */
	public function sticky() {
		foreach ($this->getObjects() as $thread) {
			$thread->update([
				'isSticky' => 1
			]);
			
			$this->addThreadData($thread->getDecoratedObject(), 'isSticky', 1);
			ThreadModificationLogHandler::getInstance()->sticky($thread->getDecoratedObject());
		}
		
		$this->unmarkItems();
		
		return $this->getThreadData();
	}
	
	/**
	 * Validates parameters to unpin threads.
	 */
	public function validateScrape() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->isSticky) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canPinThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Unpins threads.
	 * 
	 * @return	mixed[][]
	 */
	public function scrape() {
		foreach ($this->getObjects() as $thread) {
			$thread->update([
				'isSticky' => 0
			]);
			
			$this->addThreadData($thread->getDecoratedObject(), 'isSticky', 0);
			ThreadModificationLogHandler::getInstance()->scrape($thread->getDecoratedObject());
		}
		
		$this->unmarkItems();
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters for trashing threads.
	 */
	public function validateTrash() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canDeleteThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Trashes given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function trash() {
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		$isBulkProcessing = (isset($this->parameters['isBulkProcessing']) && $this->parameters['isBulkProcessing']);
		$ignorePosts = (isset($this->parameters['ignorePosts']) && $this->parameters['ignorePosts']);
		
		$boardIDs = $threadIDs = $boardStats = [];
		$reason = isset($this->parameters['data']['reason']) ? StringUtil::trim($this->parameters['data']['reason']) : '';
		
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => [
				'deleteTime' => TIME_NOW,
				'isDeleted' => 1
			],
			'isBulkProcessing' => $isBulkProcessing,
			'ignoreThreadModificationLogs' => true
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			foreach ($this->getObjects() as $thread) {
				$this->addThreadData($thread->getDecoratedObject(), 'isDeleted', 1);
				
				ThreadModificationLogHandler::getInstance()->trash($thread->getDecoratedObject(), $reason);
				
				$boardIDs[] = $thread->boardID;
				$threadIDs[] = $thread->threadID;
				
				if (!$thread->isDisabled) {
					if (!isset($boardStats[$thread->boardID])) {
						$boardStats[$thread->boardID] = [
							'threads' => 0
						];
						if (!$ignorePosts) {
							$boardStats[$thread->boardID]['posts'] = 0;
						}
					}
					$boardStats[$thread->boardID]['threads']--;
					
					if (!$ignorePosts) {
						$boardStats[$thread->boardID]['posts'] -= $thread->replies + 1;
					}
				}
			}
		}
		
		if (!$ignorePosts) {
			$this->trashPosts($this->objectIDs, $isBulkProcessing);
		}
		
		if (!$isBulkProcessing) {
			ThreadEditor::rebuildThreadData($threadIDs);
			
			// update board counters
			if (!empty($boardStats)) {
				foreach ($boardStats as $boardID => $stats) {
					$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
					$boardEditor->updateCounters($stats);
				}
			}
			
			// get delete notes
			$logList = new BoardModificationLogList();
			$logList->setThreadData($threadIDs, 'trash');
			$logList->getConditionBuilder()->add("modification_log.time = ?", [TIME_NOW]);
			$logList->readObjects();
			$logEntries = [];
			foreach ($logList as $logEntry) {
				$logEntries[$logEntry->objectID] = $logEntry->__toString();
			}
			
			foreach ($this->getObjects() as $thread) {
				$this->addThreadData($thread->getDecoratedObject(), 'deleteNote', $logEntries[$thread->threadID]);
			}
			
			$this->unmarkItems();
			
			$this->updateLastPost($boardIDs);
		}
		
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
		UserStorageHandler::getInstance()->resetAll('wbbWatchedThreads');
		
		return $this->getThreadData();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function validateDelete() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canDeleteThreadCompletely')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function delete() {
		$isBulkProcessing = (isset($this->parameters['isBulkProcessing']) && $this->parameters['isBulkProcessing']);
		$ignoreThreadModificationLogs = (isset($this->parameters['ignoreThreadModificationLogs']) && $this->parameters['ignoreThreadModificationLogs']);
		
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		$threadData = [];
		foreach ($this->getObjects() as $thread) {
			$threadData[$thread->threadID] = $thread->userID;
		}
		$threadIDs = array_keys($threadData);
		
		// remove user activity events
		// ignore posts as their activity data is deleted by PostAction::delete()
		$this->removeActivityEvents($threadData, true);
		
		// remove posts
		$postList = new PostList();
		$postList->getConditionBuilder()->add("post.threadID IN (?)", [$threadIDs]);
		$postList->readObjects();
		
		if (count($postList)) {
			$postAction = new PostAction($postList->getObjects(), 'delete', [
				'ignoreThreads' => true,
				'isBulkProcessing' => $isBulkProcessing,
				'ignorePostModificationLogs' => true,
				'noTrashing' => true
			]);
			$postAction->executeAction();
		}
		
		parent::delete();
		
		if (!$isBulkProcessing) {
			foreach ($this->getObjects() as $thread) {
				$boardLink = $thread->getBoard()->getLink();
				
				$this->addThreadData($thread->getDecoratedObject(), 'deleted', $boardLink);
				
				if (!$ignoreThreadModificationLogs) {
					ThreadModificationLogHandler::getInstance()->delete($thread->getDecoratedObject());
				}
			}
		}
		
		// delete tags
		TagEngine::getInstance()->deleteObjects('com.woltlab.wbb.thread', $threadIDs);
		
		// delete subscriptions
		UserObjectWatchHandler::getInstance()->deleteObjects('com.woltlab.wbb.thread', $threadIDs);
		
		// delete label assignments
		LabelHandler::getInstance()->removeLabels(LabelHandler::getInstance()->getObjectType('com.woltlab.wbb.thread')->objectTypeID, $threadIDs);
		
		// remove notifications
		UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.thread', $threadIDs);
		
		$modificationLogList = new ModificationLogList();
		$modificationLogList->getConditionBuilder()->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.thread')]);
		$modificationLogList->getConditionBuilder()->add('modification_log.objectID IN (?)', [$threadIDs]);
		$modificationLogList->readObjects();
		
		if (count($modificationLogList)) {
			UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.moderation.thread', $modificationLogList->getObjectIDs());
		}
		
		// delete the log entries except for deleting the thread
		if (!$ignoreThreadModificationLogs) {
			ThreadModificationLogHandler::getInstance()->deleteLogs($threadIDs, ['delete']);
		}
		
		// delete all post modification logs at once
		PostModificationLogHandler::getInstance()->deleteLogsByParentIDs($threadIDs);
		
		if (!$isBulkProcessing) {
			$this->unmarkItems();
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters for restoring threads.
	 */
	public function validateRestore() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canRestoreThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Restores given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function restore() {
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		$isBulkProcessing = (isset($this->parameters['isBulkProcessing']) && $this->parameters['isBulkProcessing']);
		
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => [
				'deleteTime' => 0,
				'isDeleted' => 0
			],
			'isBulkProcessing' => $isBulkProcessing,
			'ignoreThreadModificationLogs' => true
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			$boardIDs = [];
			foreach ($this->getObjects() as $thread) {
				$boardIDs[] = $thread->boardID;
				
				$this->addThreadData($thread->getDecoratedObject(), 'isDeleted', 0);
				ThreadModificationLogHandler::getInstance()->restore($thread->getDecoratedObject());
			}
		}
		
		$this->restorePosts($this->objectIDs, $isBulkProcessing);
		
		if (!$isBulkProcessing) {
			ThreadEditor::rebuildThreadData($this->objectIDs);
			
			// update last post
			foreach ($this->getObjects() as $thread) {
				$thread->updateLastPost();
			}
			
			// update board counters
			foreach ($boardIDs as $boardID) {
				$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
				$boardEditor->rebuildStats();
			}
			
			$this->unmarkItems();
			
			$this->updateLastPost($boardIDs);
		}
		
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
		UserStorageHandler::getInstance()->resetAll('wbbWatchedThreads');
		
		return $this->getThreadData();
	}
	
	/**
	 * Validates the `prepareEnable` action.
	 *
	 * @throws	PermissionDeniedException
	 * @throws	UserInputException
	 */
	public function validatePrepareEnable() {
		$this->threadEditor = $this->getSingleObject();
		if (!$this->threadEditor->isDisabled || $this->threadEditor->isDeleted) {
			throw new UserInputException('objectIDs');
		}
		else if (!$this->threadEditor->getBoard()->getModeratorPermission('canEnableThread')) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * Returns the dialog to set the enable time for a thread and all its posts.
	 *
	 * @return	array
	 */
	public function prepareEnable() {
		$enableTime = $this->threadEditor->getFirstPost()->enableTime;
		WCF::getTPL()->assign([
			'enableTime' => ($enableTime ? date('r', $enableTime) : ''),
			'thread' => $this->threadEditor
		]);
		
		return [
			'objectID' => $this->threadEditor->threadID,
			'template' => WCF::getTPL()->fetch('threadEnable', 'wbb')
		];
	}
	
	/**
	 * Validates the `setEnableTime` action.
	 *
	 * @throws	UserInputException
	 */
	public function validateSetEnableTime() {
		$this->threadEditor = $this->getSingleObject();
		
		$this->readString('enableTime', true);
		if (!empty($this->parameters['enableTime'])) {
			$this->parameters['enableTimeObj'] = \DateTime::createFromFormat('Y-m-d\TH:i:sP', $this->parameters['enableTime']);
			/** @noinspection PhpUndefinedMethodInspection */
			if (!$this->parameters['enableTimeObj'] || $this->parameters['enableTimeObj']->getTimestamp() < TIME_NOW) {
				throw new UserInputException('enableTime', 'invalid');
			}
		}
	}
	
	/**
	 * Sets the enable time of a thread and all its posts.
	 */
	public function setEnableTime() {
		$sql = "UPDATE  wbb".WCF_N."_post
			SET     enableTime = ?
			WHERE   threadID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		/** @noinspection PhpUndefinedMethodInspection */
		$statement->execute([
			($this->parameters['enableTime'] ? $this->parameters['enableTimeObj']->getTimestamp() : 0),
			$this->threadEditor->threadID
		]);
	}
	
	/**
	 * Validating parameters for enabling threads.
	 */
	public function validateEnable() {
		$this->readBoolean('updateTime', true, 'data');
		
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->isDisabled || $thread->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canEnableThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Enables given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function enable() {
		$isBulkProcessing = $this->isBulkProcessing();
		$ignoreThreadModificationLogs = (isset($this->parameters['ignoreThreadModificationLogs']) && $this->parameters['ignoreThreadModificationLogs']);
		
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		if (!isset($this->parameters['data']['updateTime'])) {
			$this->parameters['data']['updateTime'] = false;
		}
		
		$data = ['isDisabled' => 0];
		if ($this->parameters['data']['updateTime']) {
			$data['time'] = TIME_NOW;
		}
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => $data,
			'isBulkProcessing' => $isBulkProcessing,
			'ignoreThreadModificationLogs' => true,
			'ignoreCounterUpdate' => true
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			$boardIDs = $postIDs = [];
			foreach ($this->getObjects() as $thread) {
				$boardIDs[] = $thread->boardID;
				
				$this->addThreadData($thread->getDecoratedObject(), 'isDisabled', 0);
				
				if (!$ignoreThreadModificationLogs) {
					ThreadModificationLogHandler::getInstance()->enable($thread->getDecoratedObject());
				}
			}
		}
		
		// publish threads
		(new ThreadAction($this->objects, 'triggerPublication', [
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		// enable posts only if it was not initiated by enabling a specific post
		if (empty($this->parameters['ignorePosts'])) {
			// get affected posts
			$postList = new PostList();
			$postList->getConditionBuilder()->add("post.threadID IN (?)", [$this->objectIDs]);
			$postList->readObjects();
			
			// enable posts
			$postAction = new PostAction($postList->getObjects(), 'enable', [
				'ignoreThreads' => true,
				'isBulkProcessing' => $isBulkProcessing,
				'ignorePostModificationLogs' => true,
				'data' => [
					'updateTime' => $this->parameters['data']['updateTime'],
				],
			]);
			$postAction->executeAction();
		}
		
		if (!$isBulkProcessing) {
			ThreadEditor::rebuildThreadData($this->objectIDs);
			
			// update last post
			foreach ($this->getObjects() as $thread) {
				$thread->updateLastPost();
			}
			
			$this->unmarkItems();
			
			$this->updateLastPost($boardIDs);
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters for disabling threads.
	 */
	public function validateDisable() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->isDisabled || $thread->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$thread->getBoard()->getModeratorPermission('canEnableThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Disables given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function disable() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => ['isDisabled' => 1],
			'isBulkProcessing' => $isBulkProcessing,
			'ignoreThreadModificationLogs' => true
		]))->executeAction();
		
		$boardIDs = $postIDs = $threadData = $boardStats = [];
		foreach ($this->getObjects() as $thread) {
			$threadData[$thread->threadID] = $thread->userID;
			
			if (!$isBulkProcessing) {
				$boardIDs[] = $thread->boardID;
				
				$this->addThreadData($thread->getDecoratedObject(), 'isDisabled', 1);
				
				ThreadModificationLogHandler::getInstance()->disable($thread->getDecoratedObject());
				
				if (!isset($boardStats[$thread->boardID])) {
					$boardStats[$thread->boardID] = [
						'threads' => 0
					];
				}
				$boardStats[$thread->boardID]['threads']--;
			}
		}
		
		// update counters
		if (!$isBulkProcessing && !empty($boardStats)) {
			foreach ($boardStats as $boardID => $stats) {
				$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
				$boardEditor->updateCounters($stats);
			}
		}
		
		// remove user activity events
		$this->removeActivityEvents($threadData, true);
		
		// remove notifications
		UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.thread', array_keys($threadData));
		
		$modificationLogList = new ModificationLogList();
		$modificationLogList->getConditionBuilder()->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.thread')]);
		$modificationLogList->getConditionBuilder()->add('modification_log.objectID IN (?)', [array_keys($threadData)]);
		$modificationLogList->readObjects();
		
		if (count($modificationLogList)) {
			UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.moderation.thread', $modificationLogList->getObjectIDs());
		}
		
		// disable posts only if it was not initiated by disabling a specific post
		if (empty($this->parameters['ignorePosts'])) {
			// get affected posts
			$postList = new PostList();
			$postList->getConditionBuilder()->add("post.threadID IN (?)", [array_keys($threadData)]);
			$postList->readObjects();
			
			// disable posts
			$postAction = new PostAction($postList->getObjects(), 'disable', [
				'ignoreThreads' => true,
				'isBulkProcessing' => $isBulkProcessing,
				'ignorePostModificationLogs' => true
			]);
			$postAction->executeAction();
		}
		
		if (!$isBulkProcessing) {
			ThreadEditor::rebuildThreadData($this->objectIDs);
			
			$this->unmarkItems();
			
			$this->updateLastPost($boardIDs);
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters for moving threads.
	 */
	public function validateMove() {
		$this->readInteger('boardID');
		$board = BoardCache::getInstance()->getBoard($this->parameters['boardID']);
		if ($board === null || $board->boardType != Board::TYPE_BOARD) {
			throw new UserInputException('boardID');
		}
		if (!$board->getModeratorPermission('canMoveThread')) {
			throw new PermissionDeniedException();
		}
		$this->board = new BoardEditor($board);
		
		$this->parameters['showMoveNotice'] = (isset($this->parameters['showMoveNotice']) && $this->parameters['showMoveNotice'] != 'false');
		
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->getBoard()->getModeratorPermission('canMoveThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Moves given threads.
	 * 
	 * @return	mixed[][]
	 */
	public function move() {
		if ($this->board === null) {
			$this->board = new BoardEditor(BoardCache::getInstance()->getBoard($this->parameters['boardID']));
		}
		
		$isBulkProcessing = $this->isBulkProcessing();
		$boardIDs = $addPoints = $removePoints = $firstPosts = $threadIDs = [];
		
		// remove obsolete labels
		$labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->board->boardID);
		$labelObjectTypeID = ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.label.object', 'com.woltlab.wbb.thread')->objectTypeID;
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('objectTypeID = ?', [$labelObjectTypeID]);
		$conditionBuilder->add('objectID IN (?)', [$this->getObjectIDs()]);
		if (!empty($labelGroupIDs)) $conditionBuilder->add('labelID NOT IN (SELECT labelID FROM wcf'.WCF_N.'_label WHERE groupID IN (?))', [$labelGroupIDs]);
		$sql = "DELETE FROM	wcf".WCF_N."_label_object
			".$conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		
		// count labels to see if any of the threads require `hasLabels` to be turned off
		$labelThreadIDs = [];
		foreach ($this->getObjects() as $thread) {
			if ($thread->hasLabels) {
				$labelThreadIDs[] = $thread->threadID;
			}
		}
		
		$hasLabels = [];
		if (!empty($labelThreadIDs)) {
			$conditionBuilder = new PreparedStatementConditionBuilder();
			$conditionBuilder->add('objectTypeID = ?', [$labelObjectTypeID]);
			$conditionBuilder->add('objectID IN (?)', [$labelThreadIDs]);
			
			$sql = "SELECT          DISTINCT objectID
				FROM            wcf".WCF_N."_label_object
				".$conditionBuilder;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			while ($row = $statement->fetchArray()) {
				$hasLabels[] = $row['objectID'];
			}
		}
		
		(new ThreadAction($this->getObjects(), 'update', [
			'data' => [
				'boardID' => $this->board->boardID
			],
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		$hasNoLabelThreads = $deleteSubscriptions = [];
		foreach ($this->getObjects() as $thread) {
			// do not move thread if destination matches origin
			if ($thread->boardID === $this->board->boardID) {
				continue;
			}
			
			// collect board ids
			$boardIDs[] = $this->board->boardID;
			$boardIDs[] = $thread->boardID;
			$threadIDs[] = $thread->threadID;
			
			// update thread
			if ($thread->hasLabels && !in_array($thread->threadID, $hasLabels)) {
				$hasNoLabelThreads[] = $thread;
			}
			
			if (!$isBulkProcessing && !$thread->isDisabled) {
				if (!$thread->isDeleted) {
					// update counters
					$this->board->updateCounters([
						'threads' => 1,
						'posts' => $thread->replies + 1
					]);
					
					$boardEditor = new BoardEditor($thread->getBoard());
					$boardEditor->updateCounters([
						'threads' => -1,
						'posts' => -1 * ($thread->replies + 1)
					]);
				}
				
				if ($thread->userID && $thread->getBoard()->countUserPosts && !$this->board->countUserPosts) {
					if ($thread->userID) {
						$removePoints[$thread->threadID] = $thread->userID;
					}
				}
				else if (!$thread->getBoard()->countUserPosts && $this->board->countUserPosts) {
					if ($thread->userID) {
						$addPoints[$thread->threadID] = $thread->userID;
					}
				}
				
				$firstPosts[] = $thread->firstPostID;
			}
			
			// create fake thread for redirection
			if ($this->parameters['showMoveNotice']) {
				ThreadEditor::create([
					'boardID' => $thread->boardID,
					'languageID' => $thread->languageID,
					'firstPostID' => $thread->firstPostID,
					'topic' => $thread->topic,
					'time' => $thread->time,
					'userID' => $thread->userID,
					'username' => $thread->username,
					'movedThreadID' => $thread->threadID,
					'movedTime' => TIME_NOW,
					'lastPostID' => $thread->lastPostID,
					'lastPostTime' => $thread->lastPostTime,
					'lastPosterID' => $thread->lastPosterID,
					'lastPoster' => $thread->lastPoster,
					'replies' => $thread->replies,
					'views' => $thread->views,
					'attachments' => $thread->attachments,
					'polls' => $thread->polls,
					'cumulativeLikes' => $thread->cumulativeLikes,
					'isDeleted' => $thread->isDeleted,
					'isDisabled' => $thread->isDisabled
				]);
				
				if (!$isBulkProcessing) {
					$this->addThreadData($thread->getDecoratedObject(), 'showMoveNotice', 1);
				}
			}
			else if (!$isBulkProcessing) {
				$this->addThreadData($thread->getDecoratedObject(), 'moved', 1);
			}
			
			if (!$isBulkProcessing) {
				ThreadModificationLogHandler::getInstance()->move($thread->getDecoratedObject(), $thread->getBoard(), $this->board->getDecoratedObject());
			}
			
			if ($this->board->isPrivate) {
				$deleteSubscriptions[] = $thread->getDecoratedObject()->threadID;
			}
		}
		
		// check if any of the threads has links from the destination forum to the current
		// thread; such links are no longer necessary because they the threads are in the
		// original forum again
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('boardID = ?', [$this->board->boardID]);
		$conditionBuilder->add('movedThreadID IN (?)', [$this->objectIDs]);
		$sql = "SELECT	threadID
			FROM	wbb".WCF_N."_thread
			" . $conditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		$obsoleteLinkIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
		if (!empty($obsoleteLinkIDs)) {
			$conditionBuilder = new PreparedStatementConditionBuilder();
			$conditionBuilder->add('threadID IN (?)', [$obsoleteLinkIDs]);
			
			$sql = "DELETE FROM	wbb".WCF_N."_thread
				" . $conditionBuilder;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
		}
		
		// update the modification logs with the new parent board id
		ThreadModificationLogHandler::getInstance()->updateParentObjectID($threadIDs, $this->board->boardID);
		
		if (!empty($hasNoLabelThreads)) {
			(new ThreadAction($hasNoLabelThreads, 'update', [
				'data' => [
					'hasLabels' => 0
				],
				'isBulkProcessing' => $isBulkProcessing
			]))->executeAction();
		}
		
		if (!$isBulkProcessing) {
			$this->unmarkItems();
		}
		
		// update activity points
		if (!empty($removePoints) || !empty($addPoints)) {
			$userPosts = [];
			
			if (!empty($removePoints)) {
				$conditions = new PreparedStatementConditionBuilder();
				$conditions->add("threadID IN (?)", [array_keys($removePoints)]);
				$conditions->add("userID IS NOT NULL");
				$conditions->add("isDisabled = ?", [0]);
				
				$sql = "SELECT		COUNT(*) AS count, userID
					FROM		wbb".WCF_N."_post
					".$conditions."
					GROUP BY	userID";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($conditions->getParameters());
				while ($row = $statement->fetchArray()) {
					$userPosts[$row['userID']] = $row['count'] * -1;
				}
			}
			
			if (!empty($addPoints)) {
				$conditions = new PreparedStatementConditionBuilder();
				$conditions->add("threadID IN (?)", [array_keys($addPoints)]);
				$conditions->add("userID IS NOT NULL");
				$conditions->add("isDisabled = ?", [0]);
				
				$sql = "SELECT		COUNT(*) AS count, userID
					FROM		wbb".WCF_N."_post
					".$conditions."
					GROUP BY	userID";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($conditions->getParameters());
				while ($row = $statement->fetchArray()) {
					if (!isset($userPosts[$row['userID']])) {
						$userPosts[$row['userID']] = 0;
					}
					
					$userPosts[$row['userID']] += $row['count'];
				}
			}
			
			if (!empty($userPosts)) {
				PostEditor::updatePostCounter($userPosts);
			}
		}
		
		$userIDs = [];
		if (!empty($removePoints)) {
			// remove points for threads
			$userThreads = [];
			foreach ($removePoints as $threadID => $userID) {
				if (!isset($userThreads[$userID])) {
					$userThreads[$userID] = 0;
				}
				$userThreads[$userID]++;
			}
			UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.thread', $userThreads);
			
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("threadID IN (?)", [array_keys($removePoints)]);
			$conditions->add("postID NOT IN (?)", [$firstPosts]);
			$conditions->add("isDisabled = ?", [0]);
			
			$sql = "SELECT	userID, postID
				FROM	wbb".WCF_N."_post
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			$userToItems = [];
			while ($row = $statement->fetchArray()) {
				if (!$row['userID']) {
					continue;
				}
				
				$userIDs[] = $row['userID'];
				
				if (!isset($userToItems[$row['userID']])) {
					$userToItems[$row['userID']] = 0;
				}
				$userToItems[$row['userID']]++;
			}
			
			// remove points for posts
			UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $userToItems);
		}
		
		if (!empty($addPoints)) {
			// add points for threads
			foreach ($addPoints as $threadID => $userID) {
				UserActivityPointHandler::getInstance()->fireEvent('com.woltlab.wbb.activityPointEvent.thread', $threadID, $userID);
			}
			
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("threadID IN (?)", [array_keys($addPoints)]);
			$conditions->add("postID NOT IN (?)", [$firstPosts]);
			$conditions->add("isDisabled = ?", [0]);
			
			$sql = "SELECT	postID, userID
				FROM	wbb".WCF_N."_post
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			$userToItems = [];
			while ($row = $statement->fetchArray()) {
				if (!$row['userID']) {
					continue;
				}
				
				$userIDs[] = $row['userID'];
				
				if (!isset($userToItems[$row['userID']])) {
					$userToItems[$row['userID']] = 0;
				}
				$userToItems[$row['userID']]++;
			}
			
			// add points for posts
			UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $userToItems);
		}
		
		// update caches
		if (!empty($userIDs)) {
			UserActivityPointHandler::getInstance()->updateUsers($userIDs);
		}
		
		// delete all subscriptions when moving threads into private forums
		if (!empty($deleteSubscriptions)) {
			UserObjectWatchHandler::getInstance()->deleteObjects('com.woltlab.wbb.thread', $deleteSubscriptions);
		}
		
		if (!$isBulkProcessing) {
			$this->updateLastPost($boardIDs);
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validating parameters to prepare moving of threads.
	 */
	public function validatePrepareMove() {
		$this->readInteger('boardID', true);
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->getBoard()->getModeratorPermission('canMoveThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Prepares moving of given threads.
	 * 
	 * @return	string[]
	 */
	public function prepareMove() {
		// get current board list
		$boardNodeList = new ModeratorBoardNodeList();
		$boardNodeList->readNodeTree();
		
		WCF::getTPL()->assign([
			'boardID' => $this->parameters['boardID'],
			'boardNodeList' => $boardNodeList->getNodeList()
		]);
		
		return [
			'template' => WCF::getTPL()->fetch('moveThreads', 'wbb')
		];
	}
	
	/**
	 * Validating parameters to remove links to moved threads.
	 */
	public function validateRemoveLink() {
		$this->loadThreads();
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->getBoard()->getModeratorPermission('canMoveThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Removes links to moved threads.
	 * 
	 * @return	mixed[][]
	 */
	public function removeLink() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		$threadIDs = [];
		foreach ($this->getObjects() as $thread) {
			if (!$thread->movedThreadID) {
				continue;
			}
			
			$threadIDs[] = $thread->threadID;
			
			if (!$isBulkProcessing) {
				$this->addThreadData($thread->getDecoratedObject(), 'deleted', 1);
			}
		}
		
		if (!empty($threadIDs)) {
			// `ThreadEditor::deleteAll()` can be used because these link threads are
			// lightweight and have no additional data attached
			ThreadEditor::deleteAll($threadIDs);
		}
		
		if (!$isBulkProcessing) {
			$this->unmarkItems();
		}
		
		return $this->getThreadData();
	}
	
	/**
	 * Validates parameters to get new posts.
	 */
	public function validateGetNewPosts() {
		// validate action
		if (!isset($this->parameters['countPosts'])) {
			throw new UserInputException('countPosts');
		}
		$this->parameters['countPosts'] = ($this->parameters['countPosts'] == 'true');
		
		if (!$this->parameters['countPosts']) {
			// validate page no
			$this->parameters['pageNo'] = isset($this->parameters['pageNo']) ? intval($this->parameters['pageNo']) : 0;
			if (!$this->parameters['pageNo']) {
				throw new UserInputException('pageNo');
			}
		}
		
		// validate last post time
		$this->parameters['lastPostTime'] = isset($this->parameters['lastPostTime']) ? intval($this->parameters['lastPostTime']) : 0;
		if (!$this->parameters['lastPostTime']) {
			throw new UserInputException('lastPostTime');
		}
		
		// validate thread id and permissions
		$this->thread = $this->getSingleObject();
		if (!$this->thread->canRead()) {
			throw new UserInputException();
		}
	}
	
	/**
	 * Returns the number of new posts or the parsed template.
	 * 
	 * @return	string[]
	 */
	public function getNewPosts() {
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('threadID = ?', [$this->thread->threadID]);
		if (!$this->thread->getBoard()->getModeratorPermission('canEnablePost')) $conditionBuilder->add('isDisabled = 0');
		if (!$this->thread->getBoard()->getModeratorPermission('canReadDeletedPost')) $conditionBuilder->add('isDeleted = 0');
		$conditionBuilder->add('time > ?', [$this->parameters['lastPostTime']]);
		
		if ($this->parameters['countPosts']) {
			$sql = "SELECT	COUNT(*) AS count
				FROM	wbb".WCF_N."_post
				".$conditionBuilder;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			$row = $statement->fetchArray();
			
			return [
				'newPostsCount' => $row['count']
			];
		}
		else {
			// count message index
			$postList = new ThreadPostList($this->thread->getDecoratedObject());
			$postList->getConditionBuilder()->add("post.time < ?", [$this->parameters['lastPostTime']]);
			$exitingPosts = $postList->countObjects();
			
			// check if posts will be displayed
			$postsPerPage = $this->thread->getBoard()->getPostsPerPage();
			$postsOnCurrentPage = $exitingPosts % $postsPerPage;
			if ($postsOnCurrentPage == $postsPerPage) {
				// redirect to next page
				return [
					'redirectURL' => LinkHandler::getInstance()->getLink(
						'Thread',
						['object' => $this->thread],
						'pageNo='.($this->parameters['pageNo'] + 1)
					),
				];
			}
			
			// get post ids
			$sql = "SELECT		postID
				FROM		wbb".WCF_N."_post
				".$conditionBuilder."
				ORDER BY	time ".$this->thread->getBoard()->getPostSortOrder();
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			$postIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
			
			$maxPostsOnCurrentPage = $postsPerPage - $postsOnCurrentPage;
			$displayPostIDs = [];
			while ($maxPostsOnCurrentPage > 0) {
				$displayPostIDs[] = array_shift($postIDs);
				
				$maxPostsOnCurrentPage--;
			}
			
			$postList = new ViewablePostList();
			$postList->setObjectIDs($displayPostIDs);
			$postList->setThread($this->thread->getDecoratedObject());
			$postList->sqlOrderBy = "post.time ".$this->thread->getBoard()->getPostSortOrder();
			$postList->readObjects();
			
			// check if there are more posts not being displayed
			$morePosts = count($postIDs);
			$nextPageNotice = '';
			if ($morePosts) {
				$nextPageNotice = WCF::getLanguage()->getDynamicVariable('wbb.thread.newPosts.nextPage', [
					'count' => $morePosts,
					'pageNo' => $this->parameters['pageNo'] + 1,
					'thread' => $this->thread->getDecoratedObject()
				]);
			}
			
			// mark loaded posts as read
			$visitTime = 0;
			foreach ($postList as $post) {
				$visitTime = max($visitTime, $post->time);
			}
			
			$threadAction = new ThreadAction([$this->thread->getDecoratedObject()], 'markAsRead', ['visitTime' => $visitTime]);
			$threadAction->executeAction();
			
			WCF::getTPL()->assign([
				'attachmentList' => $postList->getAttachmentList(),
				'objects' => $postList,
				'startIndex' => $exitingPosts + 1,
				'sortOrder' => $this->thread->getBoard()->getPostSortOrder(),
				'thread' => new ViewableThread($this->thread->getDecoratedObject())
			]);
			
			return [
				'lastPostTime' => $postList->getMaxPostTime(),
				'nextPageNotice' => $nextPageNotice,
				'template' => WCF::getTPL()->fetch('threadPostList', 'wbb')
			];
		}
	}
	
	/**
	 * Validates the unmark all action.
	 */
	public function validateUnmarkAll() {
		// does nothing
	}
	
	/**
	 * Unmarks all threads.
	 */
	public function unmarkAll() {
		ClipboardHandler::getInstance()->removeItems(ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.thread'));
	}
	
	/**
	 * Updates the similar threads.
	 */
	public function updateSimilarThreads() {
		// get board ids
		$boardIDs = [];
		foreach (BoardCache::getInstance()->getBoards() as $board) {
			if ($board->searchableForSimilarThreads) $boardIDs[] = $board->boardID;
		}
		
		// reset similar threads
		$threadIDs = [];
		foreach ($this->getObjects() as $thread) {
			$threadIDs[] = $thread->threadID;
		}
		if (!empty($threadIDs)) {
			ThreadEditor::resetSimilarThreads($threadIDs);
		}
		
		// no suitable board ids found
		if (empty($boardIDs)) {
			return;
		}
		
		// save new similar threads
		foreach ($this->getObjects() as $thread) {
			// get similar threads
			$threadIDs = Thread::getSimilarThreads($thread->topic, $boardIDs, WBB_THREAD_SIMILAR_THREADS_COUNT, $thread->boardID, $thread->threadID, $thread->languageID);
			
			if (!empty($threadIDs)) {
				$thread->setSimilarThreads($threadIDs);
			}
		}
	}
	
	/**
	 * Validates the 'getSimilarThreads' function.
	 */
	public function validateGetSimilarThreads() {
		$this->readInteger('boardID');
		$this->readInteger('languageID', true);
		$this->readString('topic');
	}
	
	/**
	 * Finds similar threads.
	 * 
	 * @return	string[]
	 */
	public function getSimilarThreads() {
		$returnValue = [
			'results' => 0,
			'template' => ''
		];
		
		// get accessible boards, private boards are excluded
		$accessibleBoardIDs = Board::getAccessibleBoardIDs(['canViewBoard', 'canEnterBoard', 'canReadThread']);
		foreach ($accessibleBoardIDs as $key => $boardID) {
			$board = BoardCache::getInstance()->getBoard($boardID);
			if (!$board->searchableForSimilarThreads) unset($accessibleBoardIDs[$key]);
		}
		
		if (!empty($accessibleBoardIDs)) {
			// get thread ids
			$threadIDs = Thread::getSimilarThreads($this->parameters['topic'], $accessibleBoardIDs, 5, intval($this->parameters['boardID']), 0, $this->parameters['languageID'] ?: null);
			
			if (!empty($threadIDs)) {
				$threadList = new ThreadList();
				$threadList->setObjectIDs($threadIDs);
				$threadList->sqlOrderBy = 'thread.lastPostTime DESC, thread.lastPostID DESC';
				$threadList->readObjects();
				
				WCF::getTPL()->assign([
					'similarThreads' => $threadList->getObjects()
				]);
				
				$returnValue = [
					'results' => count($threadList->getObjects()),
					'template' => WCF::getTPL()->fetch('threadAddSimilarThreads', 'wbb')
				];
			}
		}
		
		return $returnValue;
	}
	
	/**
	 * Validates parameters to prepare threads for merging.
	 */
	public function validatePrepareMerge() {
		// read threads
		$threadList = new ViewableThreadList();
		$threadList->setObjectIDs($this->objectIDs);
		$threadList->readObjects();
		$this->objects = $threadList->getObjects();
		
		if (count($this->objects) < 2) {
			throw new UserInputException('objectIDs');
		}
		
		foreach ($this->getObjects() as $thread) {
			if (!$thread->canRead() || !$thread->getBoard()->getModeratorPermission('canMergeThread')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Prepares threads for merging.
	 * 
	 * @return	string[]
	 */
	public function prepareMerge() {
		WCF::getTPL()->assign([
			'threads' => $this->objects
		]);
		
		return [
			'template' => WCF::getTPL()->fetch('threadMerge', 'wbb')
		];
	}
	
	/**
	 * Validates parameters to merge threads.
	 */
	public function validateMerge() {
		$this->validatePrepareMerge();
		$this->readInteger('threadID');
		
		foreach ($this->getObjects() as $key => $thread) {
			if ($thread->threadID == $this->parameters['threadID']) {
				$this->threadEditor = new ThreadEditor($thread->getDecoratedObject());
				unset($this->objects[$key]);
				
				break;
			}
		}
		
		if ($this->threadEditor === null) {
			throw new UserInputException('threadID');
		}
	}
	
	/**
	 * Merges threads.
	 * 
	 * @return	array
	 */
	public function merge() {
		// 
		// step 1) prepare
		// 
		
		// if the threads are merged into a disabled thread, disable the other threads
		// first to make handling stats and activity points easier
		if ($this->threadEditor->isDisabled) {
			$disableThreadIDs = $threadIDs = [];
			foreach ($this->getObjects() as $thread) {
				$threadIDs[] = $thread->threadID;
				if (!$thread->isDisabled) {
					$disableThreadIDs[] = $thread->threadID;
				}
			}
			
			if (!empty($disableThreadIDs)) {
				$threadAction = new ThreadAction($disableThreadIDs, 'disable');
				$threadAction->executeAction();
				
				$threadList = new ThreadList();
				$threadList->decoratorClassName = ThreadEditor::class;
				$threadList->setObjectIDs($threadIDs);
				$threadList->readObjects();
				$this->objects = $threadList->getObjects();
			}
		}
		
		$addReplies = $addViews = 0;
		$threadIDs = $updateBoardThreads = [];
		
		// ids of threads with activity events to be deleted
		$activityEventThreadIDs = [];
		
		// ids of threads with activity points to be deleted
		$activityPointThreadIDs = [];
		
		// number of posts per user additionally due to first posts now being normal posts
		$postActivityPointUserIDs = [];
		
		// number of threads per user merged from boards with enabled user posts counter setting
		$threadActivityPointUserIDs = [];
		
		foreach ($this->getObjects() as $thread) {
			$threadIDs[] = $thread->threadID;
			
			// add 1 reply for initial post
			if (!$this->threadEditor->isDisabled) {
				$addReplies += ($thread->replies + 1);
			}
			$addViews += $thread->views;
			
			$activityEventThreadIDs[] = $thread->threadID;
			
			// if the merged thread is disabled, it has not been considered for its board stats,
			// thus no need to update them
			if (!$thread->isDisabled) {
				if (!isset($updateBoardThreads[$thread->boardID])) {
					$updateBoardThreads[$thread->boardID] = [
						'posts' => 0,
						'threads' => 0
					];
				}
				
				$updateBoardThreads[$thread->boardID]['posts'] -= ($thread->replies + 1);
				$updateBoardThreads[$thread->boardID]['threads']--;
			}
			
			if ($thread->getBoard()->countUserPosts != $this->threadEditor->getBoard()->countUserPosts) {
				$activityPointThreadIDs[] = $thread->threadID;
			}
			
			if ($thread->getBoard()->countUserPosts && $thread->userID && !$thread->isDisabled) {
				if (!isset($threadActivityPointUserIDs[$thread->userID])) {
					$threadActivityPointUserIDs[$thread->userID] = 0;
				}
				$threadActivityPointUserIDs[$thread->userID]++;
				
				// add activity points for threads' first posts
				if ($this->threadEditor->getBoard()->countUserPosts && !$this->threadEditor->isDisabled) {
					if (!isset($postActivityPointUserIDs[$thread->userID])) {
						$postActivityPointUserIDs[$thread->userID] = 0;
					}
					$postActivityPointUserIDs[$thread->userID]++;
				}
			}
		}
		
		$boardIDs[] = $this->threadEditor->boardID;
		if (!$this->threadEditor->isDisabled) {
			if (!isset($updateBoardThreads[$this->threadEditor->boardID])) {
				$updateBoardThreads[$this->threadEditor->boardID] = [
					'posts' => 0,
					'threads' => 0
				];
			}
			
			$updateBoardThreads[$this->threadEditor->boardID]['posts'] += $addReplies;
		}
		
		// 
		// step 2) merge
		// 
		
		// add/remove user activity point events
		UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $postActivityPointUserIDs);
		
		$countUserPosts = $this->threadEditor->getBoard()->countUserPosts;
		if (!empty($activityPointThreadIDs)) {
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("post.threadID IN (?)", [$activityPointThreadIDs]);
			$sql = "SELECT		postID, post.userID, firstPostID
				FROM		wbb".WCF_N."_post post
				LEFT JOIN	wbb".WCF_N."_thread thread
				ON		(thread.threadID = post.threadID)
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			$activityPointPostUserUpdates = [];
			$posts = [];
			while ($row = $statement->fetchArray()) {
				if (!$row['userID']) {
					continue;
				}
				
				if (!isset($posts[$row['userID']])) {
					$posts[$row['userID']] = 0;
					$activityPointPostUserUpdates[$row['userID']] = 0;
				}
				
				if ($countUserPosts) {
					$posts[$row['userID']]++;
					$activityPointPostUserUpdates[$row['userID']]++;
				}
				else {
					$posts[$row['userID']]--;
					
					// first posts never got activity points
					if ($row['postID'] != $row['firstPostID']) {
						$activityPointPostUserUpdates[$row['userID']]++;
					}
				}
			}
			
			if (!empty($posts)) {
				// add events
				if ($countUserPosts) {
					UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $activityPointPostUserUpdates);
				}
				else {
					UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $activityPointPostUserUpdates);
				}
				
				// update user posts
				PostEditor::updatePostCounter($posts);
			}
		}
		
		// move posts to new thread
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("threadID IN (?)", [$threadIDs]);
		$parameters = $conditions->getParameters();
		array_unshift($parameters, $this->threadEditor->threadID);
		
		$sql = "UPDATE	wbb".WCF_N."_post
			SET	threadID = ?
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($parameters);
		
		// 
		// step 3) remove old threads
		// 
		
		foreach ($this->getObjects() as $thread) {
			$threadEditor = new ThreadEditor($thread->getDecoratedObject());
			$threadEditor->delete();
			
			// remove tags
			TagEngine::getInstance()->deleteObjectTags('com.woltlab.wbb.thread', $thread->threadID);
		}
		
		// remove user activity events/points
		UserActivityEventHandler::getInstance()->removeEvents('com.woltlab.wbb.recentActivityEvent.thread', $activityEventThreadIDs);
		UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.thread', $threadActivityPointUserIDs);
		
		// 
		// step 4) update counters, first post and last posts
		// 
		
		// update thread replies and views
		$this->threadEditor->updateCounters([
			'replies' => $addReplies,
			'views' => $addViews
		]);
		
		// update last post
		$this->threadEditor->updateLastPost();
		
		// check if first post of the thread needs to be updated
		$sql = "SELECT		post.postID, post.time, post.userID, post.username, post.username, post.cumulativeLikes
			FROM		wbb".WCF_N."_post post
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = post.threadID)
			WHERE		post.threadID = ?
					AND post.isDeleted = thread.isDeleted
					AND post.isDisabled = thread.isDisabled
			ORDER BY	post.time ASC, post.postID ASC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute([
			$this->threadEditor->threadID
		]);
		$firstPostData = $statement->fetchArray();
		
		if ($firstPostData['postID'] != $this->threadEditor->firstPostID) {
			// first post changed, update thread's data
			$this->threadEditor->update([
				'cumulativeLikes' => $firstPostData['cumulativeLikes'],
				'firstPostID' => $firstPostData['postID'],
				'time' => $firstPostData['time'],
				'userID' => $firstPostData['userID'],
				'username' => $firstPostData['username']
			]);
			
			// check if thread's user changed and user posts are counted
			if (!$this->threadEditor->isDisabled && $this->threadEditor->userID != $firstPostData['userID'] && $this->threadEditor->getBoard()->countUserPosts) {
				if ($this->threadEditor->userID) {
					// decrease old user's activity points for threads
					UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.thread', [
						$this->threadEditor->userID => 1
					]);
					
					// increase old user's activity points for posts
					UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', [
						$this->threadEditor->userID => 1
					]);
				}
				
				if ($firstPostData['userID']) {
					// increase new user's activity points for threads
					UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.thread', [
						$firstPostData['userID'] => 1
					]);
					
					// decrease old user's activity points for posts
					UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', [
						$firstPostData['userID'] => 1
					]);
				}
			}
		}
		
		foreach ($updateBoardThreads as $boardID => $boardData) {
			$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
			
			// update post/thread counter
			$boardEditor->updateCounters([
				'posts' => $boardData['posts'],
				'threads' => $boardData['threads']
			]);
			
			// update last posts
			$boardEditor->updateLastPost();
		}
		
		BoardEditor::resetDataCache();
		
		// add modification log entry for merging threads
		ThreadModificationLogHandler::getInstance()->merge($this->threadEditor->getDecoratedObject(), $this->objects);
		
		$threadIDs[] = $this->threadEditor->threadID;
		$this->unmarkItems($threadIDs);
		
		// redirect to merge target
		return [
			'redirectURL' => $this->threadEditor->getLink(),
		];
	}
	
	/**
	 * Validates the 'stopWatching' action.
	 */
	public function validateStopWatching() {
		$this->readBoolean('stopWatchingAll', true);
		
		if (!$this->parameters['stopWatchingAll']) {
			if (!isset($this->parameters['threadIDs']) || !is_array($this->parameters['threadIDs'])) {
				throw new UserInputException('threadIDs');
			}
		}
	}
	
	/**
	 * Stops watching certain threads for a certain user.
	 */
	public function stopWatching() {
		if ($this->parameters['stopWatchingAll']) {
			$threadWatchList = new UserObjectWatchList();
			$threadWatchList->getConditionBuilder()->add('user_object_watch.objectTypeID = ?', [UserObjectWatchHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')]);
			$threadWatchList->getConditionBuilder()->add('user_object_watch.userID = ?', [WCF::getUser()->userID]);
			$threadWatchList->readObjects();
			
			$this->parameters['threadIDs'] = [];
			foreach ($threadWatchList as $watchedObject) {
				$this->parameters['threadIDs'][] = $watchedObject->objectID;
			}
		}
		
		UserObjectWatchHandler::getInstance()->deleteObjects('com.woltlab.wbb.thread', $this->parameters['threadIDs'], [WCF::getUser()->userID]);
		UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedThreads');
		UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbWatchedThreads');
	}
	
	/**
	 * Triggers initial publication.
	 * 
	 * You should only call this method if the thread is visible directly after creation
	 * or was manually approved ("enable"). Using this method in any other state is undefined.
	 */
	public function triggerPublication() {
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		$isBulkProcessing = (isset($this->parameters['isBulkProcessing']) && $this->parameters['isBulkProcessing']);
		
		$activityEvents = $boardCounter = [];
		foreach ($this->getObjects() as $thread) {
			if (!$isBulkProcessing) {
				if (!isset($boardCounter[$thread->boardID])) {
					$boardCounter[$thread->boardID] = [
						'posts' => 0,
						'threads' => 0
					];
				}
				
				$boardCounter[$thread->boardID]['posts']++;
				$boardCounter[$thread->boardID]['threads']++;
			}
			
			// fire activity event
			if ($thread->userID) {
				if ($isBulkProcessing) {
					$activityEvents[] = [
						'objectID' => $thread->threadID,
						'languageID' => $thread->languageID,
						'userID' => $thread->userID,
						'time' => $thread->time
					];
				}
				else {
					UserActivityEventHandler::getInstance()->fireEvent(
						'com.woltlab.wbb.recentActivityEvent.thread',
						$thread->threadID,
						$thread->languageID,
						$thread->userID,
						$thread->time
					);
				}
				
				if (!$isBulkProcessing && $thread->getBoard()->countUserPosts) {
					UserActivityPointHandler::getInstance()->fireEvent(
						'com.woltlab.wbb.activityPointEvent.thread', 
						$thread->threadID,
						$thread->userID
					);
				}
			}
			
			if (!$isBulkProcessing && !$thread->isDeleted) {
				UserObjectWatchHandler::getInstance()->updateObject(
					'com.woltlab.wbb.board',
					$thread->boardID,
					'thread',
					'com.woltlab.wbb.thread',
					new ThreadUserNotificationObject($thread->getDecoratedObject())
				);
			}
		}
		
		if (!empty($activityEvents)) {
			UserActivityEventHandler::getInstance()->fireEvents('com.woltlab.wbb.recentActivityEvent.thread', $activityEvents);
		}
		
		if (!$isBulkProcessing) {
			foreach ($boardCounter as $boardID => $counters) {
				$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
				$boardEditor->updateCounters([
					'posts' => $counters['posts'],
					'threads' => $counters['threads']
				]);
			}
		}
	}
	
	/**
	 * Rebuilds given threads.
	 */
	public function rebuild() {
		$this->readObjects();
		
		$deleteThreads = [];
		foreach ($this->getObjects() as $threadEditor) {
			$result = $threadEditor->rebuild(true);
			if ($result === false) {
				$deleteThreads[] = $threadEditor;
			}
		}
		
		// delete orphaned threads
		if (!empty($deleteThreads)) {
			$threadAction = new ThreadAction($deleteThreads, 'delete');
			$threadAction->executeAction();
		}
	}
	
	/**
	 * Validates parameters to copy a thread.
	 */
	public function validateCopy() {
		$this->readInteger('loopCount', true);
		$this->readInteger('sourceThreadID');
		$this->threadEditor = new ThreadEditor(new Thread($this->parameters['sourceThreadID']));
		
		// validate thread
		if (!$this->threadEditor->threadID) {
			throw new PermissionDeniedException();
		}
		else if (!$this->threadEditor->getBoard()->getModeratorPermission('canCopyThread')) {
			throw new PermissionDeniedException();
		}
		
		if ($this->parameters['loopCount'] == 0) {
			$this->readInteger('boardID');
			
			// validate target board
			$this->board = BoardCache::getInstance()->getBoard($this->parameters['boardID']);
			if ($this->board === null) {
				throw new UserInputException('boardID');
			}
			else if (!$this->board->getModeratorPermission('canCopyThread')) {
				throw new PermissionDeniedException();
			}
		}
		else {
			$this->readInteger('posts');
			$this->readInteger('threadID');
			
			$this->thread = new Thread($this->parameters['threadID']);
			if (!$this->thread->threadID) {
				throw new UserInputException('threadID');
			}
		}
	}
	
	/**
	 * Worker action to copy a thread.
	 * 
	 * @return	array
	 */
	public function copy() {
		if ($this->parameters['loopCount'] == 0) {
			// copy thread
			$this->thread = ThreadEditor::create([
				'boardID' => $this->board->boardID,
				'languageID' => $this->threadEditor->languageID ?: null,
				'topic' => $this->threadEditor->topic,
				'firstPostID' => null,
				'time' => $this->threadEditor->time,
				'userID' => $this->threadEditor->userID ?: null,
				'username' => $this->threadEditor->username,
				'lastPostID' => null,
				'lastPostTime' => 0,
				'lastPosterID' => null,
				'lastPoster' => '',
				'replies' => $this->threadEditor->replies,
				'attachments' => $this->threadEditor->attachments,
				'polls' => $this->threadEditor->polls,
				'isAnnouncement' => $this->threadEditor->isAnnouncement,
				'isSticky' => $this->threadEditor->isSticky,
				'isDisabled' => $this->threadEditor->isDisabled,
				'isClosed' => $this->threadEditor->isClosed,
				'isDeleted' => $this->threadEditor->isDeleted,
				'isDone' => $this->threadEditor->isDone,
				'cumulativeLikes' => $this->threadEditor->cumulativeLikes,
				'hasLabels' => $this->threadEditor->hasLabels,
				'deleteTime' => $this->threadEditor->deleteTime
			]);
			
			// copy announcement forums
			if ($this->threadEditor->isAnnouncement) {
				$sql = "INSERT INTO	wbb".WCF_N."_thread_announcement
							(boardID, threadID)
					SELECT		boardID, ".$this->thread->threadID."
					FROM		wbb".WCF_N."_thread_announcement
					WHERE		threadID = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute([$this->threadEditor->threadID]);
			}
			
			// copy assigned labels
			if ($this->threadEditor->hasLabels) {
				$removeHasLabelsFlag = false;
				$labelGroupIDs = BoardCache::getInstance()->getLabelGroupIDs($this->board->boardID);
				if (empty($labelGroupIDs)) {
					$removeHasLabelsFlag = true;
				}
				else {
					$conditionBuilder = new PreparedStatementConditionBuilder();
					$conditionBuilder->add('objectTypeID = ?', [LabelHandler::getInstance()->getObjectType('com.woltlab.wbb.thread')->objectTypeID]);
					$conditionBuilder->add('objectID = ?', [$this->threadEditor->threadID]);
					$conditionBuilder->add('labelID IN (SELECT labelID FROM wcf' . WCF_N . '_label WHERE groupID IN (?))', [$labelGroupIDs]);
					
					$sql = "INSERT INTO	wcf" . WCF_N . "_label_object
							(labelID, objectTypeID, objectID)
					SELECT		labelID, objectTypeID, " . $this->thread->threadID . "
					FROM		wcf" . WCF_N . "_label_object
					" . $conditionBuilder;
					$statement = WCF::getDB()->prepareStatement($sql);
					$statement->execute($conditionBuilder->getParameters());
					
					$sql = "SELECT  COUNT(*) AS count
					FROM    wcf".WCF_N."_label_object
					WHERE   objectTypeID = ?
						AND objectID = ?";
					$statement = WCF::getDB()->prepareStatement($sql);
					$statement->execute([LabelHandler::getInstance()->getObjectType('com.woltlab.wbb.thread')->objectTypeID, $this->thread->threadID]);
					if (!$statement->fetchColumn()) {
						$removeHasLabelsFlag = true;
					}
				}
				
				if ($removeHasLabelsFlag) {
					$editor = new ThreadEditor($this->thread);
					$editor->update(['hasLabels' => 0]);
				}
			}
			
			$sql = "INSERT INTO     wcf".WCF_N."_tag_to_object
						(objectID, tagID, objectTypeID, languageID)
				SELECT          ".$this->thread->threadID.", tagID, objectTypeID, languageID
				FROM            wcf".WCF_N."_tag_to_object
				WHERE           objectTypeID = ?
						AND objectID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([
				TagEngine::getInstance()->getObjectTypeID('com.woltlab.wbb.thread'),
				$this->threadEditor->threadID
			]);
			
			// count posts
			$sql = "SELECT	COUNT(*) AS count
				FROM	wbb".WCF_N."_post
				WHERE	threadID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->threadEditor->threadID]);
			$row = $statement->fetchArray();
			
			// first call, render template
			return [
				'loopCount' => 1,
				'parameters' => [
					'sourceThreadID' => $this->threadEditor->threadID,
					'posts' => $row['count'],
					'threadID' => $this->thread->threadID
				],
				'progress' => 0,
				'template' => WCF::getTPL()->fetch('worker')
			];
		}
		
		// get post ids
		$sql = "SELECT		postID
			FROM		wbb".WCF_N."_post
			WHERE		threadID = ?
			ORDER BY	postID ASC";
		$statement = WCF::getDB()->prepareStatement($sql, 5, ($this->parameters['loopCount'] - 1) * 5);
		$statement->execute([$this->threadEditor->threadID]);
		$postIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
		
		// copy posts
		$postAction = new PostAction([], 'copy', [
			'postIDs' => $postIDs,
			'thread' => $this->thread
		]);
		$postAction->executeAction();
		
		$progress = floor(($this->parameters['loopCount'] / ceil($this->parameters['posts'] / 5)) * 100);
		$returnValues = [
			'loopCount' => $this->parameters['loopCount'] + 1,
			'parameters' => [
				'sourceThreadID' => $this->threadEditor->threadID,
				'posts' => $this->parameters['posts'],
				'threadID' => $this->thread->threadID
			],
			'progress' => $progress
		];
		
		if ($returnValues['progress'] == 100) {
			// rebuild thread
			$threadEditor = new ThreadEditor($this->thread);
			$threadEditor->rebuild();
			ThreadEditor::rebuildThreadData([$this->thread->threadID]);
			
			$this->thread = new Thread($this->thread->threadID);
			$post = new Post($this->thread->firstPostID);
			if ($post->userID) {
				if (!$post->isDisabled && $this->thread->getBoard()->countUserPosts) {
					// fix activity points for first post (should be thread not post)
					UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', [$post->userID => 1]);
					UserActivityPointHandler::getInstance()->fireEvent('com.woltlab.wbb.activityPointEvent.thread', $this->thread->threadID, $post->userID);
				}
				
				// update thread's user id
				$threadEditor->update(['userID' => $post->userID]);
			}
			
			// update board stats / last post
			if (!$this->thread->isDisabled && !$this->thread->isDeleted) {
				$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($this->thread->boardID));
				$boardEditor->updateCounters([
					'posts' => $this->thread->replies + 1,
					'threads' => 1
				]);
				$boardEditor->updateLastPost();
				
				BoardEditor::resetDataCache();
			}
			
			// clear clipboard objects
			$this->unmarkItems([$this->threadEditor->threadID]);
		}
		
		return $returnValues;
	}
	
	/**
	 * Validates parameters to assign labels.
	 */
	public function validateAssignLabel() {
		$this->readInteger('boardID');
		
		// validate board
		$this->board = BoardCache::getInstance()->getBoard($this->parameters['boardID']);
		if ($this->board === null) {
			throw new UserInputException('boardID');
		}
		else if (!$this->board->getModeratorPermission('canEditPost')) {
			
			throw new PermissionDeniedException();
		}
		
		// validate threads
		$this->readObjects();
		if (empty($this->objects)) {
			throw new UserInputException('objectIDs');
		}
		
		foreach ($this->getObjects() as $thread) {
			if ($thread->boardID != $this->board->boardID) {
				throw new UserInputException('objectIDs');
			}
		}
		
		// validate label ids
		$this->parameters['labelIDs'] = empty($this->parameters['labelIDs']) ? [] : ArrayUtil::toIntegerArray($this->parameters['labelIDs']);
		if (!empty($this->parameters['labelIDs'])) {
			$labelIDs = BoardCache::getInstance()->getLabelGroupIDs($this->board->boardID);
			if (empty($labelIDs)) {
				throw new PermissionDeniedException();
			}
			
			$labelGroups = LabelHandler::getInstance()->getLabelGroups($labelIDs);
			foreach ($this->parameters['labelIDs'] as $groupID => $labelID) {
				if (!isset($labelGroups[$groupID]) || !$labelGroups[$groupID]->isValid($labelID)) {
					throw new UserInputException('labelIDs');
				}
			}
		}
	}
	
	/**
	 * Assigns labels and returns the updated list.
	 * 
	 * @return	mixed[][]
	 */
	public function assignLabel() {
		$objectTypeID = LabelHandler::getInstance()->getObjectType('com.woltlab.wbb.thread')->objectTypeID;
		
		$threadIDs = [];
		foreach ($this->getObjects() as $thread) {
			$threadIDs[] = $thread->threadID;
		}
		
		// fetch old labels for modification log creation
		$oldLabels = LabelHandler::getInstance()->getAssignedLabels($objectTypeID, $threadIDs);
		
		foreach ($this->getObjects() as $thread) {
			LabelHandler::getInstance()->setLabels($this->parameters['labelIDs'], $objectTypeID, $thread->threadID);
			
			// update hasLabels flag
			$thread->update([
				'hasLabels' => !empty($this->parameters['labelIDs']) ? 1 : 0
			]);
		}
		
		$assignedLabels = LabelHandler::getInstance()->getAssignedLabels($objectTypeID, $threadIDs);
		/** @var Label[] $labelList */
		$labelList = null;
		if (!empty($assignedLabels)) {
			// get labels from first object
			$labelList = reset($assignedLabels);
		}
		
		// log changes
		WCF::getDB()->beginTransaction();
		foreach ($this->getObjects() as $thread) {
			$groupedOldLabels = [];
			if (!empty($oldLabels[$thread->threadID])) {
				foreach ($oldLabels[$thread->threadID] as $oldLabel) {
					$groupedOldLabels[$oldLabel->groupID] = $oldLabel;
				}
			}
			
			if ($labelList !== null) {
				foreach ($labelList as $label) {
					if (!isset($groupedOldLabels[$label->groupID]) || $label->labelID != $groupedOldLabels[$label->groupID]->labelID) {
						ThreadModificationLogHandler::getInstance()->setLabel($thread->getDecoratedObject(), $label, (isset($groupedOldLabels[$label->groupID]) ? $groupedOldLabels[$label->groupID] : null));
					}
					if (isset($groupedOldLabels[$label->groupID])) unset($groupedOldLabels[$label->groupID]);
				}
			}
			foreach ($groupedOldLabels as $groupID => $label) {
				ThreadModificationLogHandler::getInstance()->setLabel($thread->getDecoratedObject(), null, $label);
			}
		}
		WCF::getDB()->commitTransaction();
		
		$labels = [];
		if ($labelList !== null) {
			$tmp = [];
			
			foreach ($labelList as $label) {
				$tmp[$label->labelID] = [
					'cssClassName' => $label->cssClassName,
					'label' => $label->getTitle(),
					'link' => LinkHandler::getInstance()->getLink(
						'Board',
						[
							'application' => 'wbb',
							'object' => $this->board,
						],
						'labelIDs['.$label->groupID.']='.$label->labelID
					),
				];
			}
			
			// sort labels by label group show order
			$labelGroups = ThreadLabelObjectHandler::getInstance()->getLabelGroups();
			foreach ($labelGroups as $labelGroup) {
				foreach ($tmp as $labelID => $labelData) {
					if ($labelGroup->isValid($labelID)) {
						$labels[] = $labelData;
						break;
					}
				}
			}
		}
		
		$this->unmarkItems($threadIDs);
		
		return [
			'labels' => $labels
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateGetSearchResultList() {
		$this->readString('searchString', false, 'data');
	}
	
	/**
	 * @inheritDoc
	 */
	public function getSearchResultList() {
		$threadList = new ThreadList();
		$threadList->applyBoardFilter();
		$threadList->getConditionBuilder()->add("topic LIKE ?", [$this->parameters['data']['searchString'].'%']);
		$threadList->sqlLimit = 5;
		$threadList->readObjects();
		
		$list = [];
		foreach ($threadList as $thread) {
			$list[] = [
				'label' => $thread->topic,
				'objectID' => $thread->threadID
			];
		}
		
		return $list;
	}
	
	/**
	 * Rebuilds discussion threads for articles.
	 */
	public function rebuildArticleThreads() {
		/** @var Article $article */
		$article = $this->parameters['article'];
		/** @var ArticleContent[] $articleContents */
		$articleContents = $this->parameters['articleContents'];
		
		// check board ids
		$board = null;
		$categoryIDs = [];
		if (WBB_ARTICLE_THREAD_SINGLE_BOARD) {
			$board = BoardCache::getInstance()->getBoard(WBB_ARTICLE_THREAD_BOARD_ID);
			if ($board === null || !$board->isBoard()) return;
			
			if (WBB_ARTICLE_THREAD_CATEGORIES) {
				$categoryIDs = explode("\n", WBB_ARTICLE_THREAD_CATEGORIES);
			}
		}
		
		// check categories
		if (!empty($categoryIDs) || !WBB_ARTICLE_THREAD_SINGLE_BOARD) {
			$result = false;
			
			if (WBB_ARTICLE_THREAD_SINGLE_BOARD) {
				if (in_array($article->categoryID, $categoryIDs)) {
					$result = true;
				}
			}
			else if ($article->getCategory()) {
				/** @noinspection PhpUndefinedFieldInspection */
				if ($article->getCategory()->articleThreadBoardID) {
					/** @noinspection PhpUndefinedFieldInspection */
					$board = BoardCache::getInstance()->getBoard($article->getCategory()->articleThreadBoardID);
					if ($board === null || !$board->isBoard()) {
						$board = null;
					}
					else {
						$result = true;
					}
				}
			}
			
			if (!$result) return;
		}
		
		foreach ($articleContents as $content) {
			$language = WCF::getLanguage();
			
			if ($content->languageID) {
				$language = LanguageFactory::getInstance()->getLanguage($content->languageID);
			}
			
			// get tags
			$tags = [];
			if (MODULE_TAGGING) {
				$tagObjects = TagEngine::getInstance()->getObjectTags(
					'com.woltlab.wcf.article',
					$content->articleContentID,
					[$content->languageID ?: LanguageFactory::getInstance()->getDefaultLanguageID()]
				);
				
				foreach ($tagObjects as $tagObject) {
					$tags[] = $tagObject->getTitle();
				}
			}
			
			// save support thread
			$htmlInputProcessor = new HtmlInputProcessor();
			$htmlInputProcessor->process($language->getDynamicVariable('wbb.thread.articleThread.message', ['articleContent' => $content]), 'com.woltlab.wbb.post');
			$topic = $language->getDynamicVariable('wbb.thread.articleThread.subject', ['articleContent' => $content]);
			
			$thread = null;
			/** @noinspection PhpUndefinedFieldInspection */
			if ($content->articleThreadID) {
				/** @noinspection PhpUndefinedFieldInspection */
				$thread = new Thread($content->articleThreadID);
				if (!$thread->threadID) $thread = null;
			}
			
			if ($thread === null) {
				$objectAction = new ThreadAction([], 'create', [
					'data' => [
						'boardID' => $board->boardID,
						'languageID' => (count(LanguageFactory::getInstance()->getContentLanguages()) ? $language->languageID : null),
						'topic' => $topic,
						'time' => $article->time,
						'userID' => $article->userID,
						'username' => $article->username
					],
					'postData' => [],
					'board' => $board,
					'tags' => $tags,
					'htmlInputProcessor' => $htmlInputProcessor
				]);
				$thread = $objectAction->executeAction()['returnValues'];
				
				// update support thread id
				$contentEditor = new ArticleContentEditor($content);
				$contentEditor->update([
					'articleThreadID' => $thread->threadID
				]);
				
				// mark thread as read
				$threadAction = new ThreadAction([$thread], 'markAsRead');
				$threadAction->executeAction();
			}
			else {
				// update first post
				(new PostAction([$thread->getFirstPost()], 'update', [
					'htmlInputProcessor' => $htmlInputProcessor
				]))->executeAction();
				
				// update the existing thread
				(new ThreadAction([$thread], 'update', ['data' => [
					'languageID' => $thread->languageID,
					'tags' => $tags,
					'topic' => $topic
				]]))->executeAction();
			}
		}
	}
	
	/**
	 * Loads threads for given object ids.
	 */
	protected function loadThreads() {
		if (empty($this->objectIDs)) {
			throw new UserInputException("objectIDs");
		}
		
		$this->readObjects();
		
		if (empty($this->objects)) {
			throw new UserInputException("objectIDs");
		}
	}
	
	/**
	 * Unmarks threads.
	 * 
	 * @param	integer[]	$threadIDs
	 */
	protected function unmarkItems(array $threadIDs = []) {
		if (empty($threadIDs)) {
			foreach ($this->getObjects() as $thread) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		if (!empty($threadIDs)) {
			ClipboardHandler::getInstance()->unmark($threadIDs, ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.thread'));
		}
	}
	
	/**
	 * Adds thread data.
	 * 
	 * @param	Thread		$thread
	 * @param	string		$key
	 * @param	mixed		$value
	 */
	protected function addThreadData(Thread $thread, $key, $value) {
		if (!isset($this->threadData[$thread->threadID])) {
			$this->threadData[$thread->threadID] = [];
		}
		
		$this->threadData[$thread->threadID][$key] = $value;
	}
	
	/**
	 * Returns thread data.
	 * 
	 * @return	mixed[][]
	 */
	protected function getThreadData() {
		return [
			'threadData' => $this->threadData
		];
	}
	
	/**
	 * Updates last posts per board.
	 * 
	 * @param	integer[]	$boardIDs
	 */
	protected function updateLastPost(array $boardIDs) {
		if (empty($boardIDs)) {
			return;
		}
		
		$boardIDs = array_unique($boardIDs);
		foreach ($boardIDs as $boardID) {
			$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
			$boardEditor->updateLastPost();
		}
		
		BoardEditor::resetDataCache();
	}
	
	/**
	 * Removes user activity events and activity points for threads and posts.
	 * 
	 * @param	integer[]	$threadData
	 * @param	boolean		$ignorePosts
	 */
	protected function removeActivityEvents(array $threadData, $ignorePosts = false) {
		$threadIDs = array_keys($threadData);
		$isBulkProcessing = $this->isBulkProcessing();
		
		$skipActivityPointThreadIDs = [];
		
		// Remove ids of thread in in boards that do not count posts.
		if (!$isBulkProcessing) {
			$sql = "SELECT  threadID
				FROM    wbb" . WCF_N . "_thread
				WHERE   threadID IN (?" . str_repeat(',?', count($threadIDs) - 1) . ")
					AND boardID IN (
						SELECT  DISTINCT boardID
						FROM    wbb" . WCF_N . "_board
						WHERE   countUserPosts = ?
					)";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute(array_merge($threadIDs, [0]));
			$skipActivityPointThreadIDs = $statement->fetchList('threadID');
		}
		
		// remove thread activity events
		UserActivityEventHandler::getInstance()->removeEvents('com.woltlab.wbb.recentActivityEvent.thread', $threadIDs);
		
		// remove thread activity points
		if (!$isBulkProcessing) {
			$userToItems = [];
			foreach ($threadData as $threadID => $userID) {
				if (!$userID || in_array($threadID, $skipActivityPointThreadIDs)) {
					continue;
				}
				
				if (!isset($userToItems[$userID])) {
					$userToItems[$userID] = 0;
				}
				$userToItems[$userID]++;
			}
			
			if (!empty($userToItems)) {
				UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.thread', $userToItems);
			}
		}
		
		// remove posts
		if (!$ignorePosts) {
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add('post.threadID IN (?)', [$threadIDs]);
			// ignore first posts of threads since they neither have
			// own activity events nor own activity points
			$conditions->add('post.postID <> thread.firstPostID');
			
			$sql = "SELECT		post.postID, post.threadID, post.userID
				FROM		wbb".WCF_N."_post post
				LEFT JOIN	wbb".WCF_N."_thread thread
				ON		(thread.threadID = post.threadID)
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			$postIDs = [];
			$userToItems = [];
			while ($row = $statement->fetchArray()) {
				$postIDs[] = $row['postID'];
				
				if (!$isBulkProcessing && $row['userID'] && !in_array($row['threadID'], $skipActivityPointThreadIDs)) {
					if (!isset($userToItems[$row['userID']])) {
						$userToItems[$row['userID']] = 0;
					}
					$userToItems[$row['userID']]++;
				}
			}
			
			UserActivityEventHandler::getInstance()->removeEvents('com.woltlab.wbb.recentActivityEvent.post', $postIDs);
			
			if (!$isBulkProcessing && !empty($userToItems)) {
				UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $userToItems);
			}
		}
	}
	
	/**
	 * Moves all posts of a thread to trash bin.
	 * 
	 * @param	integer[]	$threadIDs
	 * @param	boolean		$isBulkProcessing
	 */
	protected function trashPosts(array $threadIDs, $isBulkProcessing = false) {
		$postList = new PostList();
		$postList->getConditionBuilder()->add("post.threadID IN (?)", [$threadIDs]);
		$postList->readObjects();
		
		$postAction = new PostAction($postList->getObjects(), 'trash', [
			'ignoreThreads' => true,
			'isBulkProcessing' => $isBulkProcessing,
			'ignorePostModificationLogs' => true
		]);
		$postAction->executeAction();
	}
	
	/**
	 * Restores all posts of a thread without creating modification log entries.
	 * 
	 * @param	integer[]	$threadIDs
	 * @param	boolean		$isBulkProcessing
	 */
	protected function restorePosts(array $threadIDs, $isBulkProcessing = false) {
		$postList = new PostList();
		$postList->getConditionBuilder()->add("post.threadID IN (?)", [$threadIDs]);
		$postList->readObjects();
		
		$postAction = new PostAction($postList->getObjects(), 'restore', [
			'ignoreThreads' => true,
			'isBulkProcessing' => $isBulkProcessing,
			'ignorePostModificationLogs' => true
		]);
		$postAction->executeAction();
	}
	
	/**
	 * Returns `true` if the relevant action is executed in bulk processing mode and returns
	 * `false` otherwise.
	 * 
	 * In bulk processing mode, updates that can be achieved using rebuild data workers should
	 * not be executed. Additionally, modification log entries should not be created, thread data
	 * should not be registered, and no clipboard items should be unmarked.
	 * 
	 * @return	boolean
	 */
	protected function isBulkProcessing() {
		return isset($this->parameters['isBulkProcessing']) && $this->parameters['isBulkProcessing'];
	}
}
