<?php
namespace wbb\data\thread;
use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\post\Post;
use wbb\data\post\ViewablePost;
use wbb\system\cache\runtime\PostRuntimeCache;
use wcf\data\IPopoverObject;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\user\object\watch\UserObjectWatch;
use wcf\data\DatabaseObject;
use wcf\data\IUserContent;
use wcf\data\TUserContent;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\request\IRouteController;
use wcf\system\request\LinkHandler;
use wcf\system\search\SearchEngine;
use wcf\system\WCF;

/**
 * Represents a thread.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 * 
 * @property-read	integer		$threadID		unique id of the thread
 * @property-read	integer		$boardID		id of the board in which the thread is located
 * @property-read	integer|null	$languageID		id of the thread's language or null if no language has been set
 * @property-read	string		$topic			topic of the thread
 * @property-read	integer		$firstPostID		id of the thread's first post with the same isDeleted and isDisabled state as the thread itself
 * @property-read	integer		$time			timestamp at which the thread has been created
 * @property-read	integer|null	$userID			id of the user who created the thread or null if the user does not exist anymore or if thread has been created by guest
 * @property-read	string		$username		name of the user or guest who created the thread
 * @property-read	integer		$lastPostID		id of the thread's last post with the same isDeleted and isDisabled state as the thread itself
 * @property-read	integer		$lastPostTime		timestamp at which the thread's last post has been created
 * @property-read	integer|null	$lastPosterID		id of the user who created the thread's last post or null if the user does not exist anymore or if last post has been written by guest
 * @property-read	string		$lastPoster		name of the user who created the thread's last post
 * @property-read	integer		$replies		number of non-deleted and non-disabled posts in thread
 * @property-read	integer		$views			number of times the thread has been viewed
 * @property-read	integer		$attachments		number of attachments in the thread
 * @property-read	integer		$polls			number of polls in the thread
 * @property-read	integer		$isAnnouncement		is 1 if the thread is an announcement, otherwise 0
 * @property-read	integer		$isSticky		is 1 if the thread is sticky, otherwise 0
 * @property-read	integer		$isDisabled		is 1 if the thread is disabled, otherwise 0
 * @property-read	integer		$isClosed		is 1 if no new posts can be written in thread, otherwise 0
 * @property-read	integer		$isDeleted		is 1 if the thread is in trash bin, otherwise 0
 * @property-read	integer|null	$movedThreadID		id of the (moved) thread this thread just redirects to or null if thread does not direct to another thread
 * @property-read	integer		$movedTime		timestamp at which the thread has been moved or 0 if thread has not been moved
 * @property-read	integer		$isDone			is 1 if the thread is marked as done, otherwise 0
 * @property-read	integer		$cumulativeLikes	cumulative result of likes (counting +1) and dislikes (counting -1) of the thread's first post
 * @property-read	integer		$hasLabels		is 1 if the thread has assigned labels, otherwise 0
 * @property-read	integer		$deleteTime		timestamp at which the thread has been deleted
 * @property-read	integer		$bestAnswerPostID	id of the thread's best answer
 */
class Thread extends DatabaseObject implements IPopoverObject, IRouteController, IUserContent {
	use TUserContent;
	
	/**
	 * true, if the active user has subscribed this topic
	 * @var	boolean
	 */
	protected $subscribed;
	
	/**
	 * board object
	 * @var	Board
	 */
	public $board;
	
	/**
	 * original thread
	 * @var	Thread
	 */
	public $movedThread;
	
	/**
	 * first post object
	 * @var	Post
	 */
	public $firstPost;
	
	/**
	 * best answer post object
	 * @var	ViewablePost
	 */
	public $bestAnswerPost;
	
	// thread types (internal use)
	const TYPE_DEFAULT = 0;
	const TYPE_STICKY = 1;
	const TYPE_ANNOUNCEMENT = 2;
	
	/**
	 * Returns true if the active user has the permission to read this thread.
	 * 
	 * @return	boolean
	 */
	public function canRead() {
		$board = $this->getBoard();
		
		if (!$board->canEnter()) {
			return false;
		}
		
		if (!$board->getPermission('canReadThread') || ($this->isDeleted && !$board->getModeratorPermission('canReadDeletedThread')) || ($this->isDisabled && !$board->getModeratorPermission('canEnableThread') && (!$this->userID || $this->userID != WCF::getUser()->userID))) {
			return false;
		}
		
		if ($board->isPrivate) {
			if (!WCF::getUser()->userID) {
				return false;
			}
			
			if (!$board->canReadPrivateThreads() && $this->userID != WCF::getUser()->userID) {
				return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Returns true if the active user can reply this thread.
	 * 
	 * @param	boolean		$checkForDoublePostOnly
	 * @return	boolean
	 */
	public function canReply($checkForDoublePostOnly = null) {
		if ($checkForDoublePostOnly !== true) {
			$board = $this->getBoard();
			
			// check permission
			$canReply = (!$board->isClosed && (($this->isClosed && $board->getModeratorPermission('canReplyClosedThread'))
				|| (!$this->isClosed && ($board->getPermission('canReplyThread') || ($this->userID && $this->userID == WCF::getUser()->userID && $board->getPermission('canReplyOwnThread'))))));
			if (!$canReply) return false;
		}
		
		return true;
	}
	
	/**
	 * Returns true if the active user can edit the given post.
	 * 
	 * @param	Post	$post
	 * @return	boolean
	 */
	public function canEditPost(Post $post) {
		if ($post->threadID != $this->threadID) return false;
		
		// get board
		$board = $this->getBoard();
		
		// check permissions
		$isModerator = $board->getModeratorPermission('canEditPost') || $board->getModeratorPermission('canDeletePost');
		$isAuthor = $post->userID && $post->userID == WCF::getUser()->userID;
		
		$canEditPost = $board->getModeratorPermission('canEditPost') || $isAuthor && $board->getPermission('canEditOwnPost');
		$canDeletePost = $board->getModeratorPermission('canDeletePost') || $isAuthor && $board->getPermission('canDeleteOwnPost');
		
		if ((!$canEditPost && !$canDeletePost) || (!$isModerator && ($board->isClosed || $this->isClosed || $post->isClosed || $post->isDeleted))) {
			return false;
		}
		
		// check post edit timeout
		if (!$isModerator && WCF::getSession()->getPermission('user.board.postEditTimeout') != -1 && TIME_NOW - $post->time > WCF::getSession()->getPermission('user.board.postEditTimeout') * 60) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns true if active user can mark this thread as done or undone.
	 * 
	 * @return	boolean
	 */
	public function canMarkAsDone() {
		if (!WBB_MODULE_THREAD_MARKING_AS_DONE) {
			return false;
		}
		
		// get board
		$board = $this->getBoard();
		
		// check configuration
		if (!$board->enableMarkingAsDone) {
			return false;
		}
		
		// check permissions
		if ($board->getModeratorPermission('canMarkAsDoneThread')) {
			return true;
		}
		
		if (($this->userID == WCF::getUser()->userID) && $board->getPermission('canMarkAsDoneOwnThread')) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Returns true if active user can mark a post as best answer.
	 *
	 * @return	boolean
	 * @since       5.2
	 */
	public function canMarkBestAnswer() {
		$board = $this->getBoard();
		
		// check configuration
		if (!$board->enableBestAnswer) {
			return false;
		}
		
		// check permissions
		if ($board->getModeratorPermission('canMarkBestAnswer')) {
			return true;
		}
		
		if ($this->userID && ($this->userID === WCF::getUser()->userID) && !$this->isClosed && $board->getPermission('canMarkBestAnswerOwnThread')) {
			return true;
		}
		
		return false;
	}
	
	/**
	 * Returns true if current user can edit this thread, effectively edit the first post.
	 * 
	 * @return	boolean
	 */
	public function canEdit() {
		if (!$this->threadID) return false;
		
		// get board
		$board = $this->getBoard();
		
		// check permissions
		$isModerator = $board->getModeratorPermission('canEditPost') || $board->getModeratorPermission('canDeletePost');
		$isAuthor = $this->userID && $this->userID == WCF::getUser()->userID;
		
		$canEditPost = $board->getModeratorPermission('canEditPost') || $isAuthor && $board->getPermission('canEditOwnPost');
		$canDeletePost = $board->getModeratorPermission('canDeletePost') || $isAuthor && $board->getPermission('canDeleteOwnPost');
		
		if ((!$canEditPost && !$canDeletePost) || (!$isModerator && ($board->isClosed || $this->isClosed))) {
			return false;
		}
		
		// check post edit timeout
		if (!$isModerator && WCF::getSession()->getPermission('user.board.postEditTimeout') != -1 && TIME_NOW - $this->time > WCF::getSession()->getPermission('user.board.postEditTimeout') * 60) {
			return false;
		}
		
		return true;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		return $this->topic;
	}
	
	/**
	 * Returns associated board object.
	 * 
	 * @return	Board
	 */
	public function getBoard() {
		if ($this->board === null) {
			$this->board = BoardCache::getInstance()->getBoard($this->boardID);
		}
		
		return $this->board;
	}
	
	/**
	 * Sets associated board object.
	 * 
	 * @param	Board	$board
	 */
	public function setBoard(Board $board) {
		if ($board->boardID == $this->boardID) {
			$this->board = $board;
		}
	}
	
	/**
	 * Returns moved thread or null if not moved.
	 * 
	 * @return	Thread
	 */
	public function getMovedThread() {
		if ($this->movedThreadID) {
			if ($this->movedThread === null) {
				$this->movedThread = new Thread($this->movedThreadID);
			}
			
			return $this->movedThread;
		}
		
		return null;
	}
	
	/**
	 * Returns a list of board ids for this announcement.
	 * 
	 * @return	integer[]
	 */
	public function getAnnouncementBoardIDs() {
		if ($this->isAnnouncement) {
			$sql = "SELECT	boardID
				FROM	wbb".WCF_N."_thread_announcement
				WHERE	threadID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->threadID]);
			
			return $statement->fetchAll(\PDO::FETCH_COLUMN);
		}
		
		return [];
	}
	
	/**
	 * Returns true if the active user has subscribed this topic.
	 * 
	 * @return	boolean
	 */
	public function isSubscribed() {
		if ($this->subscribed === null) {
			$objectType = ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.user.objectWatch', 'com.woltlab.wbb.thread');
			if (UserObjectWatch::getUserObjectWatch($objectType->objectTypeID, WCF::getUser()->userID, $this->threadID) !== null) $this->subscribed = true;
			else $this->subscribed = false;
		}
		
		return $this->subscribed;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return LinkHandler::getInstance()->getLink('Thread', [
			'application' => 'wbb',
			'object' => $this,
			'forceFrontend' => true
		]);
	}
	
	/**
	 * Returns the first post object.
	 * 
	 * @return	Post
	 */
	public function getFirstPost() {
		if ($this->firstPost === null && $this->firstPostID) {
			$this->firstPost = PostRuntimeCache::getInstance()->getObject($this->firstPostID);
			$this->firstPost->setThread($this);
		}
		
		return $this->firstPost;
	}
	
	/**
	 * Sets the first post of the thread.
	 * 
	 * @param	Post	$firstPost
	 */
	public function setFirstPost(Post $firstPost) {
		$this->firstPost = $firstPost;
	}
	
	/**
	 * Returns the best answer post object.
	 * 
	 * @return	ViewablePost
	 * @since       5.2
	 */
	public function getBestAnswerPost() {
		if ($this->bestAnswerPost === null && $this->bestAnswerPostID) {
			$post = new Post($this->bestAnswerPostID);
			$post->setThread($this);
			$this->bestAnswerPost = new ViewablePost($post);
		}
		
		return $this->bestAnswerPost;
	}
	
	/**
	 * Sets the best answer post of the thread.
	 * 
	 * @param	ViewablePost	$bestAnswerPost
	 * @since       5.2
	 */
	public function setBestAnswerPost(ViewablePost $bestAnswerPost) {
		$this->bestAnswerPost = $bestAnswerPost;
	}
	
	/**
	 * Returns true, if the thread has a best answer. 
	 * 
	 * @return      boolean
	 * @since       5.2
	 */
	public function hasBestAnswer() {
		$board = $this->getBoard();
		
		// check configuration
		if (!$board->enableBestAnswer) {
			return false;
		}
		
		return $this->bestAnswerPostID !== null;
	}
	
	/**
	 * Returns the ids of similar threads.
	 * 
	 * @param	string			$topic
	 * @param	integer[]		$accessibleBoardIDs
	 * @param	integer			$limit
	 * @param	integer			$preferredBoardID
	 * @param	integer			$ignoreThreadID
	 * @param	integer			$languageID
	 * @return	integer[]
	 */
	public static function getSimilarThreads($topic, array $accessibleBoardIDs, $limit = 5, $preferredBoardID = 0, $ignoreThreadID = 0, $languageID = null) {
		// kick out characters with a special meaning, they may negatively affect result quality
		// e.g. the topic 'Martin Luther King said: "I have a dream"' won't be matched with threads
		// that contain the words 'dream'
		$topic = SearchEngine::getInstance()->removeSpecialCharacters($topic);
		
		if (empty($topic)) {
			// removing special characters can cause an empty search string (e.g. $topic = "---")
			return [];
		}
		
		$className = SearchEngine::getInstance()->getConditionBuilderClassName();
		
		/** @var PreparedStatementConditionBuilder $searchIndexCondition */
		$searchIndexCondition = new $className(false);
		$searchIndexCondition->add("time > ?", [TIME_NOW - WBB_THREAD_SIMILAR_THREADS_SEARCH_PERIOD * 86400]);
		if ($languageID) $searchIndexCondition->add("languageID = ?", [$languageID]);
		
		$innerJoin = SearchEngine::getInstance()->getInnerJoin('com.woltlab.wbb.post', $topic, false, $searchIndexCondition, 'relevance DESC', 1000);
		
		// get post ids
		$threadIDs = [];
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('post.isDisabled = 0 AND post.isDeleted = 0');
		if ($ignoreThreadID) $conditionBuilder->add('post.threadID <> ?', [$ignoreThreadID]);
		$conditionBuilder->add('thread.boardID IN (?)', [$accessibleBoardIDs]);
		$sql = "SELECT		post.threadID,
					search_index.relevance ".($preferredBoardID ? "+ IF(thread.boardID=".$preferredBoardID.",2,0)" : '')." AS relevance
			FROM		wbb".WCF_N."_post post
			INNER JOIN	(
						".$innerJoin['sql']."
					) search_index
			ON		(post.postID = search_index.objectID)
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = post.threadID)
			".$conditionBuilder."
			ORDER BY	relevance DESC";
		$statement = WCF::getDB()->prepareStatement($sql, $limit);
		
		$parameters = [];
		if ($innerJoin['fulltextCondition'] !== null) {
			/** @noinspection PhpUndefinedMethodInspection */
			$parameters = $innerJoin['fulltextCondition']->getParameters();
		}
		if ($innerJoin['searchIndexCondition'] !== null) {
			/** @noinspection PhpUndefinedMethodInspection */
			$parameters = array_merge($parameters, $innerJoin['searchIndexCondition']->getParameters());
		}
		$parameters = array_merge($parameters, $conditionBuilder->getParameters());
		
		$statement->execute($parameters);
		while ($row = $statement->fetchArray()) {
			if (!in_array($row['threadID'], $threadIDs)) $threadIDs[] = $row['threadID'];
		}
		
		return $threadIDs;
	}
	
	/**
	 * Returns true if current user can create a poll. Optionally checks for
	 * polls being created in replies.
	 * 
	 * @param       boolean         $inReply        create poll when replying
	 * @param       Post            $post           post object
	 * @return      boolean         true if user is allowed to create a poll
	 */
	public function canUsePoll($inReply, Post $post = null) {
		if (MODULE_POLL && $this->getBoard()->getPermission('canStartPoll')) {
			if ($post !== null && $post->isFirstPost()) {
				return true;
			}
			
			if ($inReply && !WCF::getSession()->getPermission('user.board.canStartPollInReply')) {
				return false;
			}
			
			return true;
		}
		
		return false;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getPopoverLinkClass() {
		return 'wbbTopicLink';
	}
}
