<?php
namespace wbb\data\thread;
use wcf\data\tag\Tag;
use wcf\system\tagging\TagEngine;
use wcf\system\tagging\TTaggedObjectList;
use wcf\system\WCF;

/**
 * Represents a list of accessible threads.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 */
class TaggedThreadList extends AccessibleThreadList {
	use TTaggedObjectList;
	
	/**
	 * @var Tag[]
	 */
	public $tags;
	
	/**
	 * Creates a new TaggedThreadList object.
	 * 
	 * @param Tag|Tag[] $tags
	 */
	public function __construct($tags) {
		parent::__construct();
		
		$this->tags = ($tags instanceof Tag) ? [$tags] : $tags;
		
		$this->getConditionBuilder()->add('tag_to_object.objectTypeID = ? AND tag_to_object.tagID IN (?)', [
			TagEngine::getInstance()->getObjectTypeID('com.woltlab.wbb.thread'),
			TagEngine::getInstance()->getTagIDs($this->tags),
		]);
		$this->getConditionBuilder()->add('thread.threadID = tag_to_object.objectID');
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function countObjects() {
		$sql = "SELECT  COUNT(*)
			FROM    (
				SELECT	        tag_to_object.objectID
				FROM	        wcf".WCF_N."_tag_to_object tag_to_object,
						wbb".WCF_N."_thread thread
				".$this->sqlConditionJoins."
				".$this->getConditionBuilder()."
				GROUP BY        tag_to_object.objectID
				HAVING          COUNT(tag_to_object.objectID) = ?
			) AS t";
		$statement = WCF::getDB()->prepareStatement($sql);
		
		$parameters = $this->getConditionBuilder()->getParameters();
		$parameters[] = count($this->tags);
		$statement->execute($parameters);
		
		return $statement->fetchSingleColumn();
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function readObjectIDs() {
		$this->objectIDs = [];
		$sql = "SELECT	tag_to_object.objectID
			FROM	wcf".WCF_N."_tag_to_object tag_to_object,
				wbb".WCF_N."_thread thread
			".$this->sqlConditionJoins."
			".$this->getConditionBuilder()."
			".$this->getGroupByFromOrderBy('tag_to_object.objectID', $this->sqlOrderBy)."
			HAVING          COUNT(tag_to_object.objectID) = ?
			".(!empty($this->sqlOrderBy) ? "ORDER BY ".$this->sqlOrderBy : '');
		$statement = WCF::getDB()->prepareStatement($sql, $this->sqlLimit, $this->sqlOffset);
		
		$parameters = $this->getConditionBuilder()->getParameters();
		$parameters[] = count($this->tags);
		$statement->execute($parameters);
		$this->objectIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
	}
}
