<?php
namespace wbb\data\thread;
use wbb\system\cache\builder\MostActiveThreadsCacheBuilder;
use wbb\data\board\Board;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;

/**
 * Represents a list of the most active threads
 * 
 * @author	Tim Duesterhus, Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 */
class MostActiveThreadList extends ViewableThreadList {
	/**
	 * timeframe to select threads from (in days)
	 */
	public $timeframe = 30;
	
	/**
	 * @inheritDoc
	 */
	public $sqlLimit = 5;
	
	/**
	 * @inheritDoc
	 */
	public $sqlOrderBy = 'thread.replies DESC, thread.lastPostTime DESC';
	
	/**
	 * Creates a new MostActiveThreadList object.
	 * 
	 * @param	integer		$boardID
	 */
	public function __construct($boardID = 0) {
		parent::__construct();
		$this->objectIDs = [];
		
		// get cache
		$cachedThreadIDs = $this->getThreadIDs($boardID);
		if (!count($cachedThreadIDs)) return;
		
		// add conditions
		$this->getConditionBuilder()->add('thread.threadID IN (?)', [$cachedThreadIDs]);
		
		if (!$boardID) {
			if (!$this->applyBoardFilter(true)) {
				return;
			}
		}
		
		$this->addConditions();
		
		// filter cache threads
		$this->readObjectIDs();
		
		// cached threads didn't contain the latest X threads for the active user
		// fallback to non-cached function
		if (count($this->objectIDs) < $this->sqlLimit && count($cachedThreadIDs) >= $this->sqlLimit) {
			$this->objectIDs = [];
			$this->conditionBuilder = new PreparedStatementConditionBuilder();
			if ($boardID) {
				if (in_array($boardID, Board::getPrivateBoardIDs())) {
					$this->getConditionBuilder()->add("(thread.boardID = ? AND thread.userID = ?)", [$boardID, WCF::getUser()->userID]);
				}
				else {
					$this->getConditionBuilder()->add('thread.boardID = ?', [$boardID]);
				}
			}
			else {
				$this->applyBoardFilter(true);
			}
			$this->getConditionBuilder()->add('thread.time > ?', [TIME_NOW - ($this->timeframe * 86400)]);
			$this->addConditions();
			
			$this->readObjectIDs();
		}
	}
	
	/**
	 * Adds the general conditions every thread (cached and non-cached)
	 * must fulfil.
	 */
	protected function addConditions() {
		$this->getConditionBuilder()->add('thread.isDeleted = 0 AND thread.isDisabled = 0 AND thread.movedThreadID IS NULL');
		if (LanguageFactory::getInstance()->multilingualismEnabled() && count(WCF::getUser()->getLanguageIDs())) {
			$this->getConditionBuilder()->add('(thread.languageID IN (?) OR thread.languageID IS NULL)', [WCF::getUser()->getLanguageIDs()]);
		}
	}
	
	/**
	 * Returns the cached thread ids for the given board.
	 * 
	 * @param	integer		$boardID
	 * @return	integer[]
	 */
	protected function getThreadIDs($boardID) {
		return array_keys(MostActiveThreadsCacheBuilder::getInstance()->getData($boardID ? ['boardID' => $boardID] : []));
	}
}
