<?php
namespace wbb\data\thread;
use wbb\data\board\Board;
use wbb\data\modification\log\BoardModificationLogList;
use wcf\data\object\type\ObjectTypeCache;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\SystemException;
use wcf\system\label\LabelHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;

/**
 * Provides extended functions for displaying a list of threads.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Thread
 */
class BoardThreadList extends ViewableThreadList {
	/**
	 * board object
	 * @var	Board
	 */
	public $board;
	
	/**
	 * days prune
	 * @var	integer
	 */
	public $daysPrune = 100;
	
	/**
	 * label ids
	 * @var	integer[]
	 */
	public $labelIDs = [];
	
	/**
	 * board modification log list
	 * @var	BoardModificationLogList
	 */
	public $logList;
	
	/**
	 * loads deletion data
	 * @var	boolean
	 */
	public $loadDeleteNote = true;
	
	/**
	 * thread status
	 * @var	string
	 */
	public $status = '';
	
	/**
	 * thread language
	 * @var	integer
	 */
	public $languageID = 0;
	
	/**
	 * sql conditions for announcements
	 * @var	PreparedStatementConditionBuilder
	 */
	protected $announcementConditionBuilder;
	
	/**
	 * Creates a new BoardThreadList object.
	 * 
	 * @param	Board		$board
	 * @param	integer		$daysPrune
	 * @param	integer[]	$labelIDs
	 * @param	integer		$tagID
	 * @param	string		$status
	 * @param	integer		$languageID
	 * @throws	SystemException
	 */
	public function __construct(Board $board, $daysPrune = 100, $labelIDs = [], $tagID = 0, $status = '', $languageID = 0) {
		$this->board = $board;
		$this->daysPrune = $daysPrune;
		$this->labelIDs = $labelIDs;
		$this->status = $status;
		$this->languageID = $languageID;
		
		parent::__construct();
		$this->announcementConditionBuilder = new PreparedStatementConditionBuilder();
		
		// add conditions
		$this->getConditionBuilder()->add('thread.boardID = ? AND thread.isAnnouncement = 0', [$this->board->boardID]);
		$this->announcementConditionBuilder->add('announcement.boardID = ?', [$this->board->boardID]);
		
		// days prune
		if ($this->daysPrune != 1000) {
			$this->getConditionBuilder()->add("((thread.isSticky = 0 AND thread.lastPostTime >= ?) OR (thread.isSticky = 1 AND thread.lastPostTime >= 0))", [TIME_NOW - ($this->daysPrune * 86400)]);
		}
		
		// visible status
		if (!$this->board->getModeratorPermission('canReadDeletedThread') && !WBB_BOARD_ENABLE_DELETED_THREAD_NOTE) {
			$this->getConditionBuilder()->add('thread.isDeleted = 0');
			$this->announcementConditionBuilder->add('thread.isDeleted = 0');
			
			$this->loadDeleteNote = false;
		}
		if (!$this->board->getModeratorPermission('canEnableThread')) {
			if (WCF::getUser()->userID) {
				$this->getConditionBuilder()->add('(thread.isDisabled = 0 OR thread.userID = ?)', [WCF::getUser()->userID]);
				$this->announcementConditionBuilder->add('(thread.isDisabled = 0 OR thread.userID = ?)', [WCF::getUser()->userID]);
			}
			else {
				$this->getConditionBuilder()->add('thread.isDisabled = 0');
				$this->announcementConditionBuilder->add('thread.isDisabled = 0');
			}
		}
		if ($this->board->isPrivate && !$this->board->canReadPrivateThreads()) {
			if (WCF::getUser()->userID) {
				$this->getConditionBuilder()->add("thread.userID = ?", [WCF::getUser()->userID]);
			}
			else {
				$this->getConditionBuilder()->add("1=0");
			}
		}
		
		// label ids
		if (!empty($labelIDs)) {
			// get object type id
			$objectType = ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.label.object', 'com.woltlab.wbb.thread');
			if ($objectType === null) {
				throw new SystemException("Missing object type for 'com.woltlab.wcf.label.objectType'");
			}
			
			foreach ($labelIDs as $groupID => $labelID) {
				if ($labelID == -1) {
					$groupLabelIDs = LabelHandler::getInstance()->getLabelGroup($groupID)->getLabelIDs();
					
					if (!empty($groupLabelIDs)) {
						$this->getConditionBuilder()->add('thread.threadID NOT IN (SELECT objectID FROM wcf'.WCF_N.'_label_object WHERE objectTypeID = ? AND labelID IN (?))', [
							$objectType->objectTypeID,
							$groupLabelIDs
						]);
					}
				}
				else {
					$this->getConditionBuilder()->add('thread.threadID IN (SELECT objectID FROM wcf'.WCF_N.'_label_object WHERE objectTypeID = ? AND labelID = ?)', [$objectType->objectTypeID, $labelID]);
				}
			}
		}
		
		// tag id
		if ($tagID) {
			// get object type id
			$objectType = ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.tagging.taggableObject', 'com.woltlab.wbb.thread');
			if ($objectType === null) {
				throw new SystemException("Missing object type for 'com.woltlab.wcf.tagging.taggableObject'");
			}
			
			$this->getConditionBuilder()->add('thread.threadID IN (SELECT objectID FROM wcf'.WCF_N.'_tag_to_object WHERE objectTypeID = ? AND tagID = ?)', [$objectType->objectTypeID, $tagID]);
		}
		
		// thread language
		if ($this->languageID) {
			$this->getConditionBuilder()->add('thread.languageID = ?', [$this->languageID]);
		}
		else if (LanguageFactory::getInstance()->multilingualismEnabled() && count(WCF::getUser()->getLanguageIDs())) {
			$this->getConditionBuilder()->add('(thread.languageID IN (?) OR thread.languageID IS NULL)', [WCF::getUser()->getLanguageIDs()]);
			$this->announcementConditionBuilder->add('(thread.languageID IN (?) OR thread.languageID IS NULL)', [WCF::getUser()->getLanguageIDs()]);
		}
		
		// enable time (delayed posts)
		if ($this->board->getModeratorPermission('canEnableThread')) {
			if (!empty($this->sqlSelects)) $this->sqlSelects .= ',';
			$this->sqlSelects .= 'first_post.enableTime';
			$this->sqlJoins .= " LEFT JOIN wbb".WCF_N."_post first_post ON (first_post.postID = thread.firstPostID)";
		}
		
		// status filter
		if (!empty($this->status)) {
			switch ($this->status) {
				case 'read':
					if (WCF::getUser()->userID) {
						if (!empty($this->sqlConditionJoins)) $this->sqlConditionJoins .= '';
						$this->sqlConditionJoins = "	LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_thread_visit
										ON		(tracked_thread_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')." AND tracked_thread_visit.objectID = thread.threadID AND tracked_thread_visit.userID = ".WCF::getUser()->userID.")
										LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_board_visit
										ON		(tracked_board_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board')." AND tracked_board_visit.objectID = thread.boardID AND tracked_board_visit.userID = ".WCF::getUser()->userID.")";
						
						$this->getConditionBuilder()->add('(thread.lastPostTime <= ? OR thread.lastPostTime <= tracked_thread_visit.visitTime OR thread.lastPostTime <= tracked_board_visit.visitTime)', [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]);
						$this->getConditionBuilder()->add('thread.movedThreadID IS NULL');
					}
				break;
				
				case 'unread':
					if (WCF::getUser()->userID) {
						if (!empty($this->sqlConditionJoins)) $this->sqlConditionJoins .= '';
						$this->sqlConditionJoins = "	LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_thread_visit
										ON		(tracked_thread_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')." AND tracked_thread_visit.objectID = thread.threadID AND tracked_thread_visit.userID = ".WCF::getUser()->userID.")
										LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_board_visit
										ON		(tracked_board_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board')." AND tracked_board_visit.objectID = thread.boardID AND tracked_board_visit.userID = ".WCF::getUser()->userID.")";
						
						$this->getConditionBuilder()->add('thread.lastPostTime > ?', [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]);
						$this->getConditionBuilder()->add("(thread.lastPostTime > tracked_thread_visit.visitTime OR tracked_thread_visit.visitTime IS NULL)");
						$this->getConditionBuilder()->add("(thread.lastPostTime > tracked_board_visit.visitTime OR tracked_board_visit.visitTime IS NULL)");
						$this->getConditionBuilder()->add('thread.movedThreadID IS NULL');
					}
				break;
				
				case 'open':
				case 'closed':
					$this->getConditionBuilder()->add('thread.isClosed = ?', [$this->status == 'open' ? 0 : 1]);
				break;
				
				case 'deleted':
					if ($this->board->getModeratorPermission('canReadDeletedThread')) {
						$this->getConditionBuilder()->add('thread.isDeleted = 1');
					}
				break;
				
				case 'disabled':
					if ($this->board->getModeratorPermission('canEnableThread')) {
						$this->getConditionBuilder()->add('thread.isDisabled = 1');
					}
				break;
				
				case 'done':
				case 'undone':
					if (WBB_MODULE_THREAD_MARKING_AS_DONE && $this->board->enableMarkingAsDone) {
						$this->getConditionBuilder()->add('thread.isDone = ?', [$this->status == 'done' ? 1 : 0]);
					}
				break;
			}
		}
	}
	
	/**
	 * Returns number of announcements.
	 * 
	 * @return	integer
	 */
	public function countAnnouncements() {
		// get announcement ids
		$sql = "SELECT		COUNT(announcement.threadID) AS count
			FROM		wbb".WCF_N."_thread_announcement announcement
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = announcement.threadID)
			".$this->announcementConditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($this->announcementConditionBuilder->getParameters());
		$row = $statement->fetchArray();
		
		return $row['count'];
	}
	
	/**
	 * @inheritDoc
	 */
	public function readObjectIDs() {
		$this->sqlOrderBy = 'thread.isAnnouncement DESC, thread.isSticky DESC, ' . $this->sqlOrderBy;
		
		parent::readObjectIDs();
		
		// get announcement ids
		$sql = "SELECT		announcement.threadID
			FROM		wbb".WCF_N."_thread_announcement announcement
			LEFT JOIN	wbb".WCF_N."_thread thread
			ON		(thread.threadID = announcement.threadID)
			".$this->announcementConditionBuilder;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($this->announcementConditionBuilder->getParameters());
		$this->objectIDs = array_merge($this->objectIDs, $statement->fetchAll(\PDO::FETCH_COLUMN));
	}
	
	/**
	 * @inheritDoc
	 */
	public function readObjects() {
		parent::readObjects();
		
		$threadIDs = [];
		foreach ($this->objects as $key => $thread) {
			if ($this->loadDeleteNote && $thread->isDeleted) {
				$threadIDs[] = $thread->threadID;
			}
		}
		
		$this->indexToObject = array_keys($this->objects);
		
		// load deletion data
		if (!empty($threadIDs)) {
			$this->logList = new BoardModificationLogList();
			$this->logList->setThreadData($threadIDs, 'trash');
			$this->logList->readObjects();
			
			foreach ($this->logList as $logEntry) {
				foreach ($this->objects as $thread) {
					if ($thread->threadID == $logEntry->objectID) {
						$thread->setLogEntry($logEntry);
					}
				}
			}
		}
	}
	
	/**
	 * Returns board modification log list.
	 * 
	 * @return	BoardModificationLogList;
	 */
	public function getLogList() {
		return $this->logList;
	}
}
