<?php
namespace wbb\data\post;
use wbb\data\board\BoardCache;
use wbb\data\thread\Thread;
use wcf\data\search\ISearchResultObject;
use wcf\system\request\LinkHandler;
use wcf\system\search\SearchResultTextParser;

/**
 * Represents a search results.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Post
 * 
 * @property	string		$topic
 * @property	integer		$boardID
 * @property	string		title
 */
class SearchResultPost extends ViewablePost implements ISearchResultObject {
	/**
	 * thread object
	 * @var	Thread
	 */
	protected $thread;
	
	/**
	 * Returns the board of this post.
	 * 
	 * @return	\wbb\data\board\Board
	 */
	public function getBoard() {
		return BoardCache::getInstance()->getBoard($this->boardID);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * Returns the thread of this post.
	 * 
	 * @return	Thread
	 */
	public function getThread() {
		if ($this->thread === null) {
			$this->thread = new Thread(null, [
				'threadID' => $this->threadID,
				'topic' => $this->topic,
				'boardID' => $this->boardID
			]);
		}
		
		return $this->thread;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getFormattedMessage() {
		return SearchResultTextParser::getInstance()->parse($this->getDecoratedObject()->getSimplifiedFormattedMessage());
	}
	
	/**
	 * @inheritDoc
	 */
	public function getSubject() {
		if ($this->subject) return $this->subject;
		
		return $this->getThread()->topic;
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getLink($query = '') {
		$parameters = [
			'application' => 'wbb',
			'object' => $this->getThread(),
			'postID' => $this->postID
		];
		
		if ($query) {
			$parameters['highlight'] = urlencode($query);
		}
		
		return LinkHandler::getInstance()->getLink('Thread', $parameters, '#post'.$this->postID);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function getTime() {
		return $this->time;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getObjectTypeName() {
		return 'com.woltlab.wbb.post';
	}
	
	/**
	 * @inheritDoc
	 */
	public function getContainerTitle() {
		return BoardCache::getInstance()->getBoard($this->boardID)->getTitle();
	}
	
	/**
	 * @inheritDoc
	 */
	public function getContainerLink() {
		return BoardCache::getInstance()->getBoard($this->boardID)->getLink();
	}
}
