<?php
namespace wbb\data\post;
use wbb\data\board\Board;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardEditor;
use wbb\data\thread\form\option\ThreadFormOptionAction;
use wbb\data\thread\form\ThreadForm;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadEditor;
use wbb\data\thread\ThreadList;
use wbb\data\thread\ViewableThread;
use wbb\system\log\modification\PostModificationLogHandler;
use wbb\system\log\modification\ThreadModificationLogHandler;
use wbb\system\option\ThreadFormOptionHandler;
use wbb\system\user\notification\object\PostUserNotificationObject;
use wcf\data\attachment\AttachmentAction;
use wcf\data\IMessageQuickReplyParametersAction;
use wcf\data\IPopoverAction;
use wcf\data\like\LikeAction;
use wcf\data\modification\log\ModificationLogList;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\poll\Poll;
use wcf\data\poll\PollAction;
use wcf\data\poll\PollEditor;
use wcf\data\user\object\watch\UserObjectWatchAction;
use wcf\data\user\UserProfileList;
use wcf\data\AbstractDatabaseObjectAction;
use wcf\data\DatabaseObject;
use wcf\data\IAttachmentMessageQuickReplyAction;
use wcf\data\IClipboardAction;
use wcf\data\IMessageInlineEditorAction;
use wcf\data\IMessageQuoteAction;
use wcf\data\TMessageQuickReplyGuestDialogAction;
use wcf\system\attachment\AttachmentHandler;
use wcf\system\bbcode\BBCodeHandler;
use wcf\system\captcha\ICaptchaHandler;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\edit\EditHistoryManager;
use wcf\system\event\EventHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\html\input\HtmlInputProcessor;
use wcf\system\like\LikeHandler;
use wcf\system\message\censorship\Censorship;
use wcf\system\message\embedded\object\MessageEmbeddedObjectManager;
use wcf\system\message\quote\MessageQuoteManager;
use wcf\system\message\QuickReplyManager;
use wcf\system\moderation\queue\ModerationQueueActivationManager;
use wcf\system\poll\PollManager;
use wcf\system\search\SearchIndexManager;
use wcf\system\user\activity\event\UserActivityEventHandler;
use wcf\system\user\activity\point\UserActivityPointHandler;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\user\object\watch\UserObjectWatchHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;
use wcf\util\ArrayUtil;
use wcf\util\MessageUtil;
use wcf\util\StringUtil;
use wcf\util\UserUtil;

/**
 * Executes post-related actions.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Post
 * 
 * @method	PostEditor[]	getObjects()
 * @method	PostEditor	getSingleObject()
 */
class PostAction extends AbstractDatabaseObjectAction implements IClipboardAction, IAttachmentMessageQuickReplyAction, IMessageInlineEditorAction, IMessageQuickReplyParametersAction, IMessageQuoteAction, IPopoverAction {
	use TMessageQuickReplyGuestDialogAction;
	
	/**
	 * @inheritDoc
	 */
	protected $className = PostEditor::class;
	
	/**
	 * @inheritDoc
	 */
	protected $allowGuestAccess = [
		'getPopover',
		'getPostPreview',
		'jumpToExtended',
		'quickReply',
		'saveFullQuote',
		'saveQuote',
	];
	
	/**
	 * current board object
	 * @var	Board
	 */
	public $board;
	
	/**
	 * @var HtmlInputProcessor
	 */
	public $htmlInputProcessor;
	
	/**
	 * current post object
	 * @var	Post
	 */
	public $post;
	
	/**
	 * list of post data
	 * @var	mixed[][]
	 */
	public $postData = [];
	
	/**
	 * current post editor object
	 * @var	PostEditor
	 */
	public $postEditor;
	
	/**
	 * list with loaded posts
	 * @var	ViewablePostList
	 */
	public $postList;
	
	/**
	 * current thread object
	 * @var	Thread
	 */
	public $thread;
	
	/**
	 * list of thread data
	 * @var	mixed[][]
	 */
	public $threadData = [];
	
	/**
	 * @inheritDoc
	 * @return	Post
	 */
	public function create() {
		if (!isset($this->parameters['data']['enableHtml'])) $this->parameters['data']['enableHtml'] = 1;
		
		// count attachments
		if (!empty($this->parameters['attachmentHandler'])) {
			$this->parameters['data']['attachments'] = count($this->parameters['attachmentHandler']);
		}
		
		if (LOG_IP_ADDRESS) {
			// add ip address
			if (!isset($this->parameters['data']['ipAddress'])) {
				$this->parameters['data']['ipAddress'] = WCF::getSession()->ipAddress;
			}
		}
		else {
			// do not track ip address
			if (isset($this->parameters['data']['ipAddress'])) {
				unset($this->parameters['data']['ipAddress']);
			}
		}
		
		// get thread
		$thread = isset($this->parameters['thread']) ? $this->parameters['thread'] : new Thread($this->parameters['data']['threadID']);
		if ($thread->isDisabled) {
			$this->parameters['data']['isDisabled'] = 1;
			if (!isset($this->parameters['data']['enableTime']) && $thread->getFirstPost() && $thread->getFirstPost()->enableTime) {
				$this->parameters['data']['enableTime'] = $thread->getFirstPost()->enableTime;
			}
		}
		if ($thread->isDeleted) {
			$this->parameters['data']['isDeleted'] = 1;
			if (!isset($this->parameters['data']['deleteTime'])) {
				// Note: We must re-use the thread's deleteTime and cannot use TIME_NOW, because otherwise
				// the post outlives the thread.
				$this->parameters['data']['deleteTime'] = $thread->deleteTime;
			}
		}
		
		if (!empty($this->parameters['htmlInputProcessor'])) {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->parameters['data']['message'] = $this->parameters['htmlInputProcessor']->getHtml();
		}
		
		/** @var Post $post */
		$post = parent::create();
		$postEditor = new PostEditor($post);
		$post->setThread($thread);
		
		// save thread form values
		if (isset($this->parameters['optionHandler'])) {
			/** @var ThreadFormOptionHandler $optionHandler */
			$optionHandler = $this->parameters['optionHandler'];
			$saveValues = $optionHandler->save();
			
			$sql = "INSERT INTO	wbb".WCF_N."_thread_form_option_value
						(postID, optionID, optionValue)
				VALUES		(?, ?, ?)";
			$statement = WCF::getDB()->prepareStatement($sql);
			foreach ($saveValues as $optionID => $optionValue) {
				if ($optionValue !== null) {
					$statement->execute([
						$post->postID,
						$optionID,
						$optionValue
					]);
				}
			}
		}
		
		// update attachments
		if (!empty($this->parameters['attachmentHandler'])) {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->parameters['attachmentHandler']->updateObjectID($post->postID);
		}
		
		// save embedded objects
		if (!empty($this->parameters['htmlInputProcessor'])) {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->parameters['htmlInputProcessor']->setObjectID($post->postID);
			if (MessageEmbeddedObjectManager::getInstance()->registerObjects($this->parameters['htmlInputProcessor'])) {
				$postEditor->update(['hasEmbeddedObjects' => 1]);
			}
		}
		
		// clear quotes
		if (isset($this->parameters['removeQuoteIDs']) && !empty($this->parameters['removeQuoteIDs'])) {
			MessageQuoteManager::getInstance()->markQuotesForRemoval($this->parameters['removeQuoteIDs']);
		}
		MessageQuoteManager::getInstance()->removeMarkedQuotes();
		
		// trigger publication
		if (!$post->isDisabled) {
			$action = new PostAction([$postEditor], 'triggerPublication', [
				'isFirstPost' => !empty($this->parameters['isFirstPost'])
			]);
			$action->executeAction();
		}
		// mark post for moderated content
		else {
			ModerationQueueActivationManager::getInstance()->addModeratedContent('com.woltlab.wbb.post', $post->postID);
		}
		
		// handle thread subscription
		if (WCF::getUser()->userID && $post->userID == WCF::getUser()->userID) {
			/** @noinspection PhpUndefinedFieldInspection */
			if (!isset($this->parameters['subscribeThread']) && WCF::getUser()->watchThreadOnReply) {
				$this->parameters['subscribeThread'] = 1;
			}
			
			if (isset($this->parameters['subscribeThread'])) {
				if ($this->parameters['subscribeThread'] && !$thread->isSubscribed()) {
					$action = new UserObjectWatchAction([], 'subscribe', [
						'data' => [
							'objectID' => $thread->threadID,
							'objectType' => 'com.woltlab.wbb.thread'
						],
						'enableNotification' => UserNotificationHandler::getInstance()->getEventSetting('com.woltlab.wbb.post', 'post') !== false ? 1 : 0
					]);
					$action->executeAction();
				}
				else if (!$this->parameters['subscribeThread'] && $thread->isSubscribed()) {
					$action = new UserObjectWatchAction([], 'unsubscribe', [
						'data' => [
							'objectID' => $thread->threadID,
							'objectType' => 'com.woltlab.wbb.thread'
						]
					]);
					$action->executeAction();
				}
			}
		}
		
		// update search index
		PostEditor::addToSearchIndex([$post], false);
		
		// return new post
		return $post;
	}
	
	/**
	 * Triggers the publication of posts.
	 */
	public function triggerPublication() {
		if (empty($this->objects)) {
			$this->readObjects();
		}
		
		$isBulkProcessing = $this->isBulkProcessing();
		$htmlInputProcessor = null;
		
		$activityEvents = [];
		foreach ($this->getObjects() as $post) {
			$threadEditor = new ThreadEditor($post->getThread());
			$boardEditor = new BoardEditor($threadEditor->getBoard());
			
			if (!$post->isFirstPost() && empty($this->parameters['isFirstPost'])) {
				if (!$isBulkProcessing) {
					// update last post
					$threadEditor->addPost($post->getDecoratedObject());
					
					// update board counters
					if (!$post->isDeleted) {
						$boardEditor->updateCounters([
							'posts' => 1
						]);
					}
				}
				
				// fire activity event
				if ($post->userID) {
					if ($isBulkProcessing) {
						$activityEvents[] = [
							'objectID' => $post->postID,
							'languageID' => $threadEditor->languageID,
							'userID' => $post->userID,
							'time' => $post->time
						];
					}
					else {
						UserActivityEventHandler::getInstance()->fireEvent(
							'com.woltlab.wbb.recentActivityEvent.post',
							$post->postID,
							$threadEditor->languageID,
							$post->userID,
							$post->time
						);
					}
					
					if (!$isBulkProcessing && $boardEditor->countUserPosts) {
						UserActivityPointHandler::getInstance()->fireEvent(
							'com.woltlab.wbb.activityPointEvent.post',
							$post->postID,
							$post->userID
						);
					}
				}
				
				// update watched objects
				if (!$isBulkProcessing && !$post->isDeleted) {
					UserObjectWatchHandler::getInstance()->updateObject(
						'com.woltlab.wbb.thread',
						$threadEditor->threadID,
						'post',
						'com.woltlab.wbb.post',
						new PostUserNotificationObject($post->getDecoratedObject())
					);
				}
			}
			
			if (!$isBulkProcessing) {
				// update board last post
				$boardEditor->setLastPost($threadEditor->getDecoratedObject());
				
				// update user post counter
				if ($boardEditor->countUserPosts && $post->userID) {
					PostEditor::updatePostCounter([$post->userID => 1]);
				}
			}
			
			// send notifications for quotes and mentions
			if (!$isBulkProcessing && !$post->isDeleted) {
				if ($htmlInputProcessor === null) $htmlInputProcessor = new HtmlInputProcessor();
				$htmlInputProcessor->processIntermediate($post->message);
				
				$usernames = MessageUtil::getQuotedUsers($htmlInputProcessor);
				if (!empty($usernames)) {
					// get user profiles
					$userList = new UserProfileList();
					$userList->getConditionBuilder()->add('user_table.username IN (?)', [$usernames]);
					if ($post->userID) {
						// ignore self-quoting
						$userList->getConditionBuilder()->add('user_table.userID <> ?', [$post->userID]);
					}
					$userList->readObjects();
					$recipientIDs = [];
					foreach ($userList as $user) {
						if ($boardEditor->checkUserPermissions($user, ['canViewBoard', 'canEnterBoard', 'canReadThread'], $post->getThread())) {
							$recipientIDs[] = $user->userID;
						}
					}
					
					// fire event
					if (!empty($recipientIDs)) {
						UserNotificationHandler::getInstance()->fireEvent(
							'quote',
							'com.woltlab.wbb.post',
							new PostUserNotificationObject($post->getDecoratedObject()),
							$recipientIDs
						);
					}
				}
				
				// check for mentions
				$userIDs = MessageUtil::getMentionedUserIDs($htmlInputProcessor);
				if (!empty($userIDs)) {
					// get user profiles
					$userList = new UserProfileList();
					$userList->getConditionBuilder()->add('user_table.userID IN (?)', [$userIDs]);
					if ($post->userID) {
						$userList->getConditionBuilder()->add('user_table.userID <> ?', [$post->userID]);
					}
					$userList->readObjects();
					$recipientIDs = [];
					foreach ($userList as $user) {
						if ($boardEditor->checkUserPermissions($user, ['canViewBoard', 'canEnterBoard', 'canReadThread'], $post->getThread())) {
							$recipientIDs[] = $user->userID;
						}
					}
					
					// fire event
					if (!empty($recipientIDs)) {
						UserNotificationHandler::getInstance()->fireEvent(
							'mention',
							'com.woltlab.wbb.post',
							new PostUserNotificationObject($post->getDecoratedObject()),
							$recipientIDs
						);
					}
				}
			}
		}
		
		if (!empty($activityEvents)) {
			UserActivityEventHandler::getInstance()->fireEvents('com.woltlab.wbb.recentActivityEvent.post', $activityEvents);
		}
		
		// reset storage
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		
		// reset cache
		BoardEditor::resetDataCache();
	}
	
	/**
	 * @inheritDoc
	 *
	 * In bulk processing mode, changing `isDisabled` does not add or remove moderation queue
	 * entries, no edit history entries are created, and no mention notifications are sent.
	 */
	public function update() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		$createModificationLogs = (!isset($this->parameters['createModificationLogs']) || $this->parameters['createModificationLogs']);
		
		// count attachments
		if (isset($this->parameters['attachmentHandler']) && $this->parameters['attachmentHandler'] !== null) {
			$this->parameters['data']['attachments'] = count($this->parameters['attachmentHandler']);
		}
		
		// set delete time
		if (isset($this->parameters['data']['isDeleted'])) {
			$this->parameters['data']['deleteTime'] = $this->parameters['data']['isDeleted'] ? TIME_NOW : 0;
		}
		
		$editReason = null;
		// prevent updating of edit reason if edit note should be hidden
		if (isset($this->parameters['data']['editReason']) && empty($this->parameters['showEditNote'])) {
			$editReason = $this->parameters['data']['editReason'];
			unset($this->parameters['data']['editReason']);
		}
		
		$posts = $this->getObjects();
		if (!empty($this->parameters['data']['message']) && empty($this->parameters['htmlInputProcessor'])) {
			$this->parameters['htmlInputProcessor'] = new HtmlInputProcessor();
			/** @noinspection PhpUndefinedMethodInspection */
			$this->parameters['htmlInputProcessor']->process($this->parameters['data']['message'], 'com.woltlab.wbb.post', current($posts)->postID);
		}
		if (!empty($this->parameters['htmlInputProcessor'])) {
			/** @noinspection PhpUndefinedMethodInspection */
			$this->parameters['data']['message'] = $this->parameters['htmlInputProcessor']->getHtml();
		}
		
		// update lastVersionTime for edit history
		if (MODULE_EDIT_HISTORY && isset($this->parameters['isEdit']) && isset($this->parameters['data']['message']) && $this->parameters['data']['message'] != current($posts)->message) {
			$this->parameters['data']['lastVersionTime'] = TIME_NOW;
		}
		
		parent::update();
		
		// handle attributes for logging
		if (!$isBulkProcessing) {
			foreach ($posts as $post) {
				// soft-delete of post
				if (!empty($this->parameters['data']['isDeleted']) && $createModificationLogs) {
					PostModificationLogHandler::getInstance()->trash($post->getDecoratedObject());
				}
				
				// moderated content
				if (isset($this->parameters['data']['isDisabled'])) {
					if ($this->parameters['data']['isDisabled']) {
						ModerationQueueActivationManager::getInstance()->addModeratedContent('com.woltlab.wbb.post', $post->postID);
					}
					else {
						ModerationQueueActivationManager::getInstance()->removeModeratedContent('com.woltlab.wbb.post', [$post->postID]);
					}
				}
				
				// edit
				if (isset($this->parameters['isEdit'])) {
					if (MODULE_EDIT_HISTORY && isset($this->parameters['data']['message']) && $this->parameters['data']['message'] != $post->message) {
						$historySavingPost = new HistorySavingPost($post->getDecoratedObject());
						EditHistoryManager::getInstance()->add(
							'com.woltlab.wbb.post',
							$post->postID,
							$post->message,
							$historySavingPost->getTime(),
							$historySavingPost->getUserID(),
							$historySavingPost->getUsername(),
							$historySavingPost->getEditReason(),
							WCF::getUser()->userID
						);
					}
					
					if ($editReason === null) {
						$reason = isset($this->parameters['data']['editReason']) ? $this->parameters['data']['editReason'] : '';
					}
					else {
						$reason = $editReason;
					}
					
					if ($createModificationLogs) {
						PostModificationLogHandler::getInstance()->edit($post->getDecoratedObject(), $reason);
					}
				}
				
				// update thread attachment counter
				if (isset($this->parameters['data']['attachments']) && $this->parameters['data']['attachments'] != $post->attachments) {
					$threadEditor = new ThreadEditor(new Thread(null, ['threadID' => $post->threadID]));
					$threadEditor->updateCounters(['attachments' => $this->parameters['data']['attachments'] - $post->attachments]);
				}
				
				// handle new mentions / quotes
				if (!empty($this->parameters['htmlInputProcessor']) && !$post->isDisabled && !$post->isDeleted) {
					$quotedUsernames = MessageUtil::getQuotedUsers($this->parameters['htmlInputProcessor']);
					$mentionedUserIDs = MessageUtil::getMentionedUserIDs($this->parameters['htmlInputProcessor']);
					
					if (!empty($quotedUsernames) || !empty($mentionedUserIDs)) {
						// process old message
						$htmlInputProcessor = new HtmlInputProcessor();
						$htmlInputProcessor->processIntermediate($post->message);
						
						if (!empty($quotedUsernames)) {
							// find users that have not been quoted in this post before
							$existingUsernames = array_map('mb_strtolower', MessageUtil::getQuotedUsers($htmlInputProcessor));
							$quotedUsernames = array_unique(array_filter($quotedUsernames, function($username) use ($existingUsernames) {
								return !in_array(mb_strtolower($username), $existingUsernames);
							}));
							
							if (!empty($quotedUsernames)) {
								// get user profiles
								$userList = new UserProfileList();
								$userList->getConditionBuilder()->add('user_table.username IN (?)', [$quotedUsernames]);
								if ($post->userID) {
									$userList->getConditionBuilder()->add('user_table.userID <> ?', [$post->userID]);
								}
								$userList->readObjects();
								$recipientIDs = [];
								$board = BoardCache::getInstance()->getBoard($post->getThread()->boardID);
								foreach ($userList as $user) {
									if ($board->checkUserPermissions($user, ['canViewBoard', 'canEnterBoard', 'canReadThread'], $post->getThread())) {
										$recipientIDs[] = $user->userID;
									}
								}
								
								// fire event
								if (!empty($recipientIDs)) {
									UserNotificationHandler::getInstance()->fireEvent(
										'quote',
										'com.woltlab.wbb.post',
										new PostUserNotificationObject($post->getDecoratedObject()),
										$recipientIDs
									);
								}
							}
						}
						
						if (!empty($mentionedUserIDs)) {
							// find users that have not been mentioned in this post before
							$existingUserIDs = MessageUtil::getMentionedUserIDs($htmlInputProcessor);
							$mentionedUserIDs = array_unique(array_filter($mentionedUserIDs, function($userID) use ($existingUserIDs) {
								return !in_array($userID, $existingUserIDs);
							}));
							
							if (!empty($mentionedUserIDs)) {
								// get user profiles
								$userList = new UserProfileList();
								$userList->getConditionBuilder()->add('user_table.userID IN (?)', [$mentionedUserIDs]);
								if ($post->userID) {
									$userList->getConditionBuilder()->add('user_table.userID <> ?', [$post->userID]);
								}
								
								$userList->readObjects();
								$recipientIDs = [];
								$board = BoardCache::getInstance()->getBoard($post->getThread()->boardID);
								foreach ($userList as $user) {
									if ($board->checkUserPermissions($user, ['canViewBoard', 'canEnterBoard', 'canReadThread'], $post->getThread())) {
										$recipientIDs[] = $user->userID;
									}
								}
								
								// fire event
								if (!empty($recipientIDs)) {
									UserNotificationHandler::getInstance()->fireEvent(
										'mention',
										'com.woltlab.wbb.post',
										new PostUserNotificationObject($post->getDecoratedObject()),
										$recipientIDs
									);
								}
							}
						}
					}
				}	
			}
		}
		
		if (!$isBulkProcessing && isset($this->parameters['data']) && (isset($this->parameters['data']['subject']) || isset($this->parameters['data']['message']))) {
			// update search index
			PostEditor::addPostIDsToSearchIndex($this->getObjectIDs());
			
			// update embedded objects
			if (!empty($this->parameters['htmlInputProcessor'])) {
				foreach ($this->getObjects() as $object) {
					if ($object->hasEmbeddedObjects != MessageEmbeddedObjectManager::getInstance()->registerObjects($this->parameters['htmlInputProcessor'])) {
						$object->update([
							'hasEmbeddedObjects' => $object->hasEmbeddedObjects ? 0 : 1
						]);
					}
				}
			}
		}
	}
	
	/**
	 * Validates the `prepareEnable` action.
	 * 
	 * @throws	PermissionDeniedException
	 * @throws	UserInputException
	 */
	public function validatePrepareEnable() {
		$this->postEditor = $this->getSingleObject();
		if (!$this->postEditor->isDisabled || $this->postEditor->isDeleted) {
			throw new UserInputException('objectIDs');
		}
		else if (!$this->postEditor->canEnable()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * Returns the dialog to set the enable time for a post.
	 * 
	 * @return	array
	 */
	public function prepareEnable() {
		WCF::getTPL()->assign([
			'enableTime' => ($this->postEditor->enableTime ? date('r', $this->postEditor->enableTime) : ''),
			'post' => $this->postEditor
		]);
		
		return [
			'objectID' => $this->postEditor->postID,
			'template' => WCF::getTPL()->fetch('postEnable', 'wbb')
		];
	}
	
	/**
	 * Validates if user can enable posts.
	 */
	public function validateEnable() {
		$this->readBoolean('updateTime', true);
		
		$this->prepareObjects();
		
		// validate permissions
		foreach ($this->getObjects() as $post) {
			if (!$post->isDisabled || $post->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$post->canEnable()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Validates the `setEnableTime` action.
	 * 
	 * @throws	UserInputException
	 */
	public function validateSetEnableTime() {
		$this->postEditor = $this->getSingleObject();
		
		$this->readString('enableTime', true);
		if (!empty($this->parameters['enableTime'])) {
			$this->parameters['enableTimeObj'] = \DateTime::createFromFormat('Y-m-d\TH:i:sP', $this->parameters['enableTime']);
			/** @noinspection PhpUndefinedMethodInspection */
			if (!$this->parameters['enableTimeObj'] || $this->parameters['enableTimeObj']->getTimestamp() < TIME_NOW) {
				throw new UserInputException('enableTime', 'invalid');
			}
		}
	}
	
	/**
	 * Sets the enable time of a post and returns the text of its enable note.
	 * 
	 * @return	string[]
	 */
	public function setEnableTime() {
		/** @noinspection PhpUndefinedMethodInspection */
		$this->postEditor->update([
			'enableTime' => ($this->parameters['enableTime'] ? $this->parameters['enableTimeObj']->getTimestamp() : 0)
		]);
		
		$post = new Post($this->postEditor->postID);
		
		return [
			'enableNote' => ($this->parameters['enableTime'] ? WCF::getLanguage()->getDynamicVariable('wbb.post.delayedPublication', ['post' => $post]) : '')
		];
	}
	
	/**
	 * Enables given posts.
	 */
	public function enable() {
		$this->prepareObjects();
		
		$isBulkProcessing = $this->isBulkProcessing();
		$ignorePostModificationLogs = (isset($this->parameters['ignorePostModificationLogs']) && $this->parameters['ignorePostModificationLogs']);
		
		$data = ['isDisabled' => 0];
		if (!empty($this->parameters['data']['updateTime'])) {
			$data['time'] = TIME_NOW;
		}
		
		(new PostAction($this->getObjects(), 'update', [
			'data' => $data,
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		$enableThreads = $postIDs = [];
		foreach ($this->getObjects() as $post) {
			if (empty($this->parameters['ignoreThreads']) && $post->getThread()->isDisabled) {
				$enableThreads[] = $post->threadID;
			}
			
			$postIDs[] = $post->postID;
			
			if (!$isBulkProcessing) {
				$this->addPostData($post->getDecoratedObject(), 'isDisabled', 0);
				
				if (!$ignorePostModificationLogs) {
					PostModificationLogHandler::getInstance()->enable($post->getDecoratedObject());
				}
			}
		}
		
		(new PostAction($this->objects, 'triggerPublication', [
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		$this->removeModeratedContent($postIDs);
		
		if (!empty($enableThreads)) {
			(new ThreadAction(array_unique($enableThreads), 'enable', [
				'data' => [
					'updateTime' => !empty($this->parameters['updateTime'])
				],
				'ignorePosts' => true, 
				'ignoreThreadModificationLogs' => $ignorePostModificationLogs,
				'isBulkProcessing' => $isBulkProcessing
			]))->executeAction();
			
			if (!$isBulkProcessing) {
				foreach ($enableThreads as $threadID) {
					$this->addThreadData($threadID, 'isDisabled', 0);
					$this->addThreadData($threadID, 'ignoreDisabledPosts', 1);
				}
			}
		}
		
		if (!$isBulkProcessing) {
			$this->unmarkItems();
		}
		
		// reset unread threads
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
		
		return $this->getPostData();
	}
	
	/**
	 * Validates if user can disable posts.
	 */
	public function validateDisable() {
		$this->prepareObjects();
		
		// validate permissions
		foreach ($this->getObjects() as $post) {
			if ($post->isDisabled || $post->isDeleted) {
				throw new UserInputException('objectIDs');
			}
			
			if (!$post->canEnable()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Disables given posts.
	 */
	public function disable() {
		$this->prepareObjects();
		
		$isBulkProcessing = $this->isBulkProcessing();
		$ignorePostModificationLogs = (isset($this->parameters['ignorePostModificationLogs']) && $this->parameters['ignorePostModificationLogs']);
		
		(new PostAction($this->getObjects(), 'update', [
			'data' => ['isDisabled' => 1],
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		$checkThreads = $postIDs = $boardStats = $boardIDs = $userPosts = $userToItems = $postIDsToBoardIDs = [];
		foreach ($this->getObjects() as $post) {
			if (empty($this->parameters['ignoreThreads']) && !$post->getThread()->isDisabled) {
				$checkThreads[] = $post->threadID;
			}
			
			$postIDs[] = $post->postID;
			
			// add moderation queue entries
			if ($isBulkProcessing) {
				$postIDsToBoardIDs[$post->postID] = $post->getThread()->boardID;
			}
			else {
				ModerationQueueActivationManager::getInstance()->addModeratedContent('com.woltlab.wbb.post', $post->postID);
			}
			
			if (!$isBulkProcessing) {
				$boardIDs[] = $post->getThread()->boardID;
				
				$this->addPostData($post->getDecoratedObject(), 'isDisabled', 1);
				
				if (!isset($boardStats[$post->getThread()->boardID])) {
					$boardStats[$post->getThread()->boardID] = [
						'posts' => 0
					];
				}
				$boardStats[$post->getThread()->boardID]['posts']--;
				
				if ($post->userID && $post->getThread()->getBoard()->countUserPosts) {
					if (!isset($userPosts[$post->userID])) {
						$userPosts[$post->userID] = 0;
					}
					
					$userPosts[$post->userID]--;
					
					if (!$post->isFirstPost()) {
						if (!isset($userToItems[$post->userID])) {
							$userToItems[$post->userID] = 0;
						}
						
						$userToItems[$post->userID]++;
					}
				}
				
				if ($post->postID == $post->getThread()->lastPostID) {
					$threadEditor = new ThreadEditor($post->getThread());
					$threadEditor->updateLastPost();
				}
				
				if (!$ignorePostModificationLogs) {
					PostModificationLogHandler::getInstance()->disable($post->getDecoratedObject());
				}
			}
		}
		
		// add moderation queue entries at once
		if (!empty($postIDsToBoardIDs)) {
			ModerationQueueActivationManager::getInstance()->addModeratedContents(
				'com.woltlab.wbb.post',
				$postIDs,
				$postIDsToBoardIDs
			);
		}
		
		// update counters
		if (!empty($boardStats)) {
			foreach ($boardStats as $boardID => $stats) {
				$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
				$boardEditor->updateCounters($stats);
			}
		}
		
		// remove activity points and activity events
		if (!$isBulkProcessing) {
			UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $userToItems);
		}
		UserActivityEventHandler::getInstance()->removeEvents('com.woltlab.wbb.recentActivityEvent.post', $postIDs);
		
		// delete notifications
		UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.post', $postIDs);
		UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.likeablePost.notification', $postIDs);
		
		$modificationLogList = new ModificationLogList();
		$modificationLogList->getConditionBuilder()->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.post')]);
		$modificationLogList->getConditionBuilder()->add('modification_log.objectID IN (?)', [$postIDs]);
		$modificationLogList->readObjects();
		
		if (count($modificationLogList)) {
			UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.moderation.post', $modificationLogList->getObjectIDs());
		}
		
		if (!empty($checkThreads)) {
			$checkThreads = array_unique($checkThreads);
			
			// rebuild data
			if (!$isBulkProcessing) {
				ThreadEditor::rebuildThreadData($checkThreads);
			}
			
			// disable threads if necessary
			$conditionBuilder = new PreparedStatementConditionBuilder();
			$conditionBuilder->add('threadID IN (?)', [$checkThreads]);
			$conditionBuilder->add('isDisabled = ?', [0]);
			$conditionBuilder->add('isDeleted = ?', [0]);
			$sql = "SELECT	threadID
				FROM	wbb".WCF_N."_post
				".$conditionBuilder;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			$ignoreThreads = $statement->fetchAll(\PDO::FETCH_COLUMN);
			
			$disableThreads = array_diff($checkThreads, $ignoreThreads);
			if (!empty($disableThreads)) {
				$threadAction = new ThreadAction($disableThreads, 'disable', [
					'ignorePosts' => true,
					'isBulkProcessing' => $isBulkProcessing
				]);
				$threadAction->executeAction();
			}
		}
		
		if (!$isBulkProcessing) {
			$this->updateLastPost($boardIDs);
			
			// update users' posts
			if (!empty($userPosts)) {
				PostEditor::updatePostCounter($userPosts);
			}
			
			$this->unmarkItems();
		}
		
		// reset unread threads
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
		UserStorageHandler::getInstance()->resetAll('wbbWatchedThreads');
		
		return $this->getPostData();
	}
	
	/**
	 * Validates if user can move posts into trash bin.
	 */
	public function validateTrash() {
		$this->prepareObjects();
		$this->readString('reason', true);
		
		// when trashing posts using the clipboard, the reason is stored
		// in the data array
		if (isset($this->parameters['data']) && isset($this->parameters['data']['reason'])) {
			$this->parameters['reason'] = $this->parameters['data']['reason'];
		}
		
		// validate permissions
		foreach ($this->getObjects() as $post) {
			if (!$post->canDelete()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function validateDelete() {
		$this->prepareObjects();
		
		// validate permissions
		foreach ($this->getObjects() as $post) {
			if (!$post->canDeleteCompletely()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Alias for validateDelete()
	 */
	public function validateDeleteCompletely() {
		$this->validateDelete();
	}
	
	/**
	 * Validates if user can restore posts.
	 */
	public function validateRestore() {
		$this->prepareObjects();
		
		// validate permissions
		foreach ($this->getObjects() as $post) {
			if (!$post->canRestore()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Prepares deletion of posts.
	 */
	protected function prepareObjects() {
		if (empty($this->objects)) {
			$this->readObjects();
			
			if (empty($this->objects)) {
				throw new UserInputException('objectIDs');
			}
		}
		
		// load threads
		$threadIDs = [];
		foreach ($this->getObjects() as $post) {
			$threadIDs[] = $post->threadID;
		}
		
		$threadList = new ThreadList();
		$threadList->setObjectIDs($threadIDs);
		$threadList->readObjects();
		
		// assign threads
		foreach ($threadList as $thread) {
			foreach ($this->getObjects() as $post) {
				if ($thread->threadID == $post->threadID) {
					$post->setThread($thread);
				}
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function delete() {
		$this->prepareObjects();
		
		$isBulkProcessing = $this->isBulkProcessing();
		$ignoreThreads = (isset($this->parameters['ignoreThreads']) && $this->parameters['ignoreThreads']);
		$ignorePostModificationLogs = (isset($this->parameters['ignorePostModificationLogs']) && $this->parameters['ignorePostModificationLogs']);
		$createModificationLogs = (!isset($this->parameters['createModificationLogs']) || $this->parameters['createModificationLogs']);
		
		// if the posts are deleted because their threads are deleted, there is no need
		// to trash them first
		// IMPORTANT: when bulk processing posts (instead of threads), posts have to be
		// trashed first to properly update the state of the relevant threads
		$noTrashing = isset($this->parameters['noTrashing']) && $this->parameters['noTrashing'];
		
		$threadIDs = $trashPosts = [];
		foreach ($this->getObjects() as $post) {
			if (!isset($threadIDs[$post->threadID])) {
				$threadIDs[$post->threadID] = $post->threadID;
			}
			
			if (!$post->isDeleted) {
				$trashPosts[$post->postID] = $post;
			}
		}
		
		if (!$noTrashing && !empty($trashPosts)) {
			$postAction = new PostAction($trashPosts, 'trash', [
				'ignoreThreads' => $ignoreThreads,
				'isBulkProcessing' => $isBulkProcessing,
				'isDeletionPreparation' => true
			]);
			$postAction->executeAction();
		}
		
		$postStats = $this->getDetailedPostStats($threadIDs);
		
		parent::delete();
		
		// update threads
		$attachmentPostIDs = $boardIDs = $pollIDs = $postIDs = $userPosts = $userToItems = $threadEditors = $deleteThreads = [];
		foreach ($this->getObjects() as $post) {
			$threadEditors[$post->threadID] = new ThreadEditor($post->getThread());
			$postIDs[] = $post->postID;
			
			if (!$isBulkProcessing) {
				if ($post->userID && !$post->isDisabled && $post->getThread()->getBoard()->countUserPosts) {
					if (!isset($userPosts[$post->userID])) {
						$userPosts[$post->userID] = 0;
					}
					
					$userPosts[$post->userID]--;
					
					if (!$post->isFirstPost()) {
						if (!isset($userToItems[$post->userID])) {
							$userToItems[$post->userID] = 0;
						}
						
						$userToItems[$post->userID]++;
					}
				}
			}
			
			$postStats[$post->threadID]['posts']--;
			if ($post->isDisabled) {
				$postStats[$post->threadID]['disabledPosts']--;
			}
			
			// thread has no posts anymore -> delete it
			if (!$ignoreThreads && !$postStats[$post->threadID]['posts']) {
				unset($threadIDs[$post->threadID]);
				
				$deleteThreads[] = $threadEditors[$post->threadID];
				
				if (!$isBulkProcessing) {
					$boardIDs[] = $threadEditors[$post->threadID]->boardID;
					
					if (!$ignorePostModificationLogs) {
						ThreadModificationLogHandler::getInstance()->delete($post->getThread());
					}
					
					$this->addThreadData($post->threadID, 'deleted', $threadEditors[$post->threadID]->getBoard()->getLink());
				}
				
				continue;
			}
			
			if (!$isBulkProcessing && $createModificationLogs) {
				PostModificationLogHandler::getInstance()->delete($post->getDecoratedObject());
			}
			
			if ($post->pollID) {
				$pollIDs[] = $post->pollID;
			}
			
			if ($post->attachments) {
				$attachmentPostIDs[] = $post->postID;
			}
			
			$this->addPostData($post->getDecoratedObject(), 'deleted', 1);
		}
		
		if (!$ignoreThreads) {
			// check if the left posts are all disabled and if the thread
			// is also disabled
			$disableThreads = [];
			foreach ($threadIDs as $threadID) {
				if ($postStats[$threadID]['posts'] == $postStats[$threadID]['disabledPosts'] && !$threadEditors[$threadID]->isDisabled) {
					$disableThreads[] = $threadEditors[$threadID];
					
					if (!$isBulkProcessing) {
						$this->addThreadData($threadID, 'isDisabled', 1);
					}
					
					// after disabling the thread, the thread data
					// will be be rebuild in that method, so no need
					// to rebuild it here, too
					unset($threadIDs[$threadID]);
				}
			}
			if (!empty($disableThreads)) {
				(new ThreadAction($disableThreads, 'disable', [
					'ignorePosts' => true,
					'isBulkProcessing' => $isBulkProcessing
				]))->executeAction();
			}
			
			if (!empty($deleteThreads)) {
				(new ThreadAction($deleteThreads, 'delete', [
					'ignorePosts' => true,
					'isBulkProcessing' => $isBulkProcessing
				]))->executeAction();
			}
			
			if (!$ignoreThreads && !$isBulkProcessing && !empty($threadIDs)) {
				ThreadEditor::rebuildThreadData($threadIDs);
			}
		}
		
		if (!empty($postIDs)) {
			// update search index
			SearchIndexManager::getInstance()->delete('com.woltlab.wbb.post', $postIDs);
			
			// remove edit history
			EditHistoryManager::getInstance()->delete('com.woltlab.wbb.post', $postIDs);
			
			// update embedded objects
			MessageEmbeddedObjectManager::getInstance()->removeObjects('com.woltlab.wbb.post', $postIDs);
			
			// unmark posts
			if (!$isBulkProcessing && isset($this->parameters['unmarkItems'])) {
				$this->unmarkItems($postIDs);
			}
			
			// delete likes
			LikeHandler::getInstance()->removeLikes('com.woltlab.wbb.likeablePost', $postIDs);
			
			// remove activity points
			if (!$isBulkProcessing) {
				UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $userToItems);
			}
			
			// remove notifications
			UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.post', $postIDs);
			
			$modificationLogList = new ModificationLogList();
			$modificationLogList->getConditionBuilder()->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.post')]);
			$modificationLogList->getConditionBuilder()->add('modification_log.objectID IN (?)', [$postIDs]);
			$modificationLogList->readObjects();
			
			if (count($modificationLogList)) {
				UserNotificationHandler::getInstance()->removeNotifications('com.woltlab.wbb.moderation.post', $modificationLogList->getObjectIDs());
			}
			
			// delete the log entries except for deleting the post
			if (!$ignorePostModificationLogs) {
				PostModificationLogHandler::getInstance()->deleteLogs($postIDs, ['delete']);
			}
		}
		
		// update last post
		if (!$ignoreThreads && !$isBulkProcessing) {
			$this->updateLastPost($boardIDs);
		}
		
		// remove polls
		if (!empty($pollIDs)) {
			PollManager::getInstance()->removePolls($pollIDs);
		}
		
		// remove attachments
		if (!empty($attachmentPostIDs)) {
			AttachmentHandler::removeAttachments('com.woltlab.wbb.post', $attachmentPostIDs);
		}
		
		// update users' posts
		if (!$isBulkProcessing && !empty($userPosts)) {
			PostEditor::updatePostCounter($userPosts);
		}
		
		return $this->getPostData();
	}
	
	/**
	 * Alias for delete()
	 */
	public function deleteCompletely() {
		return $this->delete();
	}
	
	/**
	 * Moves given posts into trash bin.
	 */
	public function trash() {
		$isBulkProcessing = $this->isBulkProcessing();
		$ignoreThreads = (isset($this->parameters['ignoreThreads']) && $this->parameters['ignoreThreads']);
		$ignorePostModificationLogs = (isset($this->parameters['ignorePostModificationLogs']) && $this->parameters['ignorePostModificationLogs']);
		
		$threadIDs = [];
		foreach ($this->getObjects() as $post) {
			if (!isset($threadIDs[$post->threadID])) {
				$threadIDs[$post->threadID] = $post->threadID;
			}
		}
		
		if (!$ignoreThreads) {
			$this->prepareObjects();
		}
		
		$postStats = [];
		if (!$ignoreThreads) {
			$postStats = $this->getDetailedPostStats($threadIDs);
		}
		
		(new PostAction($this->getObjects(), 'update', [
			'data' => [
				'deleteTime' => TIME_NOW,
				'isDeleted' => 1
			],
			'isBulkProcessing' => $isBulkProcessing,
			'createModificationLogs' => false
		]))->executeAction();
		
		// update threads
		$boardPostUpdates = $boardIDs = $postIDs = $trashThreads = $disableThreads = $disabledPostIDs = [];
		foreach ($this->getObjects() as $post) {
			// ignore posts which are already in trash bin
			if ($post->isDeleted) {
				continue;
			}
			
			$postIDs[] = $post->postID;
			if ($post->isDisabled) {
				$disabledPostIDs[] = $post->postID;
			}
			
			if (!$ignoreThreads) {
				$threadEditor = new ThreadEditor($post->getThread());
				if (!isset($boardPostUpdates[$threadEditor->boardID])) {
					$boardPostUpdates[$threadEditor->boardID] = 0;
				}
				if (!$post->isDisabled) {
					$boardPostUpdates[$threadEditor->boardID]--;
					$postStats[$post->threadID]['posts']--;
					$postStats[$post->threadID]['visiblePosts']--;
				}
				else {
					$postStats[$post->threadID]['disabledPosts']--;
				}
				
				// no more visible posts
				if (!$postStats[$post->threadID]['visiblePosts']) {
					if ($postStats[$post->threadID]['disabledPosts']) {
						// thread has disabled posts thus disable
						// thread if it isn't already disabled
						if (!$threadEditor->isDisabled) {
							$disableThreads[] = $threadEditor;
							
							if (!$isBulkProcessing) {
								$this->addThreadData($threadEditor->threadID, 'isDisabled', 1);
							}
						}
					}
					else {
						// thread only has deleted posts anymore
						// thus trash it
						$trashThreads[] = $threadEditor;
						
						if (!$isBulkProcessing) {
							$this->addThreadData($threadEditor->threadID, 'isDeleted', 1);
						}
					}
					$boardIDs[] = $threadEditor->boardID;
				}
			}
			
			if (!$isBulkProcessing) {
				if (!$ignorePostModificationLogs) {
					PostModificationLogHandler::getInstance()->trash(
						$post->getDecoratedObject(),
						!empty($this->parameters['reason']) ? $this->parameters['reason'] : ''
					);
				}
				
				$this->addPostData($post->getDecoratedObject(), 'isDeleted', 1);
				
				if (!$ignoreThreads) {
					if ($post->postID == $threadEditor->lastPostID) {
						$threadEditor->updateLastPost();
						$boardIDs[] = $threadEditor->boardID;
					}
				}
			}
		}
		
		if (!$isBulkProcessing && !empty($threadIDs)) {
			ThreadEditor::rebuildThreadData($threadIDs);
		}
		
		if (!empty($disabledPostIDs)) {
			$this->removeModeratedContent($disabledPostIDs);
		}
		
		// disable threads with no more visible posts but which still have
		// disabled posts
		if (!$ignoreThreads && !empty($disableThreads)) {
			$threadAction = new ThreadAction($disableThreads, 'disable', [
				'ignorePosts' => true,
				'isBulkProcessing' => $isBulkProcessing
			]);
			$threadAction->executeAction();
		}
		
		// trash threads with no more visible posts
		if (!$ignoreThreads && !empty($trashThreads)) {
			$threadAction = new ThreadAction($trashThreads, 'trash', [
				'data' => [
					'reason' => empty($this->parameters['reason']) ? '' : $this->parameters['reason']
				],
				'ignorePosts' => true,
				'isBulkProcessing' => $isBulkProcessing
			]);
			$threadAction->executeAction();
		}
		
		// update board stats
		if (!$isBulkProcessing) {
			$updatedBoardStats = false;
			foreach ($boardPostUpdates as $boardID => $postUpdate) {
				if ($postUpdate) {
					$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
					$boardEditor->updateCounters([
						'posts' => $postUpdate
					]);
					
					$updatedBoardStats = true;
				}
			}
			
			if ($updatedBoardStats) {
				BoardEditor::resetDataCache();
			}
			
			$this->unmarkItems($postIDs);
			
			// update last posts
			$this->updateLastPost($boardIDs);
		}
		
		// reset unread threads
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
		
		return $this->getPostData();
	}
	
	/**
	 * Restores given posts.
	 */
	public function restore() {
		$this->prepareObjects();
		
		$isBulkProcessing = $this->isBulkProcessing();
		$ignorePostModificationLogs = (isset($this->parameters['ignorePostModificationLogs']) && $this->parameters['ignorePostModificationLogs']);
		
		$threadIDs = [];
		foreach ($this->getObjects() as $post) {
			if (!isset($threadIDs[$post->threadID])) {
				$threadIDs[$post->threadID] = $post->threadID;
			}
		}
		
		(new PostAction($this->getObjects(), 'update', [
			'data' => [
				'deleteTime' => 0,
				'isDeleted' => 0
			],
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		// update threads
		$boardIDs = $boardStats = $postIDs = [];
		$softRestoreThreads = $softEnableThreads = [];
		foreach ($this->getObjects() as $post) {
			// ignore posts already restored
			if (!$post->isDeleted) {
				continue;
			}
			
			$parameters = [];
			$threadEditor = new ThreadEditor($post->getThread());
			$postIDs[] = $post->postID;
			
			if (!$isBulkProcessing) {
				if (!$post->isDisabled) {
					if (!isset($boardStats[$threadEditor->boardID])) {
						$boardStats[$threadEditor->boardID] = [
							'posts' => 0,
							'threads' => []
						];
					}
					$boardStats[$threadEditor->boardID]['posts']++;
				}
			}
			
			// restore thread if at least one post is accessible
			if ($threadEditor->isDeleted && !isset($boardStats[$threadEditor->boardID]['threads'][$threadEditor->threadID])) {
				if ($isBulkProcessing) {
					$softRestoreThreads[] = $threadEditor;
				}
				else {
					$parameters['deleteTime'] = 0;
					$parameters['isDeleted'] = 0;
					$boardIDs[] = $threadEditor->boardID;
					
					ThreadModificationLogHandler::getInstance()->restore($post->getThread());
					
					$this->addThreadData($threadEditor->threadID, 'isDeleted', 0);
					$this->addThreadData($threadEditor->threadID, 'ignoreDeletedPosts', 1);
					
					// mark that this thread will be restored already
					if (!isset($boardStats[$threadEditor->boardID])) {
						$boardStats[$threadEditor->boardID] = [
							'posts' => 0,
							'threads' => []
						];
					}
					$boardStats[$threadEditor->boardID]['threads'][$threadEditor->threadID] = $threadEditor->threadID;
				}
			}
			else if (!$threadEditor->isDeleted && $threadEditor->isDisabled && !$post->isDisabled && !isset($boardStats[$threadEditor->boardID]['threads'][$threadEditor->threadID])) {
				if ($isBulkProcessing) {
					$softEnableThreads[] = $threadEditor;
				}
				else {
					$parameters['isDisabled'] = 0;
					$boardIDs[] = $threadEditor->boardID;
					
					ThreadModificationLogHandler::getInstance()->enable($post->getThread());
					
					$this->addThreadData($threadEditor->threadID, 'isDisabled', 0);
					
					// mark that this thread will be enabled already
					if (!isset($boardStats[$threadEditor->boardID])) {
						$boardStats[$threadEditor->boardID] = [
							'posts' => 0,
							'threads' => []
						];
					}
					$boardStats[$threadEditor->boardID]['threads'][$threadEditor->threadID] = $threadEditor->threadID;
				}
			}
			
			// update thread
			if (!empty($parameters)) {
				$threadEditor->update($parameters);
			}
			
			if (!$isBulkProcessing) {
				if (!$ignorePostModificationLogs) {
					PostModificationLogHandler::getInstance()->restore($post->getDecoratedObject());
				}
				
				if (isset($this->parameters['unmarkItems'])) {
					$postIDs[] = $post->postID;
				}
				
				$this->addPostData($post->getDecoratedObject(), 'isDeleted', 0);
				
				if ($post->time >= $threadEditor->lastPostTime) {
					$threadEditor->updateLastPost();
					$boardIDs[] = $threadEditor->boardID;
				}
			}
		}
		
		if (!empty($softRestoreThreads)) {
			(new ThreadAction($softRestoreThreads, 'update', [
				'data' => [
					'deleteTime' => 0,
					'isDeleted' => 0
				],
				'isBulkProcessing' => $isBulkProcessing
			]))->executeAction();
		}
		
		if (!empty($softEnableThreads)) {
			(new ThreadAction($softEnableThreads, 'update', [
				'data' => [
					'isDisabled' => 0
				],
				'isBulkProcessing' => $isBulkProcessing
			]))->executeAction();
		}
		
		if (!$isBulkProcessing) {
			if (!empty($threadIDs)) {
				ThreadEditor::rebuildThreadData(array_unique($threadIDs));
			}
			
			if (!empty($postIDs)) {
				$this->unmarkItems($postIDs);
			}
			
			// update board counters
			$updatedBoardStats = false;
			if (!$isBulkProcessing) {
				foreach ($boardStats as $boardID => $update) {
					$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
					$boardEditor->updateCounters([
						'posts' => $update['posts'],
						'threads' => count($update['threads'])
					]);
					
					$updatedBoardStats = true;
				}
			}
			
			if ($updatedBoardStats) {
				BoardEditor::resetDataCache();
			}
			
			// update last post
			$this->updateLastPost($boardIDs);
		}
		
		// reset unread threads
		UserStorageHandler::getInstance()->resetAll('wbbUnreadThreads');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->resetAll('wbbUnreadWatchedThreads');
		UserStorageHandler::getInstance()->resetAll('wbbWatchedThreads');
		
		return $this->getPostData();
	}
	
	/**
	 * Validates parameters to close posts.
	 */
	public function validateClose() {
		$this->prepareObjects();
		
		foreach ($this->getObjects() as $post) {
			if (!$post->getThread()->getBoard()->getModeratorPermission('canClosePost')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Closes posts.
	 * 
	 * @return	mixed[][]
	 */
	public function close() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		(new PostAction($this->getObjects(), 'update', [
			'data' => ['isClosed' => 1],
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			foreach ($this->getObjects() as $post) {
				$this->addPostData($post->getDecoratedObject(), 'isClosed', 1);
				
				PostModificationLogHandler::getInstance()->close($post->getDecoratedObject());
			}
		}
		
		return $this->getPostData();
	}
	
	/**
	 * Validates parameters to open posts.
	 */
	public function validateOpen() {
		$this->validateClose();
	}
	
	/**
	 * Opens posts.
	 * 
	 * @return	mixed[][]
	 */
	public function open() {
		$isBulkProcessing = $this->isBulkProcessing();
		
		(new PostAction($this->getObjects(), 'update', [
			'data' => ['isClosed' => 0],
			'isBulkProcessing' => $isBulkProcessing
		]))->executeAction();
		
		if (!$isBulkProcessing) {
			foreach ($this->getObjects() as $post) {
				$this->addPostData($post->getDecoratedObject(), 'isClosed', 0);
				
				PostModificationLogHandler::getInstance()->open($post->getDecoratedObject());
			}
		}
		
		return $this->getPostData();
	}
	
	/**
	 * Validates parameters to prepare posts for merging.
	 */
	public function validatePrepareMerge() {
		// read posts
		$postList = new SimplifiedViewablePostList();
		$postList->setObjectIDs($this->objectIDs);
		$postList->readObjects();
		$this->objects = $postList->getObjects();
		
		if (count($this->objects) < 2) {
			throw new UserInputException('objectIDs');
		}
		
		// read threads
		$threadIDs = [];
		foreach ($this->getObjects() as $post) {
			$threadIDs[] = $post->threadID;
		}
		
		$threadList = new ThreadList();
		$threadList->setObjectIDs($threadIDs);
		$threadList->readObjects();
		$threads = $threadList->getObjects();
		foreach ($this->getObjects() as $post) {
			$post->setThread($threads[$post->threadID]);
			
			if (!$post->getThread()->canRead() || !$post->getThread()->getBoard()->getModeratorPermission('canMergePost')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Prepares posts for merging.
	 * 
	 * @return	string[]
	 */
	public function prepareMerge() {
		WCF::getTPL()->assign([
			'posts' => $this->objects
		]);
		
		return [
			'template' => WCF::getTPL()->fetch('postMerge', 'wbb')
		];
	}
	
	/**
	 * Validates parameters to merge posts.
	 */
	public function validateMerge() {
		$this->validatePrepareMerge();
		
		if (!isset($this->parameters['postOrder']) || !is_array($this->parameters['postOrder']) || count($this->parameters['postOrder']) != count($this->objects)) {
			throw new UserInputException('postOrder');
		}
		
		foreach ($this->parameters['postOrder'] as $index => $postID) {
			if (!isset($this->objects[$postID])) {
				throw new UserInputException('postOrder');
			}
			
			if ($index == 0) {
				$this->postEditor = new PostEditor($this->objects[$postID]->getDecoratedObject());
				unset($this->parameters['postOrder'][$index]);
				unset($this->objects[$postID]);
			}
		}
		
		if ($this->postEditor === null) {
			throw new UserInputException('postOrder');
		}
	}
	
	/**
	 * Merges posts.
	 * 
	 * @return	string[]
	 */
	public function merge() {
		$attachmentPostIDs = [];
		$attachmentsCount = $this->postEditor->attachments;
		$message = $this->postEditor->message;
		$pollID = $this->postEditor->pollID;
		$usernames = [];
		WCF::getDB()->beginTransaction();
		foreach ($this->parameters['postOrder'] as $postID) {
			/** @var Post $post */
			// bind &$post as reference, because it might be reloaded when moving polls
			foreach ($this->objects as &$post) {
				if ($post->postID == $postID) break;
			}
			if ($post->postID != $postID) {
				// For AJAX requests this has been checked in validateMerge(), but that
				// method cannot be used for 'internal' merges.
				throw new UserInputException('postOrder');
			}
			$usernames[$post->username] = $post->username;
			
			// merge post messages
			$message .= "<hr>";
			$message .= $post->message;
			
			// merge attachments
			if ($post->attachments) {
				$attachmentsCount += $post->attachments;
				$attachmentPostIDs[] = $post->postID;
			}
			
			if ($post->pollID && !$pollID) {
				$pollID = $post->pollID;
				// detach poll from old post, otherwise it's going to be deleted with the post
				(new PollEditor(new Poll($pollID)))->update([
					'objectID' => $this->postEditor->postID
				]);
				(new PostEditor($post->getDecoratedObject()))->update([
					'pollID' => null
				]);
				// reload post
				$post = new ViewablePost(new Post($post->postID));
			}
			unset($post);
		}
		sort($usernames);
		
		// merge attachments
		if (!empty($attachmentPostIDs)) {
			AttachmentHandler::transferAttachments('com.woltlab.wbb.post', $this->postEditor->postID, $attachmentPostIDs);
		}
		
		// update post
		$postAction = new self([$this->postEditor], 'update', [
			// pretend it's an edit to trigger edit history
			'isEdit' => true,
			'data' => [
				'attachments' => $attachmentsCount,
				'message' => $message,
				'pollID' => $pollID,
				'editReason' => WCF::getLanguage()->getDynamicVariable('wbb.post.edit.merge.reason', [
					'postOrder' => $this->parameters['postOrder'],
					'usernames' => $usernames
				]),
				'editCount' => $this->postEditor->editCount + 1,
				'lastEditTime' => TIME_NOW,
				'editor' => WCF::getUser()->username,
				'editorID' => WCF::getUser()->userID
			],
			'createModificationLogs' => false,
			'showEditNote' => true
		]);
		$postAction->executeAction();
		
		// remove likes
		LikeHandler::getInstance()->removeLikes('com.woltlab.wbb.likeablePost', array_keys($this->objects));
		
		// remove old posts
		$posts = [];
		foreach ($this->getObjects() as $postEditor) {
			$posts[$postEditor->postID] = $postEditor->getDecoratedObject();
		}
		$postAction = new self($posts, 'delete', [
			'createModificationLogs' => false
		]);
		$postAction->executeAction();
		
		// create modification log entry
		PostModificationLogHandler::getInstance()->merge($this->postEditor->getDecoratedObject(), count($this->parameters['postOrder']), $usernames);
		WCF::getDB()->commitTransaction();
		
		// unmark post ids
		$postIDs = array_keys($posts);
		$postIDs[] = $this->postEditor->postID;
		$this->unmarkItems($postIDs);
		
		// redirect to merge target
		return [
			'redirectURL' => $this->postEditor->getLink(),
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateGetPopover() {
		$this->postEditor = $this->getSingleObject();
		if (!$this->postEditor->canRead()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getPopover() {
		$postList = new SimplifiedViewablePostList();
		$postList->getConditionBuilder()->add('post.postID = ?', [$this->postEditor->postID]);
		$postList->sqlLimit = 1;
		$postList->readObjects();
		
		$post = $postList->getSingleObject();
		if ($post === null) {
			return [];
		}
		
		return [
			'template' => WCF::getTPL()->fetch('postPreview', 'wbb', [
				'post' => $post,
			]),
		];
	}
	
	/**
	 * Validates the get post preview action.
	 * 
	 * @deprecated  5.3     Use `getPopover()` instead.
	 */
	public function validateGetPostPreview() {
		return $this->validateGetPopover();
	}
	
	/**
	 * Returns a preview of a post.
	 * 
	 * @return	string[]
	 * @deprecated  5.3     Use `validateGetPopover()` instead.
	 */
	public function getPostPreview() {
		return $this->getPopover();
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateQuickReply() {
		// check flood control
		Post::enforceFloodControl();
		
		QuickReplyManager::getInstance()->setDisallowedBBCodes(explode(',', WCF::getSession()->getPermission('user.message.disallowedBBCodes')));
		QuickReplyManager::getInstance()->validateParameters($this, $this->parameters, Thread::class, ViewableThread::class);
		
		// validate poll
		$this->thread = new Thread($this->parameters['objectID']);
		if (!empty($this->parameters['data']['poll'])) {
			if (!$this->thread->canUsePoll(true)) {
				throw new UserInputException('poll');
			}
			
			PollManager::getInstance()->setObject('com.woltlab.wbb.post', 0);
			PollManager::getInstance()->readFormParameters($this->parameters['data']['poll']);
			PollManager::getInstance()->validate();
		}
		
		if (!WCF::getUser()->userID) {
			$this->readBoolean('requireGuestDialog', true);
			
			$this->setGuestDialogCaptcha();
			if (!$this->parameters['requireGuestDialog']) {
				$this->validateGuestDialogUsername();
				$this->validateGuestDialogCaptcha();
			}
		}
		
		$this->readBoolean('subscribeThread', true);
		
		if (WCF::getUser()->userID && $this->thread->getBoard()->getModeratorPermission('canEnableThread')) {
			$this->readInteger('disablePost', true);
			$this->readString('enableTime', true);
			
			if ($this->parameters['disablePost']) {
				$this->parameters['data']['isDisabled'] = 1;
				
				if (!empty($this->parameters['enableTime'])) {
					$enableTime = \DateTime::createFromFormat('Y-m-d\TH:i:sP', $this->parameters['enableTime']);
					if (!$enableTime || $enableTime->getTimestamp() < TIME_NOW) {
						throw new UserInputException('enableTime', 'invalid');
					}
					
					$this->parameters['data']['enableTime'] = $enableTime->getTimestamp();
				}
			}
		}
		else {
			unset($this->parameters['disablePost']);
			unset($this->parameters['enableTime']);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function quickReply() {
		// return guest dialog if (i) it is explicitly requested or (ii) if the guest dialog data is erroneous
		if (!WCF::getUser()->userID) {
			if ($this->parameters['requireGuestDialog'] || !empty($this->guestDialogErrors)) {
				return [
					'guestDialogID' => 'postAdd' . $this->parameters['objectID'],
					'guestDialog' => WCF::getTPL()->fetch('messageQuickReplyGuestDialog', 'wcf', [
						'ajaxCaptcha' => true,
						'supportsAsyncCaptcha' => true,
						'captchaID' => 'postAdd' . $this->parameters['objectID'],
						'captchaObjectType' => $this->guestDialogCaptchaObjectType,
						'errorType' => $this->guestDialogErrors,
						'username' => !empty($this->parameters['data']['username']) ? $this->parameters['data']['username'] : WCF::getSession()->getVar('username')
					])
				];
			}
			else {
				WCF::getSession()->register('username', $this->parameters['data']['username']);
			}
		}
		
		/** @noinspection PhpUndefinedMethodInspection */
		if (!QuickReplyManager::getInstance()->getContainer()->getBoard()->getPermission('canReplyThreadWithoutModeration')) {
			$this->parameters['data']['isDisabled'] = 1;
		}
		
		WCF::getTPL()->assign('disableAds', true);
		
		if (!empty($this->parameters['data']['poll'])) {
			unset($this->parameters['data']['poll']);
		}
		
		// Check if post should be merged with the previous one
		$sql = "SELECT		postID
			FROM		wbb".WCF_N."_post
			WHERE		threadID = ?
			ORDER BY	time DESC";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->parameters['objectID']]);
		$lastPost = new Post($statement->fetchSingleColumn());
		
		// don't merge posts of different users
		$shouldMerge = ($lastPost->userID && WCF::getUser()->userID && $lastPost->userID == WCF::getUser()->userID);
		
		// Do not merge deleted or disabled posts.
		foreach (['isDeleted', 'isDisabled'] as $property) {
			if ($lastPost->{$property} || (isset($this->parameters['data'][$property]) && $this->parameters['data'][$property])) {
				$shouldMerge = false;
				break;
			}
		}
		
		// check if maximum text length would be exceeded
		$htmlInputProcessor = new HtmlInputProcessor();
		$htmlInputProcessor->process(
			$lastPost . "<hr>" . $this->parameters['htmlInputProcessor']->getHtml(),
			'com.woltlab.wbb.post'
		);
		if (mb_strlen($htmlInputProcessor->getTextContent()) > WCF::getSession()->getPermission('user.board.maxPostLength')) {
			$shouldMerge = false;
		}
		
		// Do not check attachments if there is no tmpHash, which indicates
		// that the user was not allowed to upload them in the first place.
		if (!empty($this->parameters['tmpHash'])) {
			// Check if the maximum number of attachments would be exceeded.
			$attachmentHandler = $this->getAttachmentHandler($this->thread);
			if ($lastPost->attachments + $attachmentHandler->count() > $attachmentHandler->getMaxCount()) {
				$shouldMerge = false;
			}
		}
		
		// check if cut-off time is exceeded
		$mergePeriod = WCF::getSession()->getPermission('user.board.doublePostLock');
		if ($mergePeriod != -1 && $lastPost->time < TIME_NOW - $mergePeriod * 60) {
			$shouldMerge = false;
		}
		
		// Don't merge if previous post has a poll. We technically could
		// if this post does not, but we cannot reliably detect that at
		// this point
		if ($lastPost->pollID) {
			$shouldMerge = false;
		}
		
		// Plugins might add data to a post that prevents merging (similar to polls).
		$eventParameters = [
			'lastPost' => $lastPost,
			'shouldMerge' => $shouldMerge
		];
		EventHandler::getInstance()->fireAction($this, 'quickReplyShouldMerge', $eventParameters);
		$shouldMerge = $eventParameters['shouldMerge'];
		
		if ($shouldMerge) {
			$this->parameters['data']['isDisabled'] = 1;
		}
		
		/** @noinspection PhpUndefinedMethodInspection */
		$returnValues = QuickReplyManager::getInstance()->createMessage(
			$this,
			$this->parameters,
			ThreadAction::class,
			QuickReplyManager::getInstance()->getContainer()->getBoard()->getPostSortOrder(),
			'threadPostList',
			'wbb',
			function($message) {
				/** @var Post $message */
				
				// save polls
				if ($this->thread && $this->thread->canUsePoll(true)) {
					$addedPoll = false;
					$pollID = PollManager::getInstance()->save($message->postID);
					if ($pollID) {
						$postEditor = new PostEditor($message);
						$postEditor->update([
							'pollID' => $pollID
						]);
						
						$addedPoll = true;
					}
					
					if ($addedPoll) {
						$threadEditor = new ThreadEditor($this->thread);
						$threadEditor->updateCounters([
							'polls' => 1
						]);
					}
				}
			}
		);
		// Perform automated merge.
		if ($shouldMerge) {
			$mergeAction = new self([$returnValues['objectID']], 'merge', [
				'postOrder' => [
					$returnValues['objectID']
				]
			]);
			$mergeAction->readObjects();
			$mergeAction->postEditor = new PostEditor($lastPost);
			$mergeAction->executeAction();
			
			// Modify the return values, they point to a deleted post otherwise.
			$returnValues = [
				'objectID' => $lastPost->getObjectID(),
				'url' => $this->getRedirectUrl($lastPost->getThread(), $lastPost)
			];
		}
		
		$thread = new Thread($this->parameters['data']['threadID']);
		$threadEditor = new ThreadEditor($thread);
		
		if (WCF::getUser()->userID) {
			$returnValues['isSubscribed'] = $thread->isSubscribed() ? 1 : 0;
		}
		// handle close topic
		if ($thread->getBoard()->getModeratorPermission('canCloseThread') && !empty($this->parameters['closeThread'])) {
			if (!$thread->isClosed) {
				$threadEditor->update([
					'isClosed' => 1
				]);
				
				ThreadModificationLogHandler::getInstance()->close($threadEditor->getDecoratedObject());
			}
			
			$returnValues['isClosed'] = 1;
		}
		
		// handle mark as resolve
		if (!$thread->isDone && $thread->canMarkAsDone() && !empty($this->parameters['markAsDone'])) {
			if (!$thread->isDone) {
				$threadEditor->update([
					'isDone' => 1
				]);
				
				ThreadModificationLogHandler::getInstance()->done($threadEditor->getDecoratedObject());
			}
			
			$returnValues['isDone'] = 1;
		}
		
		if (!WCF::getUser()->userID) {
			$returnValues['guestDialogID'] = 'postAdd' . $this->parameters['objectID'];
		}
		
		// reset captcha
		if ($this->guestDialogCaptchaObjectType !== null) {
			/** @var ICaptchaHandler $processor */
			$processor = $this->guestDialogCaptchaObjectType->getProcessor();
			$processor->reset();
		}
		
		return $returnValues;
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateMessage(DatabaseObject $thread, HtmlInputProcessor $htmlInputProcessor) {
		$message = $htmlInputProcessor->getTextContent();
		
		if (WBB_POST_MIN_CHAR_LENGTH && (mb_strlen($message) < WBB_POST_MIN_CHAR_LENGTH)) {
			throw new UserInputException('message', WCF::getLanguage()->getDynamicVariable('wbb.post.message.error.minCharLength', ['minCharLength' => WBB_POST_MIN_CHAR_LENGTH]));
		}
		
		if (WBB_POST_MIN_WORD_COUNT && (count(explode(' ', $message)) < WBB_POST_MIN_WORD_COUNT)) {
			throw new UserInputException('message', WCF::getLanguage()->getDynamicVariable('wbb.post.message.error.minWordCount', ['minWordCount' => WBB_POST_MIN_WORD_COUNT]));
		}
		
		if ($htmlInputProcessor->appearsToBeEmpty()) {
			throw new UserInputException('message', WCF::getLanguage()->getDynamicVariable('wcf.global.form.error.empty'));
		}
		
		if (mb_strlen($message) > WCF::getSession()->getPermission('user.board.maxPostLength')) {
			throw new UserInputException('message', WCF::getLanguage()->getDynamicVariable('wcf.message.error.tooLong', ['maxTextLength' => WCF::getSession()->getPermission('user.board.maxPostLength')]));
		}
		
		// search for disallowed bbcodes
		$disallowedBBCodes = $htmlInputProcessor->validate();
		if (!empty($disallowedBBCodes)) {
			throw new UserInputException('text', WCF::getLanguage()->getDynamicVariable('wcf.message.error.disallowedBBCodes', ['disallowedBBCodes' => $disallowedBBCodes]));
		}
		
		// search for censored words
		if (ENABLE_CENSORSHIP) {
			$result = Censorship::getInstance()->test($message);
			if ($result) {
				throw new UserInputException('message', WCF::getLanguage()->getDynamicVariable('wcf.message.error.censoredWordsFound', ['censoredWords' => $result]));
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateContainer(DatabaseObject $thread) {
		/** @var ViewableThread $thread */
		
		if (!$thread->threadID) {
			throw new UserInputException('objectID');
		}
		
		if (!$thread->canReply()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessageList(DatabaseObject $thread, $lastMessageTime) {
		/** @var ViewableThread $thread */
		
		$messageList = new ThreadPostList($thread->getDecoratedObject());
		$messageList->getConditionBuilder()->add("post.time > ?", [$lastMessageTime]);
		$messageList->sqlOrderBy = "post.time ".$thread->getBoard()->getPostSortOrder();
		$messageList->readObjects();
		
		return $messageList;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getPageNo(DatabaseObject $thread) {
		/** @var ViewableThread $thread */
		
		$sql = "SELECT	COUNT(*) AS count
			FROM	wbb".WCF_N."_post
			WHERE	threadID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$thread->threadID]);
		$count = $statement->fetchArray();
		
		return [intval(ceil($count['count'] / $thread->getBoard()->getPostsPerPage())), $count['count']];
	}
	
	/**
	 * @inheritDoc
	 */
	public function getRedirectUrl(DatabaseObject $thread, DatabaseObject $post) {
		/** @var Post $post */
		return $post->getLink();
	}
	
	/**
	 * Validates the 'getPosts' action.
	 */
	public function validateGetPosts() {
		if (empty($this->objectIDs)) {
			throw new UserInputException('objectIDs');
		}
		
		$post = new Post(reset($this->objectIDs));
		if (!$post->postID) {
			throw new UserInputException('objectIDs');
		}
		$this->thread = new Thread($post->threadID);
		
		$this->postList = new ViewablePostList();
		$this->postList->setObjectIDs($this->objectIDs);
		$this->postList->setThread($this->thread);
		$this->postList->readObjects();
		
		// make sure that all posts belong to the same thread
		foreach ($this->postList as $post) {
			if (!$post->canRead()) {
				throw new PermissionDeniedException();
			}
			
			if ($post->threadID != $this->thread->threadID) {
				throw new UserInputException('objectIDs');
			}
		}
	}
	
	/**
	 * Loads a list of posts that belong to one thread.
	 * 
	 * @return	string[]
	 */
	public function getPosts() {
		if (isset($this->parameters['showCollapsedPosts'])) {
			WCF::getTPL()->assign([
				'showCollapsedPosts' => true
			]);
		}
		
		$this->postList->rewind();
		$firstPost = $this->postList->current();
		
		WCF::getTPL()->assign([
			'attachmentList' => $this->postList->getAttachmentList(),
			'objects' => $this->postList,
			'sortOrder' => $this->thread->getBoard()->getPostSortOrder(),
			'thread' => new ViewableThread($this->thread),
			'startIndex' => $firstPost->getPostNumber()
		]);
		
		return [
			'template' => WCF::getTPL()->fetch('threadPostList', 'wbb')
		];
	}
	
	/**
	 * Updates the thread form option for a post.
	 *
	 * @since       5.2
	 */
	public function updateThreadFormValues() {
		if (!count($this->objects)) {
			$this->readObjects();
		}
		
		$sql = "INSERT INTO	wbb".WCF_N."_thread_form_option_value
					(postID, optionID, optionValue)
			VALUES		(?, ?, ?)";
		$insertStatement = WCF::getDB()->prepareStatement($sql);
		
		$sql = "DELETE FROM       wbb".WCF_N."_thread_form_option_value
			WHERE             postID = ?";
		$deleteStatement = WCF::getDB()->prepareStatement($sql);
		
		foreach ($this->getObjects() as $object) {
			if (isset($this->parameters['optionHandler'])) {
				WCF::getDB()->beginTransaction();
				
				/** @var ThreadFormOptionHandler $optionHandler */
				$optionHandler = $this->parameters['optionHandler'];
				$saveValues = $optionHandler->save();
				
				$deleteStatement->execute([$object->postID]);
				
				foreach ($saveValues as $optionID => $optionValue) {
					$insertStatement->execute([
						$object->postID,
						$optionID,
						$optionValue
					]);
				}
				
				WCF::getDB()->commitTransaction();
			}
		}
	}
	
	/**
	 * Validates the validateThreadFormValues method.
	 *
	 * @since       5.2
	 */
	public function validateValidateThreadFormValues() {
		$post = $this->getSingleObject();
		
		if (!$post->getThread()->canEditPost($post->getDecoratedObject())) {
			throw new PermissionDeniedException();
		}
		
		if (!$post->isFirstPost()) {
			throw new IllegalLinkException();
		}
		
		if (!isset($this->parameters['values']) || !is_array($this->parameters['values'])) {
			$this->parameters['values'] = [];
		}
	}
	
	/**
	 * Validates thread form values for a specific thread.
	 *
	 * @return      string[]
	 * @since       5.2
	 */
	public function validateThreadFormValues() {
		$optionHandler = new ThreadFormOptionHandler(false);
		$optionHandler->setPost($this->getSingleObject()->getDecoratedObject());
		$optionHandler->readUserInput($this->parameters);
		
		return [
			'errors' => $optionHandler->validate()
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateJumpToExtended() {
		throw new \BadMethodCallException("This method is no longer supported.");
	}
	
	/**
	 * @inheritDoc
	 */
	public function jumpToExtended() {
		throw new \BadMethodCallException("This method is no longer supported.");
	}
	
	/**
	 * @inheritDoc
	 */
	public function getAttachmentHandler(DatabaseObject $thread) {
		/** @var Thread $thread */
		
		return new AttachmentHandler('com.woltlab.wbb.post', 0, $this->parameters['tmpHash'], $thread->boardID);
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateBeginEdit() {
		$this->parameters['containerID'] = isset($this->parameters['containerID']) ? intval($this->parameters['containerID']) : 0;
		if (!$this->parameters['containerID']) {
			throw new UserInputException('containerID');
		}
		else {
			$this->thread = new Thread($this->parameters['containerID']);
			if (!$this->thread->threadID) {
				throw new UserInputException('containerID');
			}
		}
		
		$this->parameters['objectID'] = isset($this->parameters['objectID']) ? intval($this->parameters['objectID']) : 0;
		if (!$this->parameters['objectID']) {
			throw new UserInputException('objectID');
		}
		else {
			$this->post = new Post($this->parameters['objectID']);
			if (!$this->post->postID || ($this->thread->threadID != $this->post->threadID)) {
				throw new UserInputException('objectID');
			}
			
			if (!$this->thread->canEditPost($this->post)) {
				throw new PermissionDeniedException();
			}
		}
		
		$this->post->setThread($this->thread);
		
		if ($this->thread->canUsePoll(true, $this->post)) PollManager::getInstance()->setObject('com.woltlab.wbb.post', $this->post->postID, $this->post->pollID);
		
		BBCodeHandler::getInstance()->setDisallowedBBCodes(explode(',', WCF::getSession()->getPermission('user.message.disallowedBBCodes')));
	}
	
	/**
	 * @inheritDoc
	 */
	public function beginEdit() {
		WCF::getTPL()->assign([
			'post' => $this->post,
			'thread' => $this->thread,
			'wysiwygSelector' => 'messageEditor'.$this->post->postID
		]);
		
		$tmpHash = StringUtil::getRandomID();
		$attachmentHandler = new AttachmentHandler('com.woltlab.wbb.post', $this->post->postID, $tmpHash, $this->thread->boardID);
		$attachmentList = $attachmentHandler->getAttachmentList();
		
		WCF::getTPL()->assign([
			'attachmentHandler' => $attachmentHandler,
			'attachmentList' => $attachmentList->getObjects(),
			'attachmentObjectID' => $this->post->postID,
			'attachmentObjectType' => 'com.woltlab.wbb.post',
			'attachmentParentObjectID' => $this->thread->boardID,
			'tmpHash' => $tmpHash,
		]);
		
		if ($this->thread->canUsePoll(true, $this->post)) {
			PollManager::getInstance()->assignVariables();
		}
		
		if ($this->post->isFirstPost()) {
			$optionHandler = new ThreadFormOptionHandler(false);
			$optionHandler->setPost($this->post);
			
			WCF::getTPL()->assign([
				'threadFormOptionHandlerOptions' => $optionHandler->getOptions(),
				'threadForm' => $this->thread->getBoard()->formID ? new ThreadForm($this->thread->getBoard()->formID) : null,
				'errorType' => '',
			]);
		}
		
		return [
			'actionName' => 'beginEdit',
			'template' => WCF::getTPL()->fetch('postInlineEditor', 'wbb')
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateSave() {
		if (!isset($this->parameters['data']) || !isset($this->parameters['data']['message'])) {
			throw new UserInputException('message');
		}
		
		$this->parameters['data']['message'] = StringUtil::trim(MessageUtil::stripCrap($this->parameters['data']['message']));
		
		if (empty($this->parameters['data']['message'])) {
			throw new UserInputException('message', WCF::getLanguage()->getDynamicVariable('wcf.global.form.error.empty'));
		}
		
		$this->validateBeginEdit();
		$this->board = BoardCache::getInstance()->getBoard($this->thread->boardID);
		
		$this->validateMessage($this->thread, $this->getHtmlInputProcessor($this->parameters['data']['message'], $this->post->postID));
		
		// edit note
		$this->readString('editReason', true);
		
		if ($this->board->getPermission('canHideEditNote')) {
			$this->readBoolean('showEditNote', true);
			$this->readBoolean('removeEditNote', true);
		}
		else {
			$this->parameters['showEditNote'] = true;
			$this->parameters['removeEditNote'] = false;
			
			if (WCF::getUser()->userID == $this->post->userID && $this->post->time <= TIME_NOW - WBB_POST_EDIT_HIDE_EDIT_NOTE_PERIOD * 60) {
				// suppress edit notice if current user is the author and within editing grace period
				$this->parameters['showEditNote'] = false;
			}
		}
		
		if ($this->thread->canUsePoll(true, $this->post)) {
			PollManager::getInstance()->readFormParameters($this->parameters['poll']);
			PollManager::getInstance()->validate();
		}
		
		if ($this->post->isFirstPost()) {
			$this->parameters['optionHandler'] = new ThreadFormOptionHandler(false);
			$this->parameters['optionHandler']->setPost($this->post);
			$threadFormOptions = isset($this->parameters['threadFormOptions']) && !empty($this->parameters['threadFormOptions']) ? $this->parameters['threadFormOptions'] : [];
			$this->parameters['optionHandler']->readUserInput($threadFormOptions);
			$errors = $this->parameters['optionHandler']->validate();
			
			if (!empty($errors)) {
				throw new UserInputException('options', $errors);
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		$postData = [];
		
		// append edit note
		$board = BoardCache::getInstance()->getBoard($this->thread->boardID);
		$postData['editReason'] = $this->parameters['editReason'] === null ? '' : $this->parameters['editReason'];
		if (!$board->getPermission('canHideEditNote')) {
			$this->parameters['showEditNote'] = true;
			$this->parameters['removeEditNote'] = false;
		}
		
		if ($this->parameters['removeEditNote']) {
			$postData['editCount'] = 0;
			$postData['editor'] = '';
			$postData['editorID'] = null;
			$postData['lastEditTime'] = 0;
		}
		else if ($this->parameters['showEditNote'] && (WCF::getUser()->userID != $this->post->userID || $this->post->time <= TIME_NOW - WBB_POST_EDIT_HIDE_EDIT_NOTE_PERIOD * 60)) {
			// showEditNote is true if the note should not be suppressed or the user has no permissions to do so
			$postData['editCount'] = $this->post->editCount + 1;
			$postData['editor'] = WCF::getUser()->username;
			$postData['editorID'] = WCF::getUser()->userID;
			$postData['lastEditTime'] = TIME_NOW;
		}
		
		$attachmentHandler = new AttachmentHandler('com.woltlab.wbb.post', $this->post->postID);
		$postData['attachments'] = count($attachmentHandler);
		
		// save option handler
		if ($this->post->isFirstPost()) {
			(new PostAction([$this->post], 'updateThreadFormValues', [
				'optionHandler' => $this->parameters['optionHandler']
			]))->executeAction();
			
			// refresh cache
			$this->post = new Post($this->post->postID);
		}
		
		// execute update action
		$action = new PostAction([$this->post], 'update', [
			'data' => $postData,
			'htmlInputProcessor' => $this->getHtmlInputProcessor(),
			'showEditNote' => $this->parameters['showEditNote'],
			'isEdit' => true
		]);
		$action->executeAction();
		
		// clear quotes
		if (isset($this->parameters['removeQuoteIDs']) && !empty($this->parameters['removeQuoteIDs'])) {
			MessageQuoteManager::getInstance()->markQuotesForRemoval($this->parameters['removeQuoteIDs']);
		}
		MessageQuoteManager::getInstance()->removeMarkedQuotes();
		
		// load new post
		$post = new Post($this->post->postID);
		
		// get all attachments except embedded ones
		$attachmentList = $post->getAttachments(true);
		if ($attachmentList !== null) {
			// set permissions
			$attachmentList->setPermissions([
				'canDownload' => $board->getPermission('canDownloadAttachment'),
				'canViewPreview' => $board->getPermission('canViewAttachmentPreview'),
			]);
		}
		
		$pollTemplate = '';
		if ($this->thread->canUsePoll(true, $this->post)) {
			$pollCount = 0;
			$pollID = PollManager::getInstance()->save($this->post->postID);
			if ($pollID && $pollID != $this->post->pollID) {
				$postEditor = new PostEditor($this->post);
				$postEditor->update([
					'pollID' => $pollID
				]);
				
				$pollCount++;
			}
			else if (!$pollID && $this->post->pollID) {
				$postEditor = new PostEditor($this->post);
				$postEditor->update([
					'pollID' => null
				]);
				
				$pollCount--;
			}
			
			if ($pollCount != 0) {
				$threadEditor = new ThreadEditor($this->thread);
				$threadEditor->updateCounters(['polls' => $pollCount]);
			}
			
			// reload post
			$this->post = new Post($this->post->postID);
			
			if ($this->post->getPoll()) {
				$pollTemplate = WCF::getTPL()->fetch('poll', 'wcf', ['poll' => $this->post->getPoll()]);
			}
		}
		
		// load embedded objects
		MessageEmbeddedObjectManager::getInstance()->loadObjects('com.woltlab.wbb.post', [$post->postID]);
		
		if ($this->post->isFirstPost() && isset($this->parameters['optionHandler']) && !empty($this->parameters['optionHandler']->getOptions())) {
			$threadFormOptionsTemplate = WCF::getTPL()->fetch('threadFormOptions', 'wbb', [
				'threadFormOptions' => $this->parameters['optionHandler']->getOptions()
			]);
		}
		else {
			$threadFormOptionsTemplate = "";
		}
		
		$data = [
			'actionName' => 'save',
			'message' => $post->getFormattedMessage(),
			'notes' => ['wbbPostEditNote' => ''],
			'poll' => $pollTemplate,
			'threadFormOptions' => $threadFormOptionsTemplate
		];
		
		if ($post->editCount || ($post->getThread()->canEditPost($post) && $post->hasOldVersions())) {
			$data['notes']['wbbPostEditNote'] = WCF::getLanguage()->getDynamicVariable('wbb.post.editNote', [
				'post' => new PostEditor($post),
				'thread' => $post->getThread()
			]);
		}
		
		WCF::getTPL()->assign([
			'attachmentList' => $attachmentList,
			'objectID' => $this->post->postID
		]);
		$data['attachmentList'] = WCF::getTPL()->fetch('attachments');
		
		return $data;
	}
	
	/**
	 * Validates parameters to return the logged ip addresses.
	 */
	public function validateGetIpLog() {
		if (!LOG_IP_ADDRESS) {
			throw new PermissionDeniedException();
		}
		
		if (isset($this->parameters['postID'])) {
			$this->post = new Post($this->parameters['postID']);
		}
		if ($this->post === null || !$this->post->postID) {
			throw new UserInputException('postID');
		}
		
		if (!$this->post->canRead()) {
			throw new PermissionDeniedException();
		}
		
		WCF::getSession()->checkPermissions(['admin.user.canViewIpAddress']);
	}
	
	/**
	 * Returns a list of the logged ip addresses.
	 * 
	 * @return	array
	 */
	public function getIpLog() {
		// get ip addresses of the author
		$authorIpAddresses = Post::getIpAddressByAuthor($this->post->userID, $this->post->username, $this->post->ipAddress);
		
		// resolve hostnames
		$authorIpAddresses = array_map(function ($item) {
			$item['ipAddress'] = UserUtil::convertIPv6To4($item['ipAddress']);
			$item['hostname'] = @gethostbyaddr($item['ipAddress']);
			return $item;
		}, $authorIpAddresses);
		
		// get other users of this ip address
		$otherUsers = [];
		if ($this->post->ipAddress) {
			$otherUsers = Post::getAuthorByIpAddress($this->post->ipAddress, $this->post->userID, $this->post->username);
		}
		
		$ipAddress = UserUtil::convertIPv6To4($this->post->ipAddress);
		
		if ($this->post->userID) {
			$sql = "SELECT	registrationIpAddress, registrationDate
				FROM	wcf".WCF_N."_user
				WHERE	userID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([
				$this->post->userID
			]);
			$row = $statement->fetchArray();
			
			if ($row !== false && $row['registrationIpAddress']) {
				$registrationIpAddress = UserUtil::convertIPv6To4($row['registrationIpAddress']);
				WCF::getTPL()->assign([
					'registrationIpAddress' => [
						'hostname' => @gethostbyaddr($registrationIpAddress),
						'ipAddress' => $registrationIpAddress,
						'time' => $row['registrationDate']
					]
				]);
			}
		}
		
		WCF::getTPL()->assign([
			'authorIpAddresses' => $authorIpAddresses,
			'ipAddress' => [
				'hostname' => @gethostbyaddr($ipAddress),
				'ipAddress' => $ipAddress,
				'time' => $this->post->time
			],
			'otherUsers' => $otherUsers,
			'post' => $this->post
		]);
		
		return [
			'postID' => $this->post->postID,
			'template' => WCF::getTPL()->fetch('postIpAddress', 'wbb')
		];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateSaveFullQuote() {
		$this->post = $this->getSingleObject();
		if (!$this->post->canRead()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function saveFullQuote() {
		$quoteID = MessageQuoteManager::getInstance()->addQuote(
			'com.woltlab.wbb.post',
			$this->post->threadID,
			$this->post->postID,
			$this->post->getExcerpt(),
			$this->post->getMessage()
		);
		
		if ($quoteID === false) {
			$removeQuoteID = MessageQuoteManager::getInstance()->getQuoteID(
				'com.woltlab.wbb.post',
				$this->post->postID,
				$this->post->getExcerpt(),
				$this->post->getMessage()
			);
			MessageQuoteManager::getInstance()->removeQuote($removeQuoteID);
		}
		
		$returnValues = [
			'count' => MessageQuoteManager::getInstance()->countQuotes(),
			'fullQuoteMessageIDs' => MessageQuoteManager::getInstance()->getFullQuoteObjectIDs(['com.woltlab.wbb.post'])
		];
		
		if ($quoteID) {
			$returnValues['renderedQuote'] = MessageQuoteManager::getInstance()->getQuoteComponents($quoteID);
		}
		
		return $returnValues;
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateSaveQuote() {
		$this->readString('message');
		$this->readBoolean('renderQuote', true);
		
		if (empty($this->objects)) {
			$this->readObjects();
			
			if (empty($this->objects)) {
				throw new UserInputException('objectIDs');
			}
		}
		
		$this->post = current($this->objects);
		if (!$this->post->canRead()) {
			throw new PermissionDeniedException();
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function saveQuote() {
		$quoteID = MessageQuoteManager::getInstance()->addQuote(
			'com.woltlab.wbb.post',
			$this->post->threadID,
			$this->post->postID,
			$this->parameters['message'],
			false
		);
		
		$returnValues = [
			'count' => MessageQuoteManager::getInstance()->countQuotes(),
			'fullQuoteMessageIDs' => MessageQuoteManager::getInstance()->getFullQuoteObjectIDs(['com.woltlab.wbb.post'])
		];
		
		if ($this->parameters['renderQuote']) {
			$returnValues['renderedQuote'] = MessageQuoteManager::getInstance()->getQuoteComponents($quoteID);
		}
		
		return $returnValues;
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateGetRenderedQuotes() {
		$this->readInteger('parentObjectID');
		
		$this->thread = new Thread($this->parameters['parentObjectID']);
		if (!$this->thread->threadID) {
			throw new UserInputException('parentObjectID');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getRenderedQuotes() {
		$quotes = MessageQuoteManager::getInstance()->getQuotesByParentObjectID('com.woltlab.wbb.post', $this->thread->threadID);
		
		return [
			'template' => implode("\n\n", $quotes)
		];
	}
	
	/**
	 * Validates the 'unmarkAll' action.
	 */
	public function validateUnmarkAll() {
		// does nothing
	}
	
	/**
	 * Unmarks all posts.
	 */
	public function unmarkAll() {
		ClipboardHandler::getInstance()->removeItems(ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.post'));
	}
	
	/**
	 * Validates parameters to move posts into an existing thread.
	 */
	public function validateMoveToExistingThread() {
		$this->readInteger('threadID');
		
		$this->readObjects();
		if (empty($this->objects)) {
			throw new UserInputException('objectIDs');
		}
		
		// validate thread and board permissions
		$this->thread = new Thread($this->parameters['threadID']);
		if (!$this->thread->threadID) {
			throw new UserInputException('threadID');
		}
		else if (!$this->thread->canRead() || !$this->thread->getBoard()->getModeratorPermission('canMovePost')) {
			throw new PermissionDeniedException();
		}
		
		$threadIDs = [];
		foreach ($this->getObjects() as $post) {
			$threadIDs[] = $post->threadID;
			
			// cannot move post if origin and destination are equal
			if ($post->threadID == $this->thread->threadID) {
				throw new UserInputException('threadID');
			}
		}
		
		// read threads
		$threadList = new ThreadList();
		$threadList->setObjectIDs($threadIDs);
		$threadList->readObjects();
		$threads = $threadList->getObjects();
		
		// assign threads and validate permissions
		foreach ($this->getObjects() as $post) {
			$post->setThread($threads[$post->threadID]);
			if (!$post->canRead() || !$post->getThread()->getBoard()->getModeratorPermission('canMovePost')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Moves posts into an existing thread.
	 * 
	 * @return	array
	 */
	public function moveToExistingThread() {
		$addEvents = $boardIDs = $removeEvents = $threadIDs = $userPosts = $postIDs = [];
		foreach ($this->getObjects() as $post) {
			$boardIDs[] = $post->getThread()->boardID;
			$threadIDs[] = $post->threadID;
			$postIDs[] = $post->postID;
			
			PostModificationLogHandler::getInstance()->move($post->getDecoratedObject());
			
			if (!$post->isDisabled && $post->userID) {
				if (!isset($userPosts[$post->userID])) {
					$userPosts[$post->userID] = 0;
				}
				
				if ($post->getThread()->getBoard()->countUserPosts != $this->thread->getBoard()->countUserPosts) {
					if ($post->getThread()->getBoard()->countUserPosts) {
						$userPosts[$post->userID]--;
						
						if (!isset($removeEvents[$post->userID])) {
							$removeEvents[$post->userID] = 0;
						}
						$removeEvents[$post->userID]++;
					}
					
					if ($this->thread->getBoard()->countUserPosts) {
						$userPosts[$post->userID]++;
						
						if (!isset($addEvents[$post->userID])) {
							$addEvents[$post->userID] = 0;
						}
						$addEvents[$post->userID]++;
					}
				}
			}
		}
		
		// move posts
		$postAction = new PostAction($this->objects, 'update', ['data' => ['threadID' => $this->thread->threadID]]);
		$postAction->executeAction();
		
		// update users post counter
		foreach ($userPosts as $userID => $count) {
			if (!$count) {
				unset($userPosts[$userID]);
			}
		}
		if (!empty($userPosts)) {
			PostEditor::updatePostCounter($userPosts);
		}
		
		// update activity points
		if (!empty($addEvents)) {
			UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $addEvents);
		}
		if (!empty($removeEvents)) {
			UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $removeEvents);
		}
		
		// update threads
		$threadIDs[] = $this->thread->threadID;
		$threadAction = new ThreadAction($threadIDs, 'rebuild');
		$threadAction->executeAction();
		
		// update board stats
		$boardIDs[] = $this->thread->boardID;
		$boardIDs = array_unique($boardIDs);
		foreach ($boardIDs as $boardID) {
			$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
			$boardEditor->rebuildStats();
			$boardEditor->updateLastPost();
		}
		
		// reset board cache
		BoardEditor::resetCache();
		
		// update the modification logs with the new parent thread id
		PostModificationLogHandler::getInstance()->updateParentObjectID($postIDs, $this->thread->threadID);
		
		// reindex post with thread form values
		PostEditor::addPostIDsToSearchIndex($this->objectIDs);
		
		// unmark clipboard items
		$this->unmarkItems($this->objectIDs);
		
		return [
			'threadID' => $this->thread->threadID,
			'redirectURL' => $this->thread->getLink(),
		];
	}
	
	/**
	 * Validates parameters to move posts into a new thread.
	 */
	public function validateMoveToNewThread() {
		$this->readInteger('boardID');
		$this->readString('topic');
		
		$this->readObjects();
		if (empty($this->objects)) {
			throw new UserInputException('objectIDs');
		}
		
		// validate board
		$this->board = BoardCache::getInstance()->getBoard($this->parameters['boardID']);
		if ($this->board === null) {
			throw new UserInputException('boardID');
		}
		else if (!$this->board->canStartThread()) {
			throw new PermissionDeniedException();
		}
		
		$threadIDs = [];
		foreach ($this->getObjects() as $post) {
			$threadIDs[] = $post->threadID;
		}
		
		// read threads
		$threadList = new ThreadList();
		$threadList->setObjectIDs($threadIDs);
		$threadList->readObjects();
		$threads = $threadList->getObjects();
		
		// assign threads and validate permissions
		foreach ($this->getObjects() as $post) {
			$post->setThread($threads[$post->threadID]);
			if (!$post->getThread()->getBoard()->getModeratorPermission('canMovePost')) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Moves posts into a new thread.
	 * 
	 * @return	array
	 */
	public function moveToNewThread() {
		$addEvents = $boardIDs = $removeEvents = $threadIDs = $userPosts = $postIDs = [];
		
		/** @var PostEditor|null $firstPost */
		$firstPost = null;
		$threadData = [
			'attachments' => 0,
			'polls' => 0,
			'replies' => 0
		];
		
		foreach ($this->getObjects() as $post) {
			$postIDs[] = $post->postID;
			
			// identify the chronological first post
			if ($firstPost === null || $post->time < $firstPost->time) {
				$firstPost = $post;
			}
			
			$threadData['attachments'] += $post->attachments;
			$threadData['polls'] += $post->pollID ? 1 : 0;
			$threadData['replies']++;
			
			PostModificationLogHandler::getInstance()->move($post->getDecoratedObject());
			
			if (!$post->isDisabled && $post->userID) {
				if (!isset($userPosts[$post->userID])) {
					$userPosts[$post->userID] = 0;
				}
				
				if ($post->getThread()->getBoard()->countUserPosts != $this->board->countUserPosts) {
					if ($post->getThread()->getBoard()->countUserPosts) {
						$userPosts[$post->userID]--;
						
						if (!isset($removeEvents[$post->userID])) {
							$removeEvents[$post->userID] = 0;
						}
						$removeEvents[$post->userID]++;
					}
					
					if ($this->board->countUserPosts) {
						$userPosts[$post->userID]++;
						
						if (!isset($addEvents[$post->userID])) {
							$addEvents[$post->userID] = 0;
						}
						$addEvents[$post->userID]++;
					}
				}
			}
			
			$boardIDs[] = $post->getThread()->boardID;
			$threadIDs[] = $post->threadID;
		}
		
		// do not count first post as reply
		$threadData['replies']--;
		
		// create new thread
		$thread = ThreadEditor::create(array_merge($threadData, [
			'boardID' => $this->board->boardID,
			'languageID' => $firstPost->getThread()->languageID,
			'topic' => $this->parameters['topic'],
			'firstPostID' => $firstPost->postID,
			'time' => $firstPost->time,
			'userID' => $firstPost->userID ?: null,
			'username' => $firstPost->username,
			'cumulativeLikes' => $firstPost->cumulativeLikes,
			'isDeleted' => $firstPost->isDeleted,
			'isDisabled' => $firstPost->isDisabled
		]));
		
		// move posts to new thread
		$postAction = new PostAction($this->objects, 'update', ['data' => ['threadID' => $thread->threadID]]);
		$postAction->executeAction();
		
		// update last post for newly created thread
		$threadEditor = new ThreadEditor($thread);
		$threadEditor->updateLastPost();
		
		// set subject of first post
		$firstPost->update(['subject' => $this->parameters['topic']]);
		
		// update users post counter
		foreach ($userPosts as $userID => $count) {
			if (!$count) {
				unset($userPosts[$userID]);
			}
		}
		if (!empty($userPosts)) {
			PostEditor::updatePostCounter($userPosts);
		}
		
		// update activity points
		if (!empty($addEvents)) {
			UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $addEvents);
		}
		if (!empty($removeEvents)) {
			UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', $removeEvents);
		}
		
		// update old threads
		$threadAction = new ThreadAction($threadIDs, 'rebuild');
		$threadAction->executeAction();
		
		// update board stats
		$boardIDs[] = $this->board->boardID;
		$boardIDs = array_unique($boardIDs);
		foreach ($boardIDs as $boardID) {
			$boardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($boardID));
			$boardEditor->rebuildStats();
			$boardEditor->updateLastPost();
		}
		
		// reset board cache
		BoardEditor::resetCache();
		
		// update the modification logs with the new parent thread id
		PostModificationLogHandler::getInstance()->updateParentObjectID($postIDs, $thread->threadID);
		
		// reindex post with thread form values (for caching reasons, we must refresh the posts)
		PostEditor::addPostIDsToSearchIndex($this->objectIDs);
		
		// unmark clipboard items
		$this->unmarkItems($this->objectIDs);
		
		return [
			'threadID' => $thread->threadID,
			'redirectURL' => $thread->getLink(),
		];
	}
	
	/**
	 * Validates parameters to copy posts into a new thread.
	 */
	public function validateCopyToNewThread() {
		$this->readInteger('loopCount', true);
		$this->readInteger('threadID', true);
		
		if ($this->parameters['threadID']) {
			// validate thread
			$this->thread = new Thread($this->parameters['threadID']);
			if (!$this->thread->threadID || $this->thread->firstPostID) {
				throw new UserInputException('threadID');
			}
			else if (!$this->thread->getBoard()->getModeratorPermission('canCopyPost')) {
				throw new PermissionDeniedException();
			}
		}
		else {
			$this->readInteger('boardID');
			$this->readString('topic');
			
			$this->board = BoardCache::getInstance()->getBoard($this->parameters['boardID']);
			if ($this->board === null) {
				throw new UserInputException('boardID');
			}
			else if (!$this->board->canStartThread() || !$this->board->getModeratorPermission('canCopyPost')) {
				throw new PermissionDeniedException();
			}
		}
		
		if ($this->parameters['loopCount']) {
			if (!isset($this->parameters['objectIDs']) || !is_array($this->parameters['objectIDs'])) {
				throw new UserInputException('objectIDs');
			}
			
			$this->parameters['objectIDs'] = ArrayUtil::toIntegerArray($this->parameters['objectIDs']);
			asort($this->parameters['objectIDs'], SORT_NUMERIC);
			
			$this->parameters['postIDs'] = array_slice($this->parameters['objectIDs'], ($this->parameters['loopCount'] - 1) * 5, 5);
			if (!empty($this->parameters['objectIDs'])) {
				$conditions = new PreparedStatementConditionBuilder();
				$conditions->add("post.postID IN (?)", [$this->parameters['postIDs']]);
				
				$sql = "SELECT		post.postID, thread.boardID
					FROM		wbb".WCF_N."_post post
					LEFT JOIN	wbb".WCF_N."_thread thread
					ON		(thread.threadID = post.threadID)
					".$conditions;
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($conditions->getParameters());
				$posts = $statement->fetchMap('postID', 'boardID');
				
				foreach ($this->parameters['postIDs'] as $postID) {
					if (!isset($posts[$postID])) {
						throw new UserInputException('objectIDs');
					}
					
					if (!BoardCache::getInstance()->getBoard($posts[$postID])->getModeratorPermission('canCopyPost')) {
						throw new PermissionDeniedException();
					}
				}
			}
		}
	}
	
	/**
	 * Worker action to copy posts into a new thread.
	 * 
	 * @return	array
	 */
	public function copyToNewThread() {
		if ($this->parameters['loopCount'] == 0) {
			// get thread from designated first post
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("postID IN (?)", [$this->parameters['objectIDs']]);
			
			$sql = "SELECT  	thread.languageID
				FROM    	wbb".WCF_N."_thread thread
				WHERE           threadID IN (
							SELECT  DISTINCT threadID
							FROM    wbb".WCF_N."_post
							".$conditions."
						)
				ORDER BY        time";
			$statement = WCF::getDB()->prepareStatement($sql, 1);
			$statement->execute($conditions->getParameters());
			$row = $statement->fetchSingleRow();
			
			$thread = ThreadEditor::create([
				'boardID' => $this->parameters['boardID'],
				'topic' => $this->parameters['topic'],
				'languageID' => $row['languageID'] ?: null
			]);
			
			// first call, render template
			return [
				'loopCount' => 1,
				'parameters' => [
					'objectIDs' => $this->parameters['objectIDs'],
					'threadID' => $thread->threadID
				],
				'progress' => 0,
				'template' => WCF::getTPL()->fetch('worker')
			];
		}
		
		// copy posts
		$postAction = new PostAction([], 'copy', [
			'postIDs' => $this->parameters['postIDs'],
			'thread' => $this->thread
		]);
		$postAction->executeAction();
		
		$progress = floor(($this->parameters['loopCount'] / ceil(count($this->parameters['objectIDs']) / 5)) * 100);
		$returnValues = [
			'loopCount' => $this->parameters['loopCount'] + 1,
			'parameters' => [
				'objectIDs' => $this->parameters['objectIDs'],
				'threadID' => $this->thread->threadID
			],
			'progress' => $progress
		];
		
		if ($returnValues['progress'] == 100) {
			// rebuild thread
			$threadEditor = new ThreadEditor($this->thread);
			$threadEditor->rebuild();
			ThreadEditor::rebuildThreadData([$this->thread->threadID]);
			
			$this->thread = new Thread($this->thread->threadID);
			$post = new Post($this->thread->firstPostID);
			if ($post->userID && !$post->isDisabled && $this->thread->getBoard()->countUserPosts) {
				// fix activity points for first post (should be thread not post)
				UserActivityPointHandler::getInstance()->removeEvents('com.woltlab.wbb.activityPointEvent.post', [$post->userID => 1]);
				UserActivityPointHandler::getInstance()->fireEvent('com.woltlab.wbb.activityPointEvent.thread', $this->thread->threadID, $post->userID);
			}
			
			// update thread's user id
			$threadEditor->update([
				'userID' => $post->userID ?: null,
				'username' => $post->username,
				'time' => $post->time
			]);
			
			// set subject for first post
			$postEditor = new PostEditor($post);
			$postEditor->update(['subject' => $this->thread->topic]);
			
			// clear clipboard objects
			$this->unmarkItems($this->parameters['objectIDs']);
			
			$returnValues['redirectURL'] = $this->thread->getLink();
			
			// Update the board stats.
			(new BoardEditor($this->thread->getBoard()))->updateCounters([
				'posts' => $this->thread->replies + 1,
				'threads' => 1,
			]);
		}
		
		return $returnValues;
	}
	
	/**
	 * Validates parameters to copy posts into an existing thread.
	 */
	public function validateCopyToExistingThread() {
		$this->readInteger('loopCount', true);
		$this->readInteger('threadID');
		
		// validate thread
		$this->thread = new Thread($this->parameters['threadID']);
		if (!$this->thread->threadID) {
			throw new UserInputException('threadID');
		}
		else if (!$this->thread->getBoard()->getModeratorPermission('canCopyPost')) {
			throw new PermissionDeniedException();
		}
		
		if ($this->parameters['loopCount']) {
			if (!isset($this->parameters['objectIDs']) || !is_array($this->parameters['objectIDs'])) {
				throw new UserInputException('objectIDs');
			}
			
			$this->parameters['objectIDs'] = ArrayUtil::toIntegerArray($this->parameters['objectIDs']);
			asort($this->parameters['objectIDs'], SORT_NUMERIC);
			
			$this->parameters['postIDs'] = array_slice($this->parameters['objectIDs'], ($this->parameters['loopCount'] - 1) * 5, 5);
			if (!empty($this->parameters['objectIDs'])) {
				$conditions = new PreparedStatementConditionBuilder();
				$conditions->add("post.postID IN (?)", [$this->parameters['postIDs']]);
				
				$sql = "SELECT		post.postID, thread.boardID
					FROM		wbb".WCF_N."_post post
					LEFT JOIN	wbb".WCF_N."_thread thread
					ON		(thread.threadID = post.threadID)
					".$conditions;
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($conditions->getParameters());
				$posts = $statement->fetchMap('postID', 'boardID');
				
				foreach ($this->parameters['postIDs'] as $postID) {
					if (!isset($posts[$postID])) {
						throw new UserInputException('objectIDs');
					}
					
					if (!BoardCache::getInstance()->getBoard($posts[$postID])->getModeratorPermission('canCopyPost')) {
						throw new PermissionDeniedException();
					}
				}
			}
		}
	}
	
	/**
	 * Worker action to copy posts into an existing thread.
	 * 
	 * @return	array
	 */
	public function copyToExistingThread() {
		if ($this->parameters['loopCount'] == 0) {
			// first call, render template
			return [
				'loopCount' => 1,
				'parameters' => [
					'objectIDs' => $this->parameters['objectIDs'],
					'threadID' => $this->thread->threadID
				],
				'progress' => 0,
				'template' => WCF::getTPL()->fetch('worker')
			];
		}
		
		// copy posts
		$postAction = new PostAction([], 'copy', [
			'postIDs' => $this->parameters['postIDs'],
			'thread' => $this->thread
		]);
		$postAction->executeAction();
		
		$progress = floor(($this->parameters['loopCount'] / ceil(count($this->parameters['objectIDs']) / 5)) * 100);
		$returnValues = [
			'loopCount' => $this->parameters['loopCount'] + 1,
			'parameters' => [
				'objectIDs' => $this->parameters['objectIDs'],
				'threadID' => $this->thread->threadID
			],
			'progress' => $progress
		];
		
		if ($returnValues['progress'] == 100) {
			// rebuild thread
			$threadEditor = new ThreadEditor($this->thread);
			$threadEditor->rebuild();
			ThreadEditor::rebuildThreadData([$this->thread->threadID]);
				
			// clear clipboard objects
			$this->unmarkItems($this->parameters['objectIDs']);
		}
		
		return $returnValues;
	}
	
	/**
	 * Copies a list of posts.
	 */
	public function copy() {
		$postList = new PostList();
		$postList->setObjectIDs($this->parameters['postIDs']);
		$postList->readObjects();
		
		/** @var Thread $thread */
		$thread = $this->parameters['thread'];
		
		//
		// step 1) clone posts
		//
		
		$attachments = [];
		$newPosts = [];
		$newPostIDs = [];
		$polls = [];
		foreach ($postList as $post) {
			$newPost = PostEditor::create([
				'threadID' => $thread->threadID,
				'userID' => $post->userID ?: null,
				'username' => $post->username,
				'subject' => $post->subject,
				'message' => $post->message,
				'time' => $post->time,
				'isDeleted' => $post->isDeleted,
				'isDisabled' => $post->isDisabled,
				'isClosed' => $post->isClosed,
				'editorID' => $post->editorID ?: null,
				'editor' => $post->editor,
				'lastEditTime' => $post->lastEditTime,
				'editCount' => $post->editCount,
				'editReason' => $post->editReason,
				'attachments' => $post->attachments,
				'enableHtml' => $post->enableHtml,
				'ipAddress' => $post->ipAddress,
				'cumulativeLikes' => $post->cumulativeLikes,
				'deleteTime' => $post->deleteTime
			]);
			
			$newPostIDs[$post->postID] = $newPost->postID;
			$newPosts[$newPost->postID] = $newPost;
			
			if ($post->attachments) {
				$attachments[$post->postID] = $post->message;
			}
			
			if ($post->pollID) {
				$polls[] = $post->postID;
			}
		}
		
		$postData = [];
		
		//
		// step 2) copy attachments
		//
		
		$attachmentMapping = [];
		foreach ($attachments as $postID => $message) {
			$attachmentAction = new AttachmentAction([], 'copy', [
				'sourceObjectID' => $postID,
				'sourceObjectType' => 'com.woltlab.wbb.post',
				'targetObjectID' => $newPostIDs[$postID],
				'targetObjectType' => 'com.woltlab.wbb.post'
			]);
			$returnValues = $attachmentAction->executeAction();
			
			foreach ($returnValues['returnValues']['attachmentIDs'] as $oldID => $newID) {
				if (!isset($attachmentMapping[$newPostIDs[$postID]])) $attachmentMapping[$newPostIDs[$postID]] = [];
				
				$attachmentMapping[$newPostIDs[$postID]][$oldID] = $newID;
			}
		}
		
		//
		// step 3) copy polls
		//
		
		foreach ($polls as $postID) {
			$pollAction = new PollAction([], 'copy', [
				'sourceObjectID' => $postID,
				'sourceObjectType' => 'com.woltlab.wbb.post',
				'targetObjectID' => $newPostIDs[$postID],
				'targetObjectType' => 'com.woltlab.wbb.post'
			]);
			$returnValues = $pollAction->executeAction();
			
			if ($returnValues['returnValues']['pollID'] !== null) {
				// set new poll id
				if (!isset($postData[$newPostIDs[$postID]])) {
					$postData[$newPostIDs[$postID]] = [];
				}
				
				$postData[$newPostIDs[$postID]]['pollID'] = $returnValues['returnValues']['pollID'];
			}
		}
		
		//
		// step 4) copy likes
		//
		
		foreach ($newPostIDs as $oldID => $newID) {
			$likeAction = new LikeAction([], 'copy', [
				'sourceObjectID' => $oldID,
				'sourceObjectType' => 'com.woltlab.wbb.likeablePost',
				'targetObjectID' => $newID,
				'targetObjectType' => 'com.woltlab.wbb.likeablePost'
			]);
			$likeAction->executeAction();
		}
		
		//
		// step 5) update embedded objects
		//
		
		$htmlInputProcessor = new HtmlInputProcessor();
		foreach ($newPosts as $post) {
			// fix embedded attachments
			if (isset($attachmentMapping[$post->postID])) {
				$htmlInputProcessor->processIntermediate((isset($postData[$post->postID]['message']) ? $postData[$post->postID]['message'] : $post->message));
				
				$elements = $htmlInputProcessor->getHtmlInputNodeProcessor()->getDocument()->getElementsByTagName('woltlab-metacode');
				/** @var \DOMElement $element */
				foreach ($elements as $element) {
					if ($element->getAttribute('data-name') === 'attach') {
						$attributes = $htmlInputProcessor->getHtmlInputNodeProcessor()->parseAttributes($element->getAttribute('data-attributes'));
						if (isset($attributes[0])) {
							$attachmentID = $attributes[0];
							if (isset($attachmentMapping[$post->postID][$attachmentID])) {
								$attributes[0] = $attachmentMapping[$post->postID][$attachmentID];
								$element->setAttribute('data-attributes', base64_encode(json_encode($attributes)));
							}
						}
					}
				}
				
				$postData[$post->postID]['message'] = $htmlInputProcessor->getHtml();
			}
			
			$htmlInputProcessor->process(
				isset($postData[$post->postID]['message']) ? $postData[$post->postID]['message'] : $post->message,
				'com.woltlab.wbb.post',
				$post->postID
			);
			
			if (MessageEmbeddedObjectManager::getInstance()->registerObjects($htmlInputProcessor)) {
				$postData[$post->postID]['hasEmbeddedObjects'] = 1;
			}
		}
		
		//
		// step 6) create moderation queue entry for disabled posts
		//
		
		foreach ($newPosts as $post) {
			if ($post->isDisabled) {
				ModerationQueueActivationManager::getInstance()->addModeratedContent('com.woltlab.wbb.post', $post->postID);
			}
		}
		
		// 
		// step 7) copy thread form options
		//
		foreach ($newPostIDs as $oldID => $newID) {
			(new ThreadFormOptionAction([], 'copy', [
				'oldPostID' => $oldID,
				'newPostID' => $newID
			]))->executeAction();
		}
		
		//
		// step 8) finalize posts
		//
		
		if (!empty($postData)) {
			WCF::getDB()->beginTransaction();
			foreach ($postData as $postID => $data) {
				$postEditor = new PostEditor($newPosts[$postID]);
				$postEditor->update($data);
			}
			WCF::getDB()->commitTransaction();
		}
		
		//
		// step 9) update users and activity points
		//
		
		$activityPoints = [];
		$userToPosts = [];
		
		foreach ($newPosts as $post) {
			if ($post->userID && $thread->getBoard()->countUserPosts) {
				if (!$thread->isDisabled) {
					// add activity points (not recent activity events!)
					if (!isset($activityPoints[$post->userID])) {
						$activityPoints[$post->userID] = 0;
					}
					$activityPoints[$post->userID]++;
				}
				
				// count posts
				if (!$post->isDisabled) {
					if (!isset($userToPosts[$post->userID])) {
						$userToPosts[$post->userID] = 0;
					}
					
					$userToPosts[$post->userID]++;
				}
			}
		}
		
		if (!empty($activityPoints)) {
			UserActivityPointHandler::getInstance()->fireEvents('com.woltlab.wbb.activityPointEvent.post', $activityPoints);
		}
		
		if (!empty($userToPosts)) {
			PostEditor::updatePostCounter($userToPosts);
		}
		
		// update search index
		PostEditor::addPostIDsToSearchIndex(array_keys($newPosts));
	}
	
	/**
	 * @inheritDoc
	 */
	public function getHtmlInputProcessor($message = null, $objectID = 0) {
		if ($message === null) {
			return $this->htmlInputProcessor;
		}
		
		$this->htmlInputProcessor = new HtmlInputProcessor();
		$this->htmlInputProcessor->process($message, 'com.woltlab.wbb.post', $objectID);
		
		return $this->htmlInputProcessor;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getAllowedQuickReplyParameters() {
		return ['poll', 'subscribeThread'];
	}
	
	/**
	 * Adds post data.
	 * 
	 * @param	Post	$post
	 * @param	string	$key
	 * @param	mixed	$value
	 */
	protected function addPostData(Post $post, $key, $value) {
		if (!isset($this->postData[$post->postID])) {
			$this->postData[$post->postID] = [];
		}
		
		$this->postData[$post->postID][$key] = $value;
	}
	
	/**
	 * Adds thread data.
	 * 
	 * @param	integer		$threadID
	 * @param	string		$key
	 * @param	mixed		$value
	 */
	protected function addThreadData($threadID, $key, $value) {
		if (!isset($this->threadData[$threadID])) {
			$this->threadData[$threadID] = [];
		}
		
		$this->threadData[$threadID][$key] = $value;
	}
	
	/**
	 * Returns post data.
	 * 
	 * @return	mixed[][]
	 */
	protected function getPostData() {
		return [
			'postData' => $this->postData,
			'threadData' => $this->threadData
		];
	}
	
	/**
	 * Removes moderated content entries for given post ids.
	 * 
	 * @param	integer[]		$postIDs
	 */
	protected function removeModeratedContent(array $postIDs) {
		ModerationQueueActivationManager::getInstance()->removeModeratedContent('com.woltlab.wbb.post', $postIDs);
	}
	
	/**
	 * Unmarks posts.
	 * 
	 * @param	integer[]	$objectIDs
	 */
	protected function unmarkItems(array $objectIDs = []) {
		if (empty($objectIDs)) {
			foreach ($this->getObjects() as $post) {
				$objectIDs[] = $post->postID;
			}
		}
		
		if (!empty($objectIDs)) {
			ClipboardHandler::getInstance()->unmark($objectIDs, ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.post'));
		}
	}
	
	/**
	 * Updates last post for given board ids.
	 * 
	 * @param	integer[]	$boardIDs
	 */
	protected function updateLastPost(array $boardIDs) {
		if (empty($boardIDs)) {
			return;
		}
		
		$boardIDs = array_unique($boardIDs);
		foreach ($boardIDs as $boardID) {
			$boardEditor = new BoardEditor(new Board($boardID));
			$boardEditor->updateLastPost();
		}
		
		BoardEditor::resetDataCache();
	}
	
	/**
	 * Returns detailed post statistics for the threads with the given ids.
	 * 
	 * @param	integer[]	$threadIDs
	 * @return	integer[]
	 */
	protected function getDetailedPostStats(array $threadIDs) {
		if (empty($threadIDs)) {
			return [];
		}
		
		$conditionBuilder = new PreparedStatementConditionBuilder();
		$conditionBuilder->add('threadID IN (?)', [$threadIDs]);
		
		$sql = "SELECT		threadID,
					COUNT(*) AS posts,
					SUM(CASE WHEN isDeleted = 0 AND isDisabled = 0 THEN 1 ELSE 0 END) AS visiblePosts,
					SUM(CASE WHEN isDeleted = 1 THEN 1 ELSE 0 END) AS deletedPosts,
					SUM(CASE WHEN isDisabled = 1 THEN 1 ELSE 0 END) AS disabledPosts
			FROM		wbb".WCF_N."_post
			".$conditionBuilder."
			GROUP BY	threadID";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditionBuilder->getParameters());
		
		$postStats = [];
		while ($row = $statement->fetchArray()) {
			$postStats[$row['threadID']] = [
				'deletedPosts' => $row['deletedPosts'],
				'disabledPosts' => $row['disabledPosts'],
				'posts' => $row['posts'],
				'visiblePosts' => $row['visiblePosts']
			];
		}
		
		return $postStats;
	}
	
	/**
	 * Returns `true` if the relevant action is executed in bulk processing mode and returns
	 * `false` otherwise.
	 * 
	 * In bulk processing mode, updates that can be achieved using rebuild data workers should
	 * not be executed. Additionally, modification log entries should not be created, post data
	 * should not be registered, and no clipboard items should be unmarked.
	 * 
	 * @return	boolean
	 */
	protected function isBulkProcessing() {
		return isset($this->parameters['isBulkProcessing']) && $this->parameters['isBulkProcessing'];
	}
}
