<?php
namespace wbb\data\post;
use wbb\system\WBBCore;
use wcf\data\edit\history\entry\EditHistoryEntry;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\DatabaseObject;
use wcf\data\DatabaseObjectDecorator;
use wcf\data\TUserContent;
use wcf\system\edit\IHistorySavingObject;
use wcf\system\WCF;

/**
 * History Saving Posts are posts that can be reverted to an older version.
 * 
 * @author	Tim Duesterhus
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Post
 * 
 * @method	Post	getDecoratedObject()
 * @mixin	Post
 */
class HistorySavingPost extends DatabaseObjectDecorator implements IHistorySavingObject {
	use TUserContent;
	
	/**
	 * @inheritDoc
	 */
	protected static $baseClass = Post::class;
	
	/**
	 * userID of the last editor
	 * @var	integer
	 */
	public $userID = 0;
	
	/**
	 * username of the last editor
	 * @var	string
	 */
	public $username = '';
	
	/**
	 * last edit time
	 * @var	integer
	 */
	public $time = 0;
	
	/**
	 * reason for the last edit
	 * @var	string
	 */
	public $reason = '';
	
	/**
	 * @inheritDoc
	 */
	public function __construct(DatabaseObject $object) {
		parent::__construct($object);
		$objectTypeID = ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.post');
		
		$sql = "SELECT	*
			FROM	wcf".WCF_N."_modification_log
			WHERE		objectTypeID = ?
				AND	objectID = ?
				AND	action = ?
			ORDER BY time DESC, logID DESC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute([$objectTypeID, $this->getDecoratedObject()->postID, 'edit']);
		$row = $statement->fetchArray();
		
		if ($row) {
			$this->userID = $row['userID'];
			$this->username = $row['username'];
			$this->time = $row['time'];
			$additionalData = @unserialize($row['additionalData']);
			if (isset($additionalData['reason'])) {
				$this->reason = $additionalData['reason'];
			}
			else {
				$this->reason = '';
			}
		}
		else {
			$this->userID = $this->getDecoratedObject()->getUserID();
			$this->username = $this->getDecoratedObject()->getUsername();
			$this->time = $this->getDecoratedObject()->getTime();
			$this->reason = '';
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		return $this->getDecoratedObject()->getTitle();
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return $this->getDecoratedObject()->getLink();
	}
	
	/**
	 * @inheritDoc
	 */
	public function revertVersion(EditHistoryEntry $edit) {
		$postAction = new PostAction([$this->getDecoratedObject()], 'update', [
			'isEdit' => true,
			'data' => [
				'message' => $edit->message,
				'editReason' => WCF::getLanguage()->getDynamicVariable('wcf.edit.reverted', ['edit' => $edit])
			]
		]);
		$postAction->executeAction();
	}
	
	/**
	 * @inheritDoc
	 */
	public function setLocation() {
		WBBCore::getInstance()->setLocation($this->getThread()->getBoard()->getParentBoards(), $this->getThread()->getBoard(), $this->getThread());
	}
	
	/**
	 * @inheritDoc
	 */
	public function getEditReason() {
		return $this->reason;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getMessage() {
		return $this->getDecoratedObject()->getMessage();
	}
}
