<?php
namespace wbb\data\modification\log;
use wcf\data\label\Label;
use wcf\data\modification\log\ModificationLog;
use wcf\data\modification\log\ModificationLogList;
use wcf\data\object\type\ObjectTypeCache;
use wcf\system\cache\runtime\UserProfileRuntimeCache;
use wcf\system\label\LabelHandler;
use wcf\system\WCF;
use wcf\util\JSON;

/**
 * Represents a list of modification logs for thread log page.
 *
 * @author	Joshua Ruesweg
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Modification\Log
 *
 * @method	ViewableThreadModificationLog		current()
 * @method	ViewableThreadModificationLog[]		getObjects()
 * @method	ViewableThreadModificationLog|null	search($objectID)
 * @property	ViewableThreadModificationLog[]		$objects
 */
class ViewableThreadModificationLogList extends ModificationLogList {
	/**
	 * Viewable actions.
	 * 
	 * @var string[]
	 */
	public static $supportedActions = ['close', 'trash', 'enable', 'move', 'setLabel', 'changeTopic'];
	
	/**
	 * @inheritDoc
	 */
	public function __construct($threadID) {
		parent::__construct();
		
		// set conditions
		$this->getConditionBuilder()->add('modification_log.objectTypeID = ?', [ObjectTypeCache::getInstance()->getObjectTypeIDByName('com.woltlab.wcf.modifiableContent', 'com.woltlab.wbb.thread')]);
		$this->getConditionBuilder()->add('modification_log.objectID = ?', [$threadID]);
		$this->getConditionBuilder()->add('modification_log.hidden = ?', [0]);
		$this->getConditionBuilder()->add('modification_log.action IN (?)', [self::$supportedActions]);
	}
	
	/** @noinspection PhpMissingParentCallCommonInspection */
	/**
	 * @inheritDoc
	 */
	public function readObjects() {
		$sql = "SELECT	modification_log.*
			FROM	wcf".WCF_N."_modification_log modification_log
			".$this->getConditionBuilder()."
			".(!empty($this->sqlOrderBy) ? "ORDER BY ".$this->sqlOrderBy : '');
		$statement = WCF::getDB()->prepareStatement($sql, $this->sqlLimit, $this->sqlOffset);
		$statement->execute($this->getConditionBuilder()->getParameters());
		$this->objects = $statement->fetchObjects(($this->objectClassName ?: $this->className));
		
		// use table index as array index
		$objects = $userIDs = [];
		foreach ($this->objects as $object) {
			$objectID = $object->{$this->getDatabaseTableIndexName()};
			$objects[$objectID] = $object;
			
			$this->indexToObject[] = $objectID;
			
			if ($object->userID) {
				$userIDs[] = $object->userID;
			}
		}
		$this->objectIDs = $this->indexToObject;
		$this->objects = $objects;
		
		if (!empty($userIDs)) {
			UserProfileRuntimeCache::getInstance()->cacheObjectIDs($userIDs);
		}
		
		foreach ($this->objects as $key => &$object) {
			if ($object->action == 'setLabel') {
				// check permissions
				if ($object->label instanceof Label) {
					$labelID = $object->label->labelID;
				}
				else if ($object->oldLabel instanceof Label) {
					$labelID = $object->oldLabel->labelID;
				}
				else {
					$labelID = false;
				}
				
				if (!$labelID || !LabelHandler::getInstance()->validateCanView([$labelID])[$labelID]) {
					unset($this->objects[$key]);
					continue;
				}
			}
			
			$object = new ViewableThreadModificationLog($object);
		}
		unset($object);
	}
	
	/**
	 * Returns all log entries created before given point of time. Applicable entries
	 * will be returned and removed from collection.
	 *
	 * @param	integer		$time
	 * @return	ViewableThreadModificationLog[]
	 */
	public function getEntriesUntil($time) {
		$entries = [];
		foreach ($this->objects as $index => $entry) {
			if ($entry->time < $time) {
				$entries[] = $entry;
				unset($this->objects[$index]);
			}
		}
		
		if (!empty($entries)) {
			$this->indexToObject = array_keys($this->objects);
		}
		
		return $entries;
	}
	
	/**
	 * Returns all log entries created before given point of time. Applicable entries
	 * will be summarized and removed from collection.
	 *
	 * @param	integer		$time
	 * @return	ViewableThreadModificationLog[]
	 */
	public function getSummarizedEntriesUntil($time) {
		$entries = $this->getEntriesUntil($time);
		
		$entriesSortedByAction = [];
		
		foreach ($entries as $entry) {
			if (!isset($entriesSortedByAction[$entry->action])) {
				$entriesSortedByAction[$entry->action] = [];
			}
			
			$entriesSortedByAction[$entry->action][] = $entry;
		}
		
		$summarizedEntries = [];
		foreach ($entriesSortedByAction as $action => $entries) {
			switch ($action) {
				case 'setLabel':
					$summarizedEntries = array_merge($this->summarizeSetLabelAction($entries), $summarizedEntries);
					break;
					
				case 'changeTopic':
					$summarizedEntries = array_merge($this->summarizeChangeTopicAction($entries), $summarizedEntries);
					break;
					
				default:
					$summarizedEntries = array_merge($entries, $summarizedEntries);
			}
		}
		
		// sort items by time
		ModificationLog::sort($summarizedEntries, 'time');
		
		return $summarizedEntries;
	}
	
	/**
	 * Summarizes log entries with the property action = 'changeTopic'.
	 *
	 * @param	ViewableThreadModificationLog[]		$entries
	 * @return	ViewableThreadModificationLog[]
	 */
	private function summarizeChangeTopicAction(array $entries) {
		if (count($entries) == 1) {
			return $entries;
		}
		
		$dividedEntries = $this->dividedUpEntries($entries);
		
		$summarizedEntries = [];
		foreach ($dividedEntries as $sortedEntries) {
			if (count($sortedEntries) == 1) {
				$summarizedEntries[] = array_pop($sortedEntries);
				continue;
			}
			
			$lastChange = end($sortedEntries);
			$reversedSortedEntries = array_reverse($sortedEntries);
			$firstChange = end($reversedSortedEntries);
			
			if ($firstChange->oldTopic != $lastChange->newTopic) {
				$summarizedEntries[] = new ViewableThreadModificationLog(new ModificationLog(null, [
					'logID' => $lastChange->logID,
					'objectTypeID' => $lastChange->objectTypeID,
					'objectID' => $lastChange->objectID,
					'parentObjectID' => $lastChange->parentObjectID,
					'userID' => $lastChange->userID,
					'username' => $lastChange->username,
					'time' => $lastChange->time,
					'action' => $lastChange->action,
					'additionalData' => serialize([
						'oldTopic' => $firstChange->oldTopic,
						'newTopic' => $lastChange->newTopic
					]),
					
					// special fields
					'isSummarized' => true,
					'logIDs' => array_map(function ($entry) {
						return $entry->logID;
					}, $sortedEntries),
					'encodedLogIDs' => JSON::encode(array_map(function ($entry) {
						return $entry->logID;
					}, $sortedEntries))
				]));
			}
		}
		
		return $summarizedEntries;
	}
	
	/**
	 * Summarizes log entries with the property action = 'setLabel'.
	 *
	 * @param	ViewableThreadModificationLog[]		$entries
	 * @return	ViewableThreadModificationLog[]
	 */
	private function summarizeSetLabelAction(array $entries) {
		if (count($entries) == 1) {
			return $entries;
		}
		
		$dividedEntries = $this->dividedUpEntries($entries);
		
		// divide up by labelID
		$dividedByLabelEntries = $tmpDividedEntries = [];
		foreach ($dividedEntries as $entries) {
			if (count($entries) == 1) {
				$dividedByLabelEntries[] = $entries;
				continue;
			}
			
			$tmpDividedEntries = [];
			foreach ($entries as $entry) {
				$labelGroupID = $this->getLabelGroupIDForModificationLogEntry($entry);
				
				if (!isset($tmpDividedEntries[$labelGroupID])) {
					$tmpDividedEntries[$labelGroupID] = [];
				}
				
				$tmpDividedEntries[$labelGroupID][] = $entry;
			}
			
			foreach ($tmpDividedEntries as $group) {
				$dividedByLabelEntries[] = $group;
			}
		}
		
		$summarizedEntries = [];
		foreach ($dividedByLabelEntries as $sortedEntries) {
			if (count($sortedEntries) == 1) {
				$summarizedEntries[] = array_pop($sortedEntries);
				continue;
			}
			
			$lastChange = end($sortedEntries);
			$reversedSortedEntries = array_reverse($sortedEntries);
			$firstChange = end($reversedSortedEntries);
			
			if ((($firstChange->oldLabel instanceof Label) xor ($lastChange->label instanceof Label)) || (($firstChange->oldLabel instanceof Label) && ($lastChange->label instanceof Label) && $firstChange->oldLabel->labelID != $lastChange->label->labelID)) {
				$summarizedEntries[] = new ViewableThreadModificationLog(new ModificationLog(null, [
					'logID' => $lastChange->logID,
					'objectTypeID' => $lastChange->objectTypeID,
					'objectID' => $lastChange->objectID,
					'parentObjectID' => $lastChange->parentObjectID,
					'userID' => $lastChange->userID,
					'username' => $lastChange->username,
					'time' => $lastChange->time,
					'action' => $lastChange->action,
					'additionalData' => serialize([
						'oldLabel' => $firstChange->oldLabel instanceof Label ? $firstChange->oldLabel : null,
						'label' => $lastChange->label instanceof Label ? $lastChange->label : null
					]),
					
					// special fields
					'isSummarized' => true,
					'logIDs' => array_map(function ($entry) {
						return $entry->logID;
					}, $sortedEntries),
					'encodedLogIDs' => JSON::encode(array_map(function ($entry) {
						return $entry->logID;
					}, $sortedEntries))
				]));
			}
		}
		
		return $summarizedEntries;
	}
	
	/**
	 * Returns the labelGroupID for a ModificationLogEntry with the property action = 'setLabel'.
	 * 
	 * @param 	ViewableThreadModificationLog 	$log
	 * @return 	integer
	 */
	private function getLabelGroupIDForModificationLogEntry(ViewableThreadModificationLog $log) {
		if ($log->action != 'setLabel') {
			throw new \BadMethodCallException("The parameter \$log must be an instance of a ViewableThreadModificationLog with an property action = 'setLabel'.");
		}
		
		if ($log->label instanceof Label) {
			return $log->label->groupID;
		}
		else if ($log->oldLabel instanceof Label) {
			return $log->oldLabel->groupID;
		}
		else {
			throw new \LogicException('Unreachable');
		}
	}
	
	/**
	 * Divide up entries by userID and time.
	 *
	 * @param	ViewableThreadModificationLog[]		$entries
	 * @return	ViewableThreadModificationLog[][]
	 */
	private function dividedUpEntries(array $entries) {
		$dividedEntries = $tmpDividedEntries = [];
		$currentUserID = $currentTime = null;
		foreach ($entries as $entry) {
			if ($currentUserID === null) {
				$currentUserID = $entry->userID;
				$currentTime = $entry->time;
			}
			
			if ($currentUserID == $entry->userID && $entry->time > ($currentTime - 3600)) {
				$tmpDividedEntries[] = $entry;
			}
			else {
				$dividedEntries[] = $tmpDividedEntries;
				$tmpDividedEntries = [$entry];
			}
			
			$currentUserID = $entry->userID;
			$currentTime = $entry->time;
		}
		
		$dividedEntries[] = $tmpDividedEntries;
		
		return $dividedEntries;
	}
}
