<?php
namespace wbb\data\board;

/**
 * Represents a board node tree.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Board
 */
class BoardNodeTree {
	/**
	 * parent board id
	 * @var	integer
	 */
	public $parentID;
	
	/**
	 * start depth
	 * @var	integer
	 */
	public $startDepth = 0;
	
	/**
	 * list of board ids
	 * @var	integer[]
	 */
	public $filter = [];
	
	/**
	 * root node
	 * @var	BoardNode
	 */
	public $node;
	
	/**
	 * board node class name
	 * @var	string
	 */
	public $nodeClassname = BoardNode::class;
	
	/**
	 * Creates a new BoardNodeTree object.
	 * 
	 * @param	integer		$parentID
	 * @param	integer		$startDepth
	 * @param	integer[]	$filter
	 */
	public function __construct($parentID = null, $startDepth = 0, array $filter = []) {
		$this->parentID = $parentID;
		$this->startDepth = $startDepth;
		$this->filter = $filter;
	}
	
	/**
	 * Reads the node tree.
	 */
	public function readNodeTree() {
		// generate node tree
		$this->node = new BoardNode(null, null, $this->startDepth);
		$this->node->setChildren($this->generateNodeTree($this->parentID, $this->node, $this->filter));
	}
	
	/**
	 * Generates the node tree recursively
	 * 
	 * @param	integer		$parentID
	 * @param	BoardNode	$parentNode
	 * @param	integer[]	$filter
	 * @return	BoardNode[]
	 */
	protected function generateNodeTree($parentID, BoardNode $parentNode = null, array $filter = []) {
		$nodes = [];
		
		// apply filter
		$boardIDs = BoardCache::getInstance()->getChildIDs($parentID);
		if (count($filter)) $boardIDs = array_intersect($boardIDs, $filter);
		
		foreach ($boardIDs as $boardID) {
			$board = BoardCache::getInstance()->getBoard($boardID);
			if ($this->isVisible($board)) {
				/** @var BoardNode $node */
				$node = new $this->nodeClassname($parentNode, $board, $parentNode !== null ? $parentNode->getDepth() + 1 : 0);
				$nodes[] = $node;
				
				// get children
				$node->setChildren($this->generateNodeTree($boardID, $node));
			}
		}
		
		return $nodes;
	}
	
	/**
	 * Returns the board node tree.
	 * 
	 * @return	BoardNode[]
	 */
	public function getNodeTree() {
		return $this->node->getChildren();
	}
	
	/**
	 * Returns true if the given board is visible.
	 * 
	 * @param	Board	$board
	 * @return	boolean
	 */
	protected function isVisible(/** @noinspection PhpUnusedParameterInspection */Board $board) {
		return true;
	}
}
