<?php
namespace wbb\data\board;
use wbb\data\thread\form\option\ThreadFormCache;
use wbb\data\thread\Thread;
use wcf\data\DatabaseObjectEditor;
use wcf\data\IEditableCachedObject;
use wbb\system\cache\builder\BoardCacheBuilder;
use wbb\system\cache\builder\BoardDataCacheBuilder;
use wbb\system\cache\builder\BoardModeratorPermissionCacheBuilder;
use wbb\system\cache\builder\BoardPermissionCacheBuilder;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;

/**
 * Provides functions to edit boards.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Board
 * 
 * @method static	Board	create(array $parameters = [])
 * @method		Board	getDecoratedObject()
 * @mixin		Board
 */
class BoardEditor extends DatabaseObjectEditor implements IEditableCachedObject {
	/**
	 * @inheritDoc
	 */
	protected static $baseClass = Board::class;
	
	/**
	 * Sets the last post of this board.
	 * 
	 * @param	Thread		$thread
	 */
	public function setLastPost(Thread $thread) {
		if ($thread->lastPostTime != TIME_NOW) {
			if (($lastPostThread = BoardCache::getInstance()->getLastPost($this->boardID, $thread->languageID)) !== null) {
				if ($thread->lastPostTime < $lastPostThread->lastPostTime) return;
			}
		}
		
		// delete old entry
		if ($thread->languageID === null) {
			$sql = "DELETE FROM	wbb".WCF_N."_board_last_post
				WHERE		boardID = ?
						AND languageID IS NULL";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->boardID]);
		}
		else {
			$sql = "DELETE FROM	wbb".WCF_N."_board_last_post
				WHERE		boardID = ?
						AND languageID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$this->boardID, $thread->languageID]);
		}
		
		// save new one
		$sql = "INSERT IGNORE INTO	wbb".WCF_N."_board_last_post
						(boardID, languageID, threadID)
			VALUES		        (?, ?, ?)";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->boardID, $thread->languageID, $thread->threadID]);
	}
	
	/**
	 * Adds the board to a specific position in the board tree.
	 * 
	 * @param	integer		$parentID
	 * @param	integer		$position
	 */
	public function setPosition($parentID, $position = 0) {
		if ($position) {
			// shift boards
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("boardID <> ?", [$this->boardID]);
			if ($parentID === null) {
				$conditions->add("parentID IS NULL");
			}
			else {
				$conditions->add("parentID = ?", [$parentID]);
			}
			$conditions->add("position >= ?", [$position]);
			
			$sql = "UPDATE	wbb".WCF_N."_board
				SET	position = position + 1
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
		}
		else {
			// get next free position
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("boardID <> ?", [$this->boardID]);
			if ($parentID === null) {
				$conditions->add("parentID IS NULL");
			}
			else {
				$conditions->add("parentID = ?", [$parentID]);
			}
			
			$sql = "SELECT	MAX(position) AS position
				FROM	wbb".WCF_N."_board
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			$row = $statement->fetchArray();
			
			if (!$row) $position = 1;
			else $position = $row['position'] + 1;
		}
		
		// save position
		$this->update([
			'position' => $position
		]);
	}
	
	/**
	 * @inheritDoc
	 */
	public static function resetCache() {
		static::resetDataCache();
		static::resetPermissionCache();
		
		BoardCacheBuilder::getInstance()->reset();
		ThreadFormCache::getInstance()->resetCache();
	}
	
	/**
	 * Resets the board permission cache.
	 */
	public static function resetPermissionCache() {
		BoardPermissionCacheBuilder::getInstance()->reset();
		BoardModeratorPermissionCacheBuilder::getInstance()->reset();
	}
	
	/**
	 * Resets the board data cache.
	 */
	public static function resetDataCache() {
		BoardDataCacheBuilder::getInstance()->reset();
	}
	
	/**
	 * Updates last post.
	 */
	public function updateLastPost() {
		// reset current last posts
		$sql = "DELETE FROM	wbb".WCF_N."_board_last_post
			WHERE		boardID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->boardID]);
		
		// get available language ids
		$contentLanguages = LanguageFactory::getInstance()->getContentLanguages();
		if (empty($contentLanguages)) {
			$this->setLastPostByLanguage(null);
		}
		else {
			foreach ($contentLanguages as $contentLanguage) {
				$this->setLastPostByLanguage($contentLanguage->languageID);
			}
		}
	}
	
	/**
	 * Rebuilds statistics for current board.
	 */
	public function rebuildStats() {
		$sql = "UPDATE	wbb".WCF_N."_board board
			SET	board.threads = (
					SELECT	COUNT(*)
					FROM	wbb".WCF_N."_thread
					WHERE	boardID = board.boardID
						AND isDeleted = 0
						AND isDisabled = 0
				),
				board.posts = COALESCE((
					SELECT	SUM(replies + 1)
					FROM	wbb".WCF_N."_thread
					WHERE	boardID = board.boardID
						AND isDeleted = 0
						AND isDisabled = 0
				), 0),
				board.time = COALESCE((
					SELECT		MIN(time)
					FROM		wbb".WCF_N."_thread
					WHERE		boardID = board.boardID
				), 0)
			WHERE	boardID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([$this->boardID]);
	}
	
	/**
	 * Sets last post by language id.
	 * 
	 * @param	integer		$languageID
	 */
	protected function setLastPostByLanguage($languageID) {
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("boardID = ?", [$this->boardID]);
		$conditions->add("isDeleted = ?", [0]);
		$conditions->add("isDisabled = ?", [0]);
		$conditions->add("movedThreadID IS NULL");
		if ($languageID !== null) $conditions->add("languageID = ?", [$languageID]);
		
		$sql = "SELECT		threadID
			FROM		wbb".WCF_N."_thread
			".$conditions."
			ORDER BY	lastPostTime DESC, lastPostID DESC";
		$statement = WCF::getDB()->prepareStatement($sql, 1);
		$statement->execute($conditions->getParameters());
		$row = $statement->fetchArray();
		
		if ($row !== false) {
			$sql = "INSERT INTO	wbb".WCF_N."_board_last_post
						(boardID, threadID, languageID)
				VALUES		(?, ?, ?)";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([
				$this->boardID,
				$row['threadID'],
				$languageID
			]);
		}
	}
}
