<?php
namespace wbb\data\board;
use wbb\data\thread\ViewableThread;
use wbb\data\thread\ViewableThreadList;
use wbb\system\cache\builder\BoardCacheBuilder;
use wbb\system\cache\builder\BoardDataCacheBuilder;
use wcf\data\page\PageCache;
use wcf\data\user\online\UserOnline;
use wcf\data\user\online\UsersOnlineList;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\user\collapsible\content\UserCollapsibleContentHandler;
use wcf\system\user\UserProfileHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\SingletonFactory;
use wcf\system\WCF;

/**
 * Manages the board cache.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Board
 */
class BoardCache extends SingletonFactory {
	/**
	 * cached board structure
	 * @var	array
	 */
	protected $cachedBoardStructure = [];
	
	/**
	 * cached boards
	 * @var	Board[]
	 */
	protected $cachedBoards = [];
	
	/**
	 * cached label groups
	 * @var	integer[][]
	 */
	protected $cachedLabelGroups = [];
	
	/**
	 * list of cache moderator user ids per board id
	 * @var	mixed[][]
	 */
	protected $cachedModerators = [];
	
	/**
	 * cached board counter
	 * @var	array
	 */
	protected $counts = [];
	
	/**
	 * cached last posts (thread ids)
	 * @var	integer[]
	 */
	protected $lastPostThreadIDs = [];
	
	/**
	 * cached last posts
	 * @var	ViewableThread[]
	 */
	protected $lastPosts;
	
	/**
	 * list of closed boards
	 * @var	integer[]
	 */
	protected $closedBoardIDs;
	
	/**
	 * list of ignored boards
	 * @var	integer[]
	 */
	protected $ignoredBoardIDs;
	
	/**
	 * number of unread threads
	 * @var	integer[]
	 */
	protected $unreadThreads;
	
	/**
	 * users online
	 * @var	UserOnline[][]
	 */
	protected $usersOnline;
	
	/**
	 * @inheritDoc
	 */
	protected function init() {
		// get board cache
		$this->cachedBoardStructure = BoardCacheBuilder::getInstance()->getData([], 'boardStructure');
		$this->cachedBoards = BoardCacheBuilder::getInstance()->getData([], 'boards');
		$this->cachedLabelGroups = BoardCacheBuilder::getInstance()->getData([], 'labelGroups');
		$this->cachedModerators = BoardCacheBuilder::getInstance()->getData([], 'moderators');
		
		// get board data cache
		$this->counts = BoardDataCacheBuilder::getInstance()->getData([], 'counts');
		$this->lastPostThreadIDs = BoardDataCacheBuilder::getInstance()->getData([], 'lastPostThreadIDs');
	}
	
	/**
	 * Calculates the number of unread threads.
	 */
	protected function initUnreadThreads() {
		$this->unreadThreads = [];
		if (WCF::getUser()->userID) {
			$conditionBuilder = new PreparedStatementConditionBuilder();
			$conditionBuilder->add('thread.lastPostTime > ?', [VisitTracker::getInstance()->getVisitTime('com.woltlab.wbb.thread')]);
			$conditionBuilder->add('thread.isDeleted = 0 AND thread.isDisabled = 0 AND thread.movedThreadID IS NULL');
			$conditionBuilder->add('(thread.lastPostTime > tracked_thread_visit.visitTime OR tracked_thread_visit.visitTime IS NULL)');
			$conditionBuilder->add('(thread.lastPostTime > tracked_board_visit.visitTime OR tracked_board_visit.visitTime IS NULL)');
			
			// apply language filter
			if (LanguageFactory::getInstance()->multilingualismEnabled() && count(WCF::getUser()->getLanguageIDs())) {
				$conditionBuilder->add('(thread.languageID IN (?) OR thread.languageID IS NULL)', [WCF::getUser()->getLanguageIDs()]);
			}
			
			// get board ids
			$boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs());
			$privateBoardIDs = Board::filterBoardIDs(Board::getPrivateBoardIDs());
			
			if (empty($boardIDs) && empty($privateBoardIDs)) {
				return;
			}
			else {
				if (empty($privateBoardIDs)) {
					$conditionBuilder->add("thread.boardID IN (?)", [$boardIDs]);
				}
				else if (empty($boardIDs)) {
					$conditionBuilder->add("(thread.boardID IN (?) AND thread.userID = ?)", [$privateBoardIDs, WCF::getUser()->userID]);
				}
				else {
					$conditionBuilder->add("(thread.boardID IN (?) OR (thread.boardID IN (?) AND thread.userID = ?))", [$boardIDs, $privateBoardIDs, WCF::getUser()->userID]);
				}
			}
			
			if (!empty(UserProfileHandler::getInstance()->getIgnoredUsers())) {
				$conditionBuilder->add("(thread.userID IS NULL OR thread.userID NOT IN (?))", [UserProfileHandler::getInstance()->getIgnoredUsers()]);
			}
			
			$sql = "SELECT		COUNT(*) AS count, thread.boardID
				FROM		wbb".WCF_N."_thread thread
				LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_thread_visit
				ON		(tracked_thread_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.thread')." AND tracked_thread_visit.objectID = thread.threadID AND tracked_thread_visit.userID = ".WCF::getUser()->userID.")
				LEFT JOIN	wcf".WCF_N."_tracked_visit tracked_board_visit
				ON		(tracked_board_visit.objectTypeID = ".VisitTracker::getInstance()->getObjectTypeID('com.woltlab.wbb.board')." AND tracked_board_visit.objectID = thread.boardID AND tracked_board_visit.userID = ".WCF::getUser()->userID.")
				".$conditionBuilder."
				GROUP BY	thread.boardID";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditionBuilder->getParameters());
			$this->unreadThreads = $statement->fetchMap('boardID', 'count');
		}
	}
	
	/**
	 * Reads the users online.
	 */
	protected function initUsersOnline() {
		$this->usersOnline = [];
		
		$usersOnlineList = new UsersOnlineList();
		$usersOnlineList->getConditionBuilder()->add('(session.pageID = ? OR session.parentPageID = ?)', [PageCache::getInstance()->getPageByIdentifier('com.woltlab.wbb.Board')->pageID, PageCache::getInstance()->getPageByIdentifier('com.woltlab.wbb.Board')->pageID]);
		$usersOnlineList->getConditionBuilder()->add('session.userID IS NOT NULL');
		$usersOnlineList->readObjects();
		
		foreach ($usersOnlineList as $user) {
			$boardID = ($user->pageID == PageCache::getInstance()->getPageByIdentifier('com.woltlab.wbb.Board')->pageID ? $user->pageObjectID : $user->parentPageObjectID);
			if (!isset($this->usersOnline[$boardID])) $this->usersOnline[$boardID] = [];
			
			$this->usersOnline[$boardID][] = $user;
		}
	}
	
	/**
	 * Returns the board with the given board id from cache.
	 * 
	 * @param	integer		$boardID
	 * @return	Board
	 */
	public function getBoard($boardID) {
		if (!isset($this->cachedBoards[$boardID])) {
			return null;
		}
		
		return $this->cachedBoards[$boardID];
	}
	
	/**
	 * Returns the direct children of a board.
	 * 
	 * @param	integer		$parentID
	 * @return	integer[]
	 */
	public function getChildIDs($parentID = null) {
		if ($parentID === null) $parentID = '';
		
		if (!isset($this->cachedBoardStructure[$parentID])) return [];
		
		return $this->cachedBoardStructure[$parentID];
	}
	
	/**
	 * Returns all children of a board (recursively).
	 *
	 * @param	integer		$parentID
	 * @return	integer[]
	 */
	public function getAllChildIDs($parentID = null) {
		if ($parentID === null) $parentID = '';
		
		$boardIDs = [];
		
		if (isset($this->cachedBoardStructure[$parentID])) {
			$boardIDs = $this->cachedBoardStructure[$parentID];
			foreach ($this->cachedBoardStructure[$parentID] as $boardID) {
				$boardIDs = array_merge($boardIDs, $this->getAllChildIDs($boardID));
			}
		}
		
		return $boardIDs;
	}
	
	/**
	 * Returns the number of clicks.
	 * 
	 * @param	integer		$boardID
	 * @return	integer
	 */
	public function getClicks($boardID) {
		if (isset($this->counts[$boardID])) {
			return $this->counts[$boardID]['clicks'];
		}
		
		return 0;
	}
	
	/**
	 * Returns the number of threads.
	 * 
	 * @param	integer		$boardID
	 * @return	integer
	 */
	public function getThreads($boardID) {
		if (isset($this->counts[$boardID])) {
			return $this->counts[$boardID]['threads'];
		}
		
		return 0;
	}
	
	/**
	 * Returns the number of posts.
	 * 
	 * @param	integer		$boardID
	 * @return	integer
	 */
	public function getPosts($boardID) {
		if (isset($this->counts[$boardID])) {
			return $this->counts[$boardID]['posts'];
		}
		
		return 0;
	}
	
	/**
	 * Returns the last post.
	 * 
	 * @param	integer		$boardID
	 * @param	integer		$languageID
	 * @return	ViewableThread
	 */
	public function getLastPost($boardID, $languageID = null) {
		if ($this->lastPosts === null) $this->initLastPosts();
		
		if (isset($this->lastPosts[$boardID][$languageID])) {
			return $this->lastPosts[$boardID][$languageID];
		}
		
		return null;
	}
	
	/**
	 * Loads the last posts.
	 */
	protected function initLastPosts() {
		$this->lastPosts = [];
		
		// handle private boards
		$privateBoardIDs = $privateThreadIDs = [];
		if (WCF::getUser()->userID) {
			foreach ($this->getBoards() as $board) {
				if ($board->isPrivate && !$board->canReadPrivateThreads()) {
					$privateBoardIDs[] = $board->boardID;
					$privateThreadIDs[$board->boardID] = 0;
				}
			}
			
			if (!empty($privateBoardIDs)) {
				$innerConditions = new PreparedStatementConditionBuilder();
				$innerConditions->add("boardID = board.boardID");
				$innerConditions->add("userID = ?", [WCF::getUser()->userID]);
				$innerConditions->add("isDeleted = ?", [0]);
				$innerConditions->add("isDisabled = ?", [0]);
				$innerConditions->add("movedThreadID IS NULL");
				
				$languageIDs = LanguageFactory::getInstance()->getContentLanguageIDs();
				if (!empty($languageIDs)) $innerConditions->add("languageID IN (?)", [$languageIDs]);
				
				$outerConditions = new PreparedStatementConditionBuilder();
				$outerConditions->add("board.boardID IN (?)", [$privateBoardIDs]);
				
				$sql = "SELECT  board.boardID,
						(
							SELECT          threadID
							FROM            wbb" . WCF_N . "_thread
							" . $innerConditions . "
							ORDER BY        lastPostTime DESC
							LIMIT           1
						) AS threadID
					FROM    wbb" . WCF_N . "_board board
					" . $outerConditions;
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute(
					array_merge($innerConditions->getParameters(), $outerConditions->getParameters())
				);
				while ($row = $statement->fetchArray()) {
					if (!$row['threadID']) continue;
					
					$this->lastPostThreadIDs[] = $row['threadID'];
					$privateThreadIDs[$row['boardID']] = $row['threadID'];
				}
			}
		}
		
		if (!empty($this->lastPostThreadIDs)) {
			$threadList = new ViewableThreadList(false, true);
			$threadList->setObjectIDs($this->lastPostThreadIDs);
			$threadList->sqlOrderBy = 'lastPostTime DESC, lastPostID DESC';
			$threadList->readObjects();
			
			foreach ($threadList as $thread) {
				if (isset($privateThreadIDs[$thread->boardID]) && $privateThreadIDs[$thread->boardID] != $thread->threadID) {
					// ignore threads for private boards that have been loaded
					// using to the global last post cache
					continue;
				}
				
				$this->lastPosts[$thread->boardID][$thread->languageID] = $thread;
			}
		}
	}
	
	/**
	 * Returns true if the requested board is opened.
	 * 
	 * @param	integer		$boardID
	 * @return	boolean
	 */
	public function isOpen($boardID) {
		if ($this->closedBoardIDs === null) {
			$this->closedBoardIDs = UserCollapsibleContentHandler::getInstance()->getCollapsedContent(UserCollapsibleContentHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board'));
		}
		
		return (!in_array($boardID, $this->closedBoardIDs));
	}
	
	/**
	 * Returns true if the user has ignored the given board.
	 * 
	 * @param	integer		$boardID
	 * @return	boolean
	 */
	public function isIgnored($boardID) {
		if ($this->ignoredBoardIDs === null) {
			$this->ignoredBoardIDs = UserCollapsibleContentHandler::getInstance()->getCollapsedContent(UserCollapsibleContentHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.ignoredBoard'));
		}
		
		return in_array($boardID, $this->ignoredBoardIDs);
	}
	
	/**
	 * Returns the number of unread threads.
	 * 
	 * @param	integer		$boardID
	 * @return	integer
	 */
	public function getUnreadThreads($boardID) {
		if ($this->unreadThreads === null) {
			$this->initUnreadThreads();
		}
		
		if (isset($this->unreadThreads[$boardID])) return $this->unreadThreads[$boardID];
		return 0;
	}
	
	/**
	 * Returns a list of all boards.
	 * 
	 * @return	Board[]
	 */
	public function getBoards() {
		return $this->cachedBoards;
	}
	
	/**
	 * Returns the ids of the label groups associated with the board with the
	 * given id. If no board is given, the complete mapping data is returned.
	 *
	 * @param	integer		$boardID
	 * @return	array
	 * @deprecated	since 4.1, use getLabelGroupIDs() instead
	 */
	public function getLabelGroups($boardID) {
		return $this->getLabelGroupIDs($boardID);
	}
	
	/**
	 * Returns the ids of the label groups associated with the board with the
	 * given id. If no board is given, the complete mapping data is returned.
	 * 
	 * @param	integer		$boardID
	 * @return	array
	 */
	public function getLabelGroupIDs($boardID = null) {
		if ($boardID === null) {
			return $this->cachedLabelGroups;
		}
		
		if (isset($this->cachedLabelGroups[$boardID])) {
			return $this->cachedLabelGroups[$boardID];
		}
		
		return [];
	}
	
	/**
	 * Returns the users online list.
	 * 
	 * @param	integer		$boardID
	 * @return	UserOnline[]
	 */
	public function getUsersOnline($boardID) {
		if ($this->usersOnline === null) {
			$this->initUsersOnline();
		}
		
		if (isset($this->usersOnline[$boardID])) return $this->usersOnline[$boardID];
		return [];
	}
	
	/**
	 * Returns a list of user moderators for given board id.
	 * 
	 * @param	integer		$boardID
	 * @return	integer[]
	 */
	public function getUserModerators($boardID) {
		if (isset($this->cachedModerators['users'][$boardID])) {
			return $this->cachedModerators['users'][$boardID];
		}
		
		return [];
	}
	
	/**
	 * Returns a list of group moderators for given board id.
	 * 
	 * @param	integer		$boardID
	 * @return	integer[]
	 */
	public function getGroupModerators($boardID) {
		if (isset($this->cachedModerators['groups'][$boardID])) {
			return $this->cachedModerators['groups'][$boardID];
		}
		
		return [];
	}
}
