<?php
namespace wbb\data\board;
use wbb\data\thread\ThreadAction;
use wbb\data\thread\ThreadList;
use wbb\system\log\modification\ThreadModificationLogHandler;
use wcf\data\object\type\ObjectTypeCache;
use wcf\data\package\PackageCache;
use wcf\data\user\object\watch\UserObjectWatchList;
use wcf\data\AbstractDatabaseObjectAction;
use wcf\data\ILoadableContainerAction;
use wcf\data\ISortableAction;
use wcf\data\IToggleContainerAction;
use wcf\system\acl\ACLHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\event\EventHandler;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\UserInputException;
use wcf\system\label\LabelHandler;
use wcf\system\language\LanguageFactory;
use wcf\system\request\LinkHandler;
use wcf\system\style\StyleHandler;
use wcf\system\user\collapsible\content\UserCollapsibleContentHandler;
use wcf\system\user\notification\UserNotificationHandler;
use wcf\system\user\object\watch\UserObjectWatchHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\visitTracker\VisitTracker;
use wcf\system\WCF;
use wcf\util\JSON;

/**
 * Executes board-related actions.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Board
 * 
 * @method	BoardEditor[]	getObjects()
 * @method	BoardEditor	getSingleObject()
 */
class BoardAction extends AbstractDatabaseObjectAction implements ILoadableContainerAction, ISortableAction, IToggleContainerAction {
	/**
	 * @inheritDoc
	 */
	protected $allowGuestAccess = ['loadContainer', 'markAsRead', 'markAllAsRead', 'toggleContainer'];
	
	/**
	 * @inheritDoc
	 */
	protected $className = BoardEditor::class;
	
	/**
	 * @inheritDoc
	 */
	protected $permissionsDelete = ['admin.board.canDeleteBoard'];
	
	/**
	 * @inheritDoc
	 */
	protected $requireACP = ['copy', 'create', 'delete', 'update', 'updatePosition'];
	
	/**
	 * board object
	 * @var	Board
	 */
	public $board;
	
	/**
	 * board editor object
	 * @var	BoardEditor
	 */
	public $boardEditor;
	
	/**
	 * @inheritDoc
	 * @return	Board
	 */
	public function create() {
		// remove position from data
		$position = 0;
		if (isset($this->parameters['data']['position'])) {
			$position = $this->parameters['data']['position'];
			unset($this->parameters['data']['position']);
		}
		// set default values
		if (!isset($this->parameters['data']['time'])) {
			$this->parameters['data']['time'] = TIME_NOW;
		}
		
		/** @var Board $board */
		$board = parent::create();
		$boardEditor = new BoardEditor($board);
		
		// set position
		$boardEditor->setPosition($this->parameters['data']['parentID'], $position);
		
		return $board;
	}
	
	/**
	 * @inheritDoc
	 */
	public function update() {
		parent::update();
		
		// update position if required
		if (count($this->objects) == 1 && isset($this->parameters['data']['parentID']) && isset($this->parameters['data']['position'])) {
			/** @var BoardEditor $board */
			$board = $this->objects[0];
			
			if (($board->parentID != $this->parameters['data']['parentID']) || $board->position != $this->parameters['data']['position']) {
				$board->setPosition($this->parameters['data']['parentID'], $this->parameters['data']['position']);
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateDelete() {
		parent::validateDelete();
		
		foreach ($this->getObjects() as $board) {
			if ($board->posts > 5000) {
				throw new UserInputException('objectIDs');
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function delete() {
		// update board parents and positions
		$boardIDs = [];
		foreach ($this->getObjects() as $boardEditor) {
			$boardIDs[] = $boardEditor->boardID;
			$childBoardIDs = BoardCache::getInstance()->getChildIDs($boardEditor->boardID);
			
			if (empty($childBoardIDs)) {
				// no children: reduce the position value of each board
				// with a higher position by one
				$nextBoardsPositionUpdate = -1;
			}
			else {
				// increase the position value of each board with a higher
				// position depending on the number of children
				$nextBoardsPositionUpdate = count($childBoardIDs) - 1;
			}
			
			// if the board has just one child, the position
			// of the other boards at the same level stays the
			// same
			if ($nextBoardsPositionUpdate) {
				$conditionBuilder = new PreparedStatementConditionBuilder();
				if ($boardEditor->parentID === null) {
					$conditionBuilder->add('parentID IS NULL');
				}
				else {
					$conditionBuilder->add('parentID = ?', [$boardEditor->parentID]);
				}
				$conditionBuilder->add('position >= ?', [$boardEditor->position]);
				
				$sql = "UPDATE	wbb".WCF_N."_board
					SET	position = position + ?
					".$conditionBuilder;
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute(array_merge([$nextBoardsPositionUpdate], $conditionBuilder->getParameters()));
			}
			
			// update the positions of the child boards
			$position = $boardEditor->position;
			foreach ($childBoardIDs as $childBoardID) {
				$childBoardEditor = new BoardEditor(BoardCache::getInstance()->getBoard($childBoardID));
				$childBoardEditor->update([
					'parentID' => $boardEditor->parentID,
					'position' => $position++
				]);
			}
		}
		
		if (!empty($boardIDs)) {
			// delete threads
			$threadList = new ThreadList();
			$threadList->getConditionBuilder()->add('boardID IN (?)', [$boardIDs]);
			$threadList->readObjectIDs();
			if (count($threadList->getObjectIDs())) {
				$threadAction = new ThreadAction($threadList->getObjectIDs(), 'delete', [
					'ignoreThreadModificationLogs' => true
				]);
				$threadAction->executeAction();
			}
			
			// delete subscriptions
			UserObjectWatchHandler::getInstance()->deleteObjects('com.woltlab.wbb.board', $boardIDs);
			
			// delete all thread modification logs at once
			ThreadModificationLogHandler::getInstance()->deleteLogsByParentIDs($boardIDs);
		}
		
		$count = parent::delete();
		
		BoardEditor::resetCache();
		UserStorageHandler::getInstance()->clear();
		
		return $count;
	}
	
	/**
	 * Checks the permissions for the given board ids.
	 */
	protected function checkPermissions() {
		// read data
		if (empty($this->objects)) {
			$this->readObjects();
			
			if (empty($this->objects)) {
				throw new UserInputException('objectIDs');
			}
		}
		
		// check permissions
		foreach ($this->getObjects() as $board) {
			if (!$board->getPermission()) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Validates parameters for collapsible content actions.
	 */
	protected function validateCollapsibleContentAction() {
		$this->readString('containerID');
		$this->readString('newState');
		
		if (count($this->objectIDs) != 1) {
			throw new UserInputException('objectIDs');
		}
		
		$boardID = reset($this->objectIDs);
		$this->board = BoardCache::getInstance()->getBoard($boardID);
		if ($this->board === null) {
			throw new UserInputException('objectIDs');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateToggleContainer() {
		if (!WCF::getSession()->getPermission('admin.board.canEditBoard')) {
			throw new PermissionDeniedException();
		}
		
		$this->validateCollapsibleContentAction();
		if (!$this->board->isCategory()) {
			throw new UserInputException('objectIDs');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateLoadContainer() {
		$this->validateCollapsibleContentAction();
		if (!$this->board->isCategory() || !$this->board->getPermission()) {
			throw new UserInputException('objectIDs');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function toggleContainer() {
		$objectTypeID = UserCollapsibleContentHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.acpBoard');
		
		if ($this->parameters['newState'] == 'open') {
			UserCollapsibleContentHandler::getInstance()->markAsOpened($objectTypeID, $this->board->boardID);
		}
		else {
			UserCollapsibleContentHandler::getInstance()->markAsCollapsed($objectTypeID, $this->board->boardID);
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function loadContainer() {
		$objectTypeID = UserCollapsibleContentHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board');
		
		if ($this->parameters['newState'] == 'open') {
			UserCollapsibleContentHandler::getInstance()->markAsOpened($objectTypeID, $this->board->boardID);
		}
		else {
			UserCollapsibleContentHandler::getInstance()->markAsCollapsed($objectTypeID, $this->board->boardID);
		}
		
		$startDepth = !empty($this->parameters['depth']) ? intval($this->parameters['depth']) - 1 : 0;
		$nodeList = new DetailedBoardNodeList($this->board->parentID, $startDepth, [$this->board->boardID]);
		$nodeList->readNodeTree();
		
		WCF::getTPL()->assign([
			'boardNodeList' => $nodeList->getNodeList(),
			'startDepth' => $startDepth,
			'disableAds' => true
		]);
		
		return [
			'containerID' => $this->parameters['containerID'],
			'content' => WCF::getTPL()->fetch('boardNodeList', 'wbb'),
			'isOpen' => $this->parameters['newState'] == 'open' ? 1 : 0
		];
	}
	
	/**
	 * Marks a board as read.
	 */
	public function markAsRead() {
		// get all board ids (include children)
		$boardIDs = [];
		$getBoardIDs = function($boardID, $getBoardIDs) use (&$boardIDs) {
			if (in_array($boardID, $boardIDs)) {
				return;
			}
			
			$boardIDs[] = $boardID;
			
			foreach (BoardCache::getInstance()->getChildIDs($boardID) as $childBoardID) {
				$getBoardIDs($childBoardID, $getBoardIDs);
			}
		};
		
		foreach ($this->objectIDs as $boardID) {
			$getBoardIDs($boardID, $getBoardIDs);
		}
		
		foreach ($boardIDs as $boardID) {
			VisitTracker::getInstance()->trackObjectVisit('com.woltlab.wbb.board', $boardID);
		}
		
		// reset storage
		if (WCF::getUser()->userID) {
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadThreads');
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedBoards');
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedThreads');
			
			// delete obsolete notifications
			if (!empty($boardIDs)) {
				$parameters = [ObjectTypeCache::getInstance()->getObjectTypeByName('com.woltlab.wcf.user.objectWatch', 'com.woltlab.wbb.thread')->objectTypeID, WCF::getUser()->userID];
				$parameters = array_merge($parameters, $boardIDs);
				$parameters[] = PackageCache::getInstance()->getPackageID('com.woltlab.wbb');
				$parameters[] = UserNotificationHandler::getInstance()->getEvent('com.woltlab.wbb.post', 'post')->eventID;
				$parameters[] = WCF::getUser()->userID;
				$parameters[] = TIME_NOW;
				
				$sql = "SELECT		post.postID
					FROM		wbb".WCF_N."_post post,
							wcf".WCF_N."_user_notification notification,
							wcf".WCF_N."_user_notification_to_user notification_to_user,
							(
								SELECT	threadID
								FROM	wcf".WCF_N."_user_object_watch object_watch,
									wbb".WCF_N."_thread thread
								WHERE	thread.threadID = object_watch.objectID
									AND object_watch.objectTypeID = ?
									AND object_watch.userID = ?
									AND thread.boardID IN (?".(count($boardIDs) > 1 ? str_repeat(',?', count($boardIDs) - 1) : '').")
							) AS thread
					WHERE		post.threadID = thread.threadID
							AND notification.packageID = ?
							AND notification.eventID = ?
							AND notification.objectID = post.postID
							AND notification_to_user.notificationID = notification.notificationID
							AND notification_to_user.userID = ?
							AND post.time <= ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($parameters);
				$postIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
				
				if (!empty($postIDs)) {
					UserNotificationHandler::getInstance()->markAsConfirmed('post', 'com.woltlab.wbb.post', [WCF::getUser()->userID], $postIDs);
				}
				
				$conditionBuilder = new PreparedStatementConditionBuilder();
				$conditionBuilder->add('notification.objectID = thread.threadID');
				$conditionBuilder->add('thread.boardID IN (?)', [$boardIDs]);
				
				$sql = "SELECT	thread.threadID
					FROM	wbb".WCF_N."_thread thread,
						wcf".WCF_N."_user_notification notification
					".$conditionBuilder;
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute($conditionBuilder->getParameters());
				$threadIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
				
				if (!empty($threadIDs)) {
					UserNotificationHandler::getInstance()->markAsConfirmed('thread', 'com.woltlab.wbb.thread', [WCF::getUser()->userID], $threadIDs);
				}
			}
		}
	}
	
	/**
	 * Validates the mark as read action.
	 */
	public function validateMarkAsRead() {
		$this->checkPermissions();
	}
	
	/**
	 * Marks all boards as read.
	 */
	public function markAllAsRead() {
		VisitTracker::getInstance()->trackTypeVisit('com.woltlab.wbb.thread');
		VisitTracker::getInstance()->deleteObjectVisits('com.woltlab.wbb.board');
		
		// reset storage and notifications
		if (WCF::getUser()->userID) {
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadThreads');
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedBoards');
			UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedThreads');
			UserNotificationHandler::getInstance()->markAsConfirmed('post', 'com.woltlab.wbb.post', [WCF::getUser()->userID]);
		}
	}
	
	/**
	 * Validates the mark all as read action.
	 */
	public function validateMarkAllAsRead() {
		// does nothing
	}
	
	/**
	 * Loads the list of ignored boards.
	 */
	public function loadIgnoredBoards() {
		$boardNodeList = new RestrictedBoardNodeList();
		$boardNodeList->readNodeTree();
		
		WCF::getTPL()->assign([
			'boardNodeList' => $boardNodeList->getNodeList()
		]);
		return [
			'template' => WCF::getTPL()->fetch('ignoredBoards', 'wbb')
		];
	}
	
	/**
	 * Validates the load ignored boards action.
	 */
	public function validateLoadIgnoredBoards() {
		// does nothing
	}
	
	/**
	 * Saves the ignored boards.
	 */
	public function saveIgnoredBoards() {
		$objectTypeID = UserCollapsibleContentHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.ignoredBoard');
		
		// delete old values
		UserCollapsibleContentHandler::getInstance()->reset($objectTypeID);
		
		// save board ids
		foreach ($this->objectIDs as $boardID) {
			UserCollapsibleContentHandler::getInstance()->markAsCollapsed($objectTypeID, $boardID);
		}
		
		// reset user storage
		UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadThreads');
		
		// return new board list
		$boardNodeList = new DetailedBoardNodeList();
		$boardNodeList->readNodeTree();
		
		WCF::getTPL()->assign([
			'boardNodeList' => $boardNodeList->getNodeList()
		]);
		return [
			'template' => WCF::getTPL()->fetch('boardNodeList', 'wbb')
		];
	}
	
	/**
	 * Validates the save ignored boards action.
	 */
	public function validateSaveIgnoredBoards() {
		if (count($this->objectIDs)) {
			$this->checkPermissions();
		}
		
		// get ignorable board ids
		$boardIDs = array_keys(BoardCache::getInstance()->getBoards());
		foreach (BoardCache::getInstance()->getChildIDs() as $boardID) {
			$this->removeNonIgnorableBoardIDs($boardIDs, $boardID);
		}
		
		foreach ($this->getObjects() as $board) {
			if (!in_array($board->boardID, $boardIDs)) {
				throw new UserInputException('objectIDs');
			}
		}
	}
	
	/**
	 * Checks if the board with the given id can be ignored. If that is not
	 * possible, its id is removed from the first argument and true is returned.
	 * 
	 * @param	integer[]		$boardIDs
	 * @param	integer			$boardID
	 * @return	boolean
	 */
	protected function removeNonIgnorableBoardIDs(array &$boardIDs, $boardID) {
		$childBoardIDs = BoardCache::getInstance()->getChildIDs($boardID);
		
		// check if board is a leaf
		if (empty($childBoardIDs)) {
			if (!BoardCache::getInstance()->getBoard($boardID)->ignorable) {
				unset($boardIDs[array_search($boardID, $boardIDs)]);
				return true;
			}
			else {
				return false;
			}
		}
		else {
			// if the board has at least one child that cannot be ignored,
			// the board itself can also be not ignored
			$hasNonIgnorableChild = false;
			foreach ($childBoardIDs as $childBoardID) {
				if ($this->removeNonIgnorableBoardIDs($boardIDs, $childBoardID)) {
					$hasNonIgnorableChild = true;
				}
			}
			
			if ($hasNonIgnorableChild) {
				unset($boardIDs[array_search($boardID, $boardIDs)]);
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * @inheritDoc
	 */
	public function validateUpdatePosition() {
		if (!WCF::getSession()->getPermission('admin.board.canEditBoard')) {
			throw new PermissionDeniedException();
		}
		
		if (!isset($this->parameters['data']['structure'])) {
			throw new UserInputException('structure');
		}
		
		// collect board ids
		$boardIDs = [];
		foreach ($this->parameters['data']['structure'] as $parentID => $childBoardIDs) {
			if ($parentID) {
				$boardIDs[] = $parentID;
			}
			
			foreach ($childBoardIDs as $boardID) {
				if (!$boardID) {
					throw new UserInputException('structure');
				}
				
				$boardIDs[] = $boardID;
			}
		}
		
		// validate board ids against database
		$boardIDs = array_unique($boardIDs);
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("boardID IN (?)", [$boardIDs]);
		$sql = "SELECT	boardID
			FROM	wbb".WCF_N."_board
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		while ($row = $statement->fetchArray()) {
			$index = array_search($row['boardID'], $boardIDs);
			if ($index !== false) {
				unset($boardIDs[$index]);
			}
		}
		
		if (!empty($boardIDs)) {
			throw new UserInputException('structure');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function updatePosition() {
		// handle top boards first
		$sql = "UPDATE	wbb".WCF_N."_board
			SET	parentID = NULL,
				position = ?
			WHERE	boardID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		WCF::getDB()->beginTransaction();
		foreach ($this->parameters['data']['structure'][0] as $position => $boardID) {
			$statement->execute([
				$position + 1,
				$boardID
			]);
		}
		WCF::getDB()->commitTransaction();
		
		// handle non-top boards
		if (count($this->parameters['data']['structure']) > 1) {
			$sql = "UPDATE	wbb".WCF_N."_board
				SET	parentID = ?,
					position = ?
				WHERE	boardID = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			WCF::getDB()->beginTransaction();
			foreach ($this->parameters['data']['structure'] as $parentID => $boardIDs) {
				if ($parentID == 0) {
					continue;
				}
				
				foreach ($boardIDs as $position => $boardID) {
					$statement->execute([
						$parentID,
						$position + 1,
						$boardID
					]);
				}
			}
			WCF::getDB()->commitTransaction();
		}
	}
	
	/**
	 * Validates parameters to copy a board.
	 */
	public function validateCopy() {
		WCF::getSession()->checkPermissions(['admin.board.canAddBoard', 'admin.board.canEditBoard']);
		
		$this->readBoolean('recursive');
		
		$this->boardEditor = $this->getSingleObject();
	}
	
	/**
	 * Copies a board, optionally recursively copies the child elements.
	 * 
	 * @return	string[]
	 */
	public function copy() {
		$board = $this->cloneBoard(
			$this->boardEditor->getDecoratedObject(),
			null,
			$this->parameters['recursive'] ? true : false
		);
		
		// reset cache
		BoardEditor::resetCache();
		
		// reset language cache
		LanguageFactory::getInstance()->deleteLanguageCache();
		
		// for recursive copying, always reset the stylesheets to avoid having to check
		// every board separately for icon data
		if (!empty($board->getIconData()) || $this->parameters['recursive']) {
			StyleHandler::resetStylesheets(false);
		}
		
		return [
			'redirectURL' => LinkHandler::getInstance()->getLink('BoardEdit', [
				'application' => 'wbb',
				'id' => $board->boardID
			])
		];
	}
	
	/**
	 * Clones a board.
	 * 
	 * @param	Board		$origin
	 * @param	integer		$parentID
	 * @param	boolean		$recursive
	 * @return	Board
	 */
	protected function cloneBoard(Board $origin, $parentID = null, $recursive = false) {
		$parameters = [
			'data' => [
				'parentID' => $parentID === null ? $origin->parentID : $parentID,
				'boardType' => $origin->boardType,
				'descriptionUseHtml' => $origin->descriptionUseHtml,
				'externalURL' => $origin->externalURL,
				'time' => TIME_NOW,
				'countUserPosts' => $origin->countUserPosts,
				'daysPrune' => $origin->daysPrune,
				'enableMarkingAsDone' => $origin->enableMarkingAsDone,
				'ignorable' => $origin->ignorable,
				'isClosed' => $origin->isClosed,
				'isInvisible' => $origin->isInvisible,
				'postSortOrder' => $origin->postSortOrder,
				'postsPerPage' => $origin->postsPerPage,
				'searchable' => $origin->searchable,
				'searchableForSimilarThreads' => $origin->searchableForSimilarThreads,
				'showSubBoards' => $origin->showSubBoards,
				'sortField' => $origin->sortField,
				'sortOrder' => $origin->sortOrder,
				'styleID' => $origin->styleID,
				'threadsPerPage' => $origin->threadsPerPage,
				'iconData' => (!empty($origin->getIconData()) ? JSON::encode($origin->getIconData()) : null),
				'formID' => $origin->formID,
				'isPrivate' => $origin->isPrivate,
				'enableBestAnswer' => $origin->enableBestAnswer,
				'metaDescription' => $origin->metaDescription,
			]
		];
		
		if ($parentID !== null) {
			// use the same position if this board is a children (recursive copy)
			$parameters['data']['position'] = $origin->position;
		}
		
		$data = ['parameters' => $parameters, 'origin' => $origin];
		EventHandler::getInstance()->fireAction($this, 'cloneBoard', $data);
		$parameters = $data['parameters'];
		
		$boardAction = new BoardAction([], 'create', $parameters);
		$boardAction->executeAction();
		$returnValues = $boardAction->getReturnValues();
		
		$board = $returnValues['returnValues'];
		$boardEditor = new BoardEditor($board);
		
		// inherit the positioning with the exception that the new board will be succeeding the origin
		if ($parentID === null) {
			$boardEditor->setPosition($board->parentID, $origin->position + 1);
		}
		
		// update board title
		$title = $origin->title;
		if (preg_match('~^wbb\.board\.board\d+$~', $origin->title)) {
			$title = 'wbb.board.board'.$board->boardID;
			
			// add language items
			$sql = "INSERT INTO	wcf".WCF_N."_language_item
						(languageID, languageItem, languageItemValue, languageItemOriginIsSystem, languageCategoryID, packageID)
				SELECT		languageID, '".$title."', CONCAT(languageItemValue, ' (2)'), 0, languageCategoryID, packageID
				FROM		wcf".WCF_N."_language_item
				WHERE		languageItem = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$origin->title]);
		}
		else {
			$title .= ' (2)';
		}
		
		// update board description
		$description = $origin->description;
		if (preg_match('~^wbb\.board\.board\d+\.description$~', $origin->description)) {
			$description = 'wbb.board.board'.$board->boardID.'.description';
			
			// add language items
			$sql = "INSERT INTO	wcf".WCF_N."_language_item
						(languageID, languageItem, languageItemValue, languageItemOriginIsSystem, languageCategoryID, packageID)
				SELECT		languageID, '".$description."', languageItemValue, 0, languageCategoryID, packageID
				FROM		wcf".WCF_N."_language_item
				WHERE		languageItem = ?";
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute([$origin->description]);
		}
		
		$boardEditor->update([
			'description' => $description,
			'title' => $title
		]);
		
		// get ACL option ids
		$sql = "SELECT	optionID
			FROM	wcf".WCF_N."_acl_option
			WHERE	objectTypeID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([
			ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board')
		]);
		$optionIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
		
		// copy ACLs
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("optionID IN (?)", [$optionIDs]);
		$conditions->add("objectID = ?", [$origin->boardID]);
		
		$sql = "INSERT INTO	wcf".WCF_N."_acl_option_to_group
					(optionID, objectID, groupID, optionValue)
			SELECT		optionID, ".$board->boardID.", groupID, optionValue
			FROM		wcf".WCF_N."_acl_option_to_group
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		$sql = "INSERT INTO	wcf".WCF_N."_acl_option_to_user
					(optionID, objectID, userID, optionValue)
			SELECT		optionID, ".$board->boardID.", userID, optionValue
			FROM		wcf".WCF_N."_acl_option_to_user
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		// copy label group assignments
		$sql = "INSERT INTO	wcf".WCF_N."_label_group_to_object
					(groupID, objectTypeID, objectID)
			SELECT		groupID, objectTypeID, ".$board->boardID."
			FROM		wcf".WCF_N."_label_group_to_object
			WHERE		objectTypeID = ?
					AND objectID = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute([
			LabelHandler::getInstance()->getObjectType('com.woltlab.wbb.thread')->objectTypeID,
			$origin->boardID
		]);
		
		if ($recursive) {
			// get origin's children and clone them too
			foreach (BoardCache::getInstance()->getChildIDs($origin->boardID) as $boardID) {
				$childBoard = BoardCache::getInstance()->getBoard($boardID);
				
				$this->cloneBoard(
					$childBoard,
					$board->boardID,
					true
				);
			}
		}
		
		return $board;
	}
	
	/**
	 * Validates the 'stopWatching' action.
	 * 
	 * @since	5.0
	 */
	public function validateStopWatching() {
		if (empty($this->objectIDs)) {
			$boardWatchList = new UserObjectWatchList();
			$boardWatchList->getConditionBuilder()->add('user_object_watch.objectTypeID = ?', [
				UserObjectWatchHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board')
			]);
			$boardWatchList->getConditionBuilder()->add('user_object_watch.userID = ?', [WCF::getUser()->userID]);
			$boardWatchList->readObjects();
			
			foreach ($boardWatchList as $watchedBoard) {
				$this->objectIDs[] = $watchedBoard->objectID;
			}
		}
		
		if (empty($this->objectIDs)) {
			throw new IllegalLinkException();
		}
	}
	
	/**
	 * Stops watching certain boards for a certain user.
	 *
	 * @since	5.0
	 */
	public function stopWatching() {
		UserObjectWatchHandler::getInstance()->deleteObjects('com.woltlab.wbb.board', $this->objectIDs, [WCF::getUser()->userID]);
		UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbUnreadWatchedBoards');
		UserStorageHandler::getInstance()->reset([WCF::getUser()->userID], 'wbbWatchedBoards');
	}
}
