<?php
namespace wbb\data\board;
use wbb\data\thread\Thread;
use wbb\system\board\BoardPermissionCache;
use wbb\system\board\BoardPermissionHandler;
use wcf\data\user\object\watch\UserObjectWatch;
use wcf\data\user\UserProfile;
use wcf\data\DatabaseObject;
use wcf\data\ITitledLinkObject;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\SystemException;
use wcf\system\request\IRouteController;
use wcf\system\request\LinkHandler;
use wcf\system\style\StyleHandler;
use wcf\system\user\object\watch\UserObjectWatchHandler;
use wcf\system\WCF;
use wcf\util\JSON;

/**
 * Represents a board.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Data\Board
 * 
 * @property-read	integer		$boardID			unique id of the board
 * @property-read	integer|null	$parentID			id of the board's parent board
 * @property-read	integer		$position			position of the board in relation to its siblings
 * @property-read	integer		$boardType			numeric representation of the board type (0 = board, 1 = category, 2 = link)
 * @property-read	string		$title				title of the board or name of the language item containing the title
 * @property-read	string		$description			description of the board or name of the language item containing the description
 * @property-read	integer		$descriptionUseHtml		is 1 if HTML will be rendered in the description, otherwise 0
 * @property-read	string		$externalURL			external link used if `$boardType` = 2
 * @property-read	integer		$time				timestamp at which the board has been created
 * @property-read	integer		$countUserPosts			is 1 if posts in the board are included in users' post counters, otherwise 0
 * @property-read	integer		$daysPrune			default number of days in which the last post has to have been written to be shown in thread list
 * @property-read	integer		$enableMarkingAsDone		is 1 if threads in board can marked as done, otherwise 0
 * @property-read	integer		$ignorable			is 1 if users may ignore the board, otherwise 0
 * @property-read	integer		$isClosed			is 1 if no new threads may be created in board, otherwise 0
 * @property-read	integer		$isInvisible			is 1 if board will not be shown in board list, otherwise 0
 * @property-read	integer		$isPrivate			is 1 if threads inside this board are private, otherwise 0
 * @property-read	string		$postSortOrder			sort order of posts in threads
 * @property-read	integer		$postsPerPage			number of posts per page in threads
 * @property-read	integer		$searchable			is 1 if board can be searched, otherwise 0
 * @property-read	integer		$searchableForSimilarThreads	is 1 if board will be searched when determining similar threads, otherwise 0
 * @property-read	integer		$showSubBoards			is 1 if sub-boards are shown in board list, otherwise 0
 * @property-read	string		$sortField			default sort field of threads in board
 * @property-read	string		$sortOrder			default sort order of threads in board
 * @property-read	integer|null	$styleID			id of board-specific style or null if no style is set
 * @property-read	integer		$threadsPerPage			number of threads per page
 * @property-read	integer		$clicks				number of times an external link has been clicked, thus only relevant if `$boardType` = 2
 * @property-read	integer		$posts				number of non-disabled and non-deleted posts in board
 * @property-read	integer		$threads			number of non-disabled and non-deleted threads in board
 * @property-read	integer		$enableBestAnswer		is 1 if posts can mark as best answer, otherwise 0
 * @property-read       string          $metaDescription                meta description of the board or the name of the language item containing the description
 * @property-read       integer         $formID                         the formID of the linked thread form
 */
class Board extends DatabaseObject implements IRouteController, ITitledLinkObject {
	/**
	 * icon data per type
	 * @var string[][]
	 */
	protected $iconData = [];
	
	/**
	 * true if the board can be ignored (also considering settings of child boards)
	 * @var	boolean|null
	 */
	protected $isIgnorable;
	
	/**
	 * true if the active user is subscribed to the board
	 * @var	boolean|null
	 */
	protected $subscribed;
	
	/**
	 * Defines that a board acts as a container for threads.
	 * @var	integer
	 */
	const TYPE_BOARD = 0;
	
	/**
	 * Defines that a board acts as a category.
	 * @var	integer
	 */
	const TYPE_CATEGORY = 1;
	
	/**
	 * Defines that a board acts as an external link.
	 * @var	integer
	 */
	const TYPE_LINK = 2;
	
	/**
	 * list of default icons per type
	 * @var string[]
	 */
	protected static $defaultIcons = [
		'default' => 'folder-open-o',
		'unread' => 'folder-open',
		'closed' => 'lock',
		'external' => 'globe'
	];
	
	/**
	 * @inheritDoc
	 */
	protected function handleData($data) {
		$this->iconData = [];
		
		if (isset($data['iconData'])) {
			try {
				$this->iconData = JSON::decode($data['iconData']);
			}
			catch (SystemException $e) {}
			
			unset($data['iconData']);
		}
		
		parent::handleData($data);
	}
	
	/**
	 * Returns the icon name by type.
	 * 
	 * @param       string          $type           type identifier
	 * @return      string          icon name
	 */
	public function getIconName($type) {
		return (isset($this->iconData[$type]['icon'])) ? $this->iconData[$type]['icon'] : self::getDefaultIcon($type);
	}
	
	/**
	 * Returns the icon color by type.
	 * 
	 * @param       string          $type           type identifier
	 * @return      string          color
	 */
	public function getIconColor($type) {
		return (isset($this->iconData[$type]['color']) && !empty($this->iconData[$type]['useColor'])) ? $this->iconData[$type]['color'] : '';
	}
	
	/**
	 * Returns the internal icon data.
	 * 
	 * @return      string[][]      internal icon data grouped by type
	 */
	public function getIconData() {
		return $this->iconData;
	}
	
	/**
	 * Builds the custom SCSS style rules for the board icon per type.
	 * 
	 * @param       string          $type           type identifier
	 * @return      string          SCSS style rules
	 */
	public function getStyleRules($type) {
		$icon = $color = '';
		if (isset($this->iconData[$type]['icon'])) {
			$icon = $this->iconData[$type]['icon'];
		}
		$color = $this->getIconColor($type);
		
		if (empty($icon) && empty($color)) {
			return '';
		}
		
		$selector = '.wbbBoardIcon'.$this->boardID.'.fa-'.self::$defaultIcons[$type];
		$scss = '';
		
		$subBoardSelector = '';
		if ($type === 'default' || $type === 'unread') {
			$subBoardSelector = '.wbbBoardIcon'.$this->boardID.'.fa-'.($type === 'default' ? 'folder-o' : 'folder');
		}
		
		if ($icon) {
			$scss = "{$selector}:before".($subBoardSelector ? ", {$subBoardSelector}:before" : "")." { content: \$fa-var-{$icon}; }\n";
		}
		if ($color) {
			$scss .= "{$selector}".($subBoardSelector ? ", {$subBoardSelector}" : "")." { color: {$color}; }\n";
		}
		
		return $scss;
	}
	
	/**
	 * Returns true if this board is no category and no external link.
	 * 
	 * @return	boolean
	 */
	public function isBoard() {
		return $this->boardType == self::TYPE_BOARD;
	}
	
	/**
	 * Returns true if this board is a category.
	 * 
	 * @return	boolean
	 */
	public function isCategory() {
		return $this->boardType == self::TYPE_CATEGORY;
	}
	
	/**
	 * Returns true if this board is an external link.
	 * 
	 * @return	boolean
	 */
	public function isExternalLink() {
		return $this->boardType == self::TYPE_LINK;
	}
	
	/**
	 * Returns a list of the parent boards of this board.
	 * 
	 * @return	Board[]
	 */
	public function getParentBoards() {
		$parentBoards = [];
		$parentBoard = $this;
		while ($parentBoard->parentID != null) {
			$parentBoard = BoardCache::getInstance()->getBoard($parentBoard->parentID);
			array_unshift($parentBoards, $parentBoard);
		}
		
		return $parentBoards;
	}
	
	/**
	 * @inheritDoc
	 */
	public function getLink() {
		return LinkHandler::getInstance()->getLink('Board', [
			'application' => 'wbb',
			'object' => $this,
			'forceFrontend' => true
		]);
	}
	
	/**
	 * Checks the given board permissions.
	 * 
	 * @param	string[]		$permissions
	 * @throws 	PermissionDeniedException
	 */
	public function checkPermission(array $permissions = ['canViewBoard']) {
		foreach ($permissions as $permission) {
			if (!$this->getPermission($permission)) {
				throw new PermissionDeniedException();
			}
		}
	}
	
	/**
	 * Checks whether the active user has the permission with the given name on this board.
	 * 
	 * @param	string		$permission	name of the requested permission
	 * @return	boolean
	 */
	public function getPermission($permission = 'canViewBoard') {
		if (BoardPermissionHandler::getInstance()->getPermission($this->boardID, $permission)) {
			return !WCF::getSession()->getNeverPermission('user.board.'.$permission);
		}
		
		return false;
	}
	
	/**
	 * Checks board permissions for given user.
	 * 
	 * @param	UserProfile	$user
	 * @param	string[]	$permissions
	 * @param       Thread          $thread
	 * @return	boolean
	 */
	public function checkUserPermissions(UserProfile $user, array $permissions = ['canViewBoard'], Thread $thread = null) {
		$userPermissions = BoardPermissionCache::getInstance()->getPermissions($user->getDecoratedObject());
		
		foreach ($permissions as $permission) {
			if (!isset($userPermissions[$this->boardID][$permission])) {
				// no board-specific permission -> check user group permission
				if (!$user->getPermission('user.board.'.$permission)) {
					return false;
				}
			}
			else if (!$userPermissions[$this->boardID][$permission]) {
				return false;
			}
		}
		
		if ($this->isPrivate) {
			$canReadPrivateThread = false;
			if (isset($userPermissions[$this->boardID]['canReadPrivateThread'])) {
				$canReadPrivateThread = $userPermissions[$this->boardID]['canReadPrivateThread'];
			}
			else if ($user->getPermission('mod.board.canReadPrivateThread')) {
				$canReadPrivateThread = true;
			}
			
			if (!$canReadPrivateThread) {
				if ($thread !== null && $thread->userID == $user->userID) {
					return true;
				}
				
				return false;
			}
		}
		
		return true;
	}
	
	/**
	 * Checks whether the active user has the moderator permission with the given name on this board.
	 * 
	 * @param	string		$permission	name of the requested permission
	 * @return	boolean
	 */
	public function getModeratorPermission($permission) {
		return BoardPermissionHandler::getInstance()->getModeratorPermission($this->boardID, $permission);
	}
	
	/**
	 * Returns true if the active user can start new threads in this board.
	 * 
	 * @return	boolean
	 */
	public function canStartThread() {
		return ($this->isBoard() && $this->getPermission('canStartThread') && !$this->isClosed);
	}
	
	/**
	 * @inheritDoc
	 */
	public function getTitle() {
		return WCF::getLanguage()->get($this->title);
	}
	
	/**
	 * Returns true if the board can be ignored.
	 * 
	 * @return	boolean
	 */
	public function isIgnorable() {
		if ($this->isIgnorable === null) {
			if (!$this->ignorable) {
				$this->isIgnorable = false;
			}
			else {
				$this->isIgnorable = true;
				foreach (BoardCache::getInstance()->getChildIDs($this->boardID) as $childBoardID) {
					if (!BoardCache::getInstance()->getBoard($childBoardID)->isIgnorable()) {
						$this->isIgnorable = false;
						break;
					}
				}
			}
		}
		
		return $this->isIgnorable;
	}
	
	/**
	 * Returns true if the user has ignored this board.
	 * 
	 * @return	boolean
	 */
	public function isIgnored() {
		return WBB_MODULE_IGNORE_BOARDS && BoardCache::getInstance()->isIgnored($this->boardID);
	}
	
	/**
	 * Returns true if the active user has the permission to enter this board.
	 * 
	 * @return	boolean
	 */
	public function canEnter() {
		if ($this->getPermission('canViewBoard') && $this->getPermission('canEnterBoard')) {
			// guests are never allowed to enter
			return (!$this->isPrivate || WCF::getUser()->userID);
		}
		
		return false;
	}
	
	/**
	 * Returns true if the active user has the permission to edit threads within this board.
	 * 
	 * @return	boolean
	 */
	public function canEditThreads() {
		return $this->getModeratorPermission('canEditPost');
	}
	
	/**
	 * Returns post sort order.
	 * 
	 * @return	string
	 */
	public function getPostSortOrder() {
		return $this->postSortOrder ?: WBB_THREAD_DEFAULT_POST_SORT_ORDER;
	}
	
	/**
	 * Returns posts per page.
	 * 
	 * @return	integer
	 */
	public function getPostsPerPage() {
		/** @noinspection PhpUndefinedFieldInspection */
		return WCF::getUser()->postsPerPage ?: ($this->postsPerPage ?: WBB_THREAD_POSTS_PER_PAGE);
	}
	
	/**
	 * Returns the number of posts directly in this board (excluding child boards).
	 * 
	 * @return	integer
	 */
	public function getPosts() {
		return BoardCache::getInstance()->getPosts($this->boardID);
	}
	
	/**
	 * Returns the number of threads directly in this board (excluding child
	 * boards).
	 * 
	 * @return	integer
	 */
	public function getThreads() {
		return BoardCache::getInstance()->getThreads($this->boardID);
	}
	
	/**
	 * Returns the number of posts per day in this board.
	 * 
	 * @return	float
	 */
	public function getPostsPerDay() {
		$days = ceil((TIME_NOW - $this->time) / 86400);
		if ($days <= 0) $days = 1;
		return $this->getPosts() / $days;
	}
	
	/**
	 * Activates the board style.
	 */
	public function initStyle() {
		if ($this->styleID && StyleHandler::getInstance()->getStyle()->styleID != $this->styleID) {
			StyleHandler::getInstance()->changeStyle($this->styleID, true);
		}
	}
	
	/**
	 * Returns true if the active user has subscribed this board.
	 * 
	 * @return	boolean
	 * @since	5.0
	 */
	public function isSubscribed() {
		if ($this->subscribed === null) {
			$objectTypeID = UserObjectWatchHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board');
			
			$this->subscribed = (UserObjectWatch::getUserObjectWatch($objectTypeID, WCF::getUser()->userID, $this->boardID) !== null);
		}
		
		return $this->subscribed;
	}
	
	/**
	 * Returns true if the current user may read all private threads in this board.
	 * 
	 * @return boolean
	 * @since 5.2
	 */
	public function canReadPrivateThreads() {
		// Guests can never read private threads.
		if (!WCF::getUser()->userID) {
			return false;
		}
		
		return !!$this->getModeratorPermission('canReadPrivateThread');
	}
	
	/**
	 * Inherits the board permissions.
	 * 
	 * @param	integer		$parentID
	 * @param	array		$permissions
	 */
	public static function inheritPermissions($parentID = null, &$permissions) {
		foreach (BoardCache::getInstance()->getChildIDs($parentID) as $boardID) {
			$board = BoardCache::getInstance()->getBoard($boardID);
			
			// inherit permissions from parent board
			if ($board->parentID && isset($permissions[$board->parentID])) {
				foreach ($permissions[$board->parentID] as $permissionName => $permissionValue) {
					if (!isset($permissions[$boardID][$permissionName])) {
						$permissions[$boardID][$permissionName] = $permissionValue;
					}
				}
			}
			
			self::inheritPermissions($boardID, $permissions);
		}
	}
	
	/**
	 * Returns the list of accessible boards ids. Excludes the list of board
	 * ids that may be accessible, but flagged as private. Use the companion
	 * function `getPrivateBoardIDs()` to retrieve these.
	 * 
	 * @param       string[]        $permissions    filters boards by given permissions
	 * @return      integer[]
	 */
	public static function getAccessibleBoardIDs(array $permissions = ['canViewBoard', 'canEnterBoard']) {
		$boardIDs = [];
		foreach (BoardCache::getInstance()->getBoards() as $board) {
			$result = true;
			foreach ($permissions as $permission) {
				$result = $result && $board->getPermission($permission);
			}
			
			if ($board->isPrivate && !$board->canReadPrivateThreads()) {
				$result = false;
			}
			
			if ($result) {
				$boardIDs[] = $board->boardID;
			}
		}
		
		return $boardIDs;
	}
	
	/**
	 * Returns the list of private boards ids. Includes only boards that
	 * are flagged as private, but the current user may access their own
	 * threads only.
	 *
	 * @param       string[]        $permissions    filters boards by given permissions
	 * @return      integer[]
	 */
	public static function getPrivateBoardIDs(array $permissions = ['canViewBoard', 'canEnterBoard']) {
		// Guests may never access private boards.
		if (!WCF::getUser()->userID) {
			return [];
		}
		
		$boardIDs = [];
		foreach (BoardCache::getInstance()->getBoards() as $board) {
			if (!$board->isPrivate || $board->canReadPrivateThreads()) {
				continue;
			}
			
			$result = true;
			foreach ($permissions as $permission) {
				$result = $result && $board->getPermission($permission);
			}
			
			if ($result) {
				$boardIDs[] = $board->boardID;
			}
		}
		
		return $boardIDs;
	}
	
	/**
	 * Filters a list of board ids by removing boards that are ignored by
	 * the current user, or excluded from search, if applicable.
	 * 
	 * @param       integer[]       $boardIDs
	 * @param       boolean         $checkSearchable
	 * @param       callable        $callback
	 * @return      integer[]
	 */
	public static function filterBoardIDs(array $boardIDs, $checkSearchable = false, callable $callback = null) {
		if (!WCF::getUser()->userID && !$checkSearchable && $callback === null) {
			return $boardIDs;
		}
		
		return array_filter($boardIDs, function($boardID) use ($checkSearchable, $callback) {
			$board = BoardCache::getInstance()->getBoard($boardID);
			if ($board === null || (WCF::getUser()->userID && $board->isIgnored()) || ($checkSearchable && !$board->searchable)) {
				return false;
			}
			
			if ($callback !== null) {
				return $callback($board);
			}
			
			return true;
		});
	}
	
	/**
	 * Returns a comma separated list of the ids of accessible boards.
	 * 
	 * @param	array		$permissions		filters boards by given permissions
	 * @return	integer[]
	 */
	public static function getAccessibleModeratorBoardIDs(array $permissions) {
		$boardIDs = [];
		foreach (BoardCache::getInstance()->getBoards() as $board) {
			$result = true;
			foreach ($permissions as $permission) {
				$result = $result && $board->getModeratorPermission($permission);
			}
			
			if ($result) {
				$boardIDs[] = $board->boardID;
			}
		}
		
		return $boardIDs;
	}
	
	/**
	 * Returns the default icon for provided type.
	 * 
	 * @param       string          $type           type identifier
	 * @return      string          default icon name
	 */
	public static function getDefaultIcon($type) {
		if (!isset(self::$defaultIcons[$type])) {
			throw new \InvalidArgumentException();
		}
		
		return self::$defaultIcons[$type];
	}
	
	/**
	 * Returns the list of type identifiers for icon states.
	 * 
	 * @return      string[]        list of type identifiers
	 */
	public static function getDefaultIconTypes() {
		return array_keys(self::$defaultIcons);
	}
}
