<?php
namespace wbb\acp\form;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardEditor;
use wbb\data\board\RealtimeBoardNodeList;
use wcf\data\user\group\UserGroup;
use wcf\data\user\group\UserGroupList;
use wcf\form\AbstractForm;
use wcf\system\acl\ACLHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\SystemException;
use wcf\system\exception\UserInputException;
use wcf\system\request\LinkHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;
use wcf\util\JSON;

/**
 * Shows the board permission edit form for user groups.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Acp\Form
 */
class BoardPermissionUserGroupForm extends AbstractForm {
	/**
	 * list of ACL category names and their readable name
	 * @var	string[]
	 */
	public $aclCategories = [];
	
	/**
	 * list of ACL options grouped by category name
	 * @var	string[][]
	 */
	public $aclOptions = [];
	
	/**
	 * list of ACL option ids
	 * @var	integer[]
	 */
	public $aclOptionIDs = [];
	
	/**
	 * @inheritDoc
	 */
	public $activeMenuItem = 'wcf.acp.menu.link.group';
	
	/**
	 * realtime board list object
	 * @var	RealtimeBoardNodeList
	 */
	public $boardList;
	
	/**
	 * @inheritDoc
	 */
	public $neededPermissions = ['admin.board.canEditBoard'];
	
	/**
	 * ACL object type id
	 * @var	integer
	 */
	public $objectTypeID = 0;
	
	/**
	 * @inheritDoc
	 */
	public $templateName = 'boardPermissionUserGroup';
	
	/**
	 * user group object
	 * @var	UserGroup
	 */
	public $userGroup;
	
	/**
	 * user group id
	 * @var	integer
	 */
	public $userGroupID = 0;
	
	/**
	 * list of user groups
	 * @var	UserGroup[]
	 */
	public $userGroups;
	
	/**
	 * list of option values
	 * 
	 * array(
	 * 	<aclOptionID> => array(
	 * 		<boardID> => <optionValue>
	 * 	)
	 * )
	 * 
	 * @var	integer[][]
	 */
	public $values = [];
	
	/**
	 * @var integer[]
	 */
	public $userGroupOptionValues = []; 
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		$this->readAclOptions();
		
		$userGroupList = new UserGroupList();
		$userGroupList->readObjects();
		$this->userGroups = $userGroupList->getObjects();
		uasort($this->userGroups, function(UserGroup $a, UserGroup $b) {
			return strcasecmp($a->getName(), $b->getName());
		});
		
		if (isset($_REQUEST['id'])) {
			$this->userGroupID = intval($_REQUEST['id']);
			$this->userGroup = UserGroup::getGroupByID($this->userGroupID);
			
			if (!$this->userGroup->groupID) {
				throw new IllegalLinkException();
			}
		}
		else {
			// redirect to 'Everyone' group
			HeaderUtil::redirect(LinkHandler::getInstance()->getLink('BoardPermissionUserGroup', [
				'application' => 'wbb',
				'id' => 1
			]));
			exit;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['values']) && is_array($this->values)) $this->values = $_POST['values'];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		parent::validate();
		
		if (!empty($this->values)) {
			// exceptions thrown here do not correspond to actual input fields, an exception should
			// not occur unless one befuddles the data, therefore we don't care that much at all
			foreach ($this->values as $aclOptionID => &$optionValues) {
				if (!in_array($aclOptionID, $this->aclOptionIDs)) {
					throw new UserInputException('values', 'invalid');
				}
				
				try {
					$optionValues = JSON::decode($optionValues);
				}
				catch (SystemException $e) {
					throw new UserInputException('values', 'invalid');
				}
				
				foreach ($optionValues as $boardID => &$optionValue) {
					$board = BoardCache::getInstance()->getBoard($boardID);
					if ($board === null || $board->isExternalLink()) {
						throw new UserInputException('values', 'invalid');
					}
					
					if ($optionValue != 0 && $optionValue != 1) {
						$optionValue = 0;
					}
				}
				unset($optionValue);
			}
			unset($optionValues);
		}
	}
	
	/**
	 * Reads associated ACL options.
	 */
	protected function readAclOptions() {
		$this->objectTypeID = ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board');
		foreach (ACLHandler::getInstance()->getOptions($this->objectTypeID) as $aclOption) {
			if (!isset($this->aclCategories[$aclOption->categoryName])) {
				$this->aclCategories[$aclOption->categoryName] = WCF::getLanguage()->get('wcf.acl.option.category.com.woltlab.wbb.board.'.$aclOption->categoryName);
			}
			
			if (!isset($this->aclOptions[$aclOption->categoryName])) {
				$this->aclOptions[$aclOption->categoryName] = [];
			}
			
			$this->aclOptions[$aclOption->categoryName][$aclOption->optionID] = WCF::getLanguage()->get('wcf.acl.option.com.woltlab.wbb.board.'.$aclOption->optionName);
			$this->aclOptionIDs[] = $aclOption->optionID;
		}
	}
	
	/**
	 * Reads group options default values of the selected user group.
	 */
	protected function readUserGroupOptionValues() {
		foreach (ACLHandler::getInstance()->getOptions($this->objectTypeID) as $aclOption) {
			$optionName = '';
			if (strpos($aclOption->categoryName, 'user.') === 0) {
				$optionName = 'user.board.' . $aclOption->optionName;
			}
			else if (strpos($aclOption->categoryName, 'mod.') === 0) {
				$optionName = 'mod.board.' . $aclOption->optionName;
			}
			
			if ($optionName) {
				$this->userGroupOptionValues[$aclOption->optionID] = intval($this->userGroup->getGroupOption($optionName));
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		$this->boardList = new RealtimeBoardNodeList();
		$this->boardList->readNodeTree();
		
		$this->readUserGroupOptionValues();
		
		if (empty($_POST)) {
			// read ACL values
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("optionID IN (?)", [$this->aclOptionIDs]);
			$conditions->add("groupID = ?", [$this->userGroup->groupID]);
			
			$sql = "SELECT	*
				FROM	wcf".WCF_N."_acl_option_to_group
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			while ($row = $statement->fetchArray()) {
				$optionID = $row['optionID'];
					
				if (!isset($this->values[$optionID])) {
					$this->values[$optionID] = [];
				}
				
				$board = BoardCache::getInstance()->getBoard($row['objectID']);
				if ($board !== null && !$board->isExternalLink()) {
					$this->values[$optionID][$row['objectID']] = $row['optionValue'];
				}
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		parent::save();
		
		// purge values of target user group
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("groupID = ?", [$this->userGroup->groupID]);
		$conditions->add("optionID IN (?)", [$this->aclOptionIDs]);
		
		$sql = "DELETE FROM	wcf".WCF_N."_acl_option_to_group
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		// insert new option values
		if (!empty($this->values)) {
			$sql = "INSERT INTO	wcf".WCF_N."_acl_option_to_group
						(optionID, objectID, groupID, optionValue)
				VALUES		(?, ?, ?, ?)";
			$statement = WCF::getDB()->prepareStatement($sql);
			
			WCF::getDB()->beginTransaction();
			foreach ($this->values as $aclOptionID => $optionValues) {
				foreach ($optionValues as $boardID => $optionValue) {
					$statement->execute([
						$aclOptionID,
						$boardID,
						$this->userGroup->groupID,
						$optionValue
					]);
				}
			}
			WCF::getDB()->commitTransaction();
		}
		
		// reset cache
		BoardEditor::resetPermissionCache();
		
		$this->saved();
		
		WCF::getTPL()->assign('success', true);
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'aclCategories' => $this->aclCategories,
			'aclOptions' => $this->aclOptions,
			'boardList' => $this->boardList->getNodeList(),
			'userGroup' => $this->userGroup,
			'userGroupID' => $this->userGroupID,
			'userGroups' => $this->userGroups,
			'values' => $this->values,
			'userGroupOptionValues' => $this->userGroupOptionValues
		]);
	}
}
