<?php
namespace wbb\acp\form;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardEditor;
use wbb\data\board\RealtimeBoardNodeList;
use wcf\data\user\User;
use wcf\data\user\UserProfile;
use wcf\form\AbstractForm;
use wcf\system\acl\ACLHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\SystemException;
use wcf\system\exception\UserInputException;
use wcf\system\request\LinkHandler;
use wcf\system\user\storage\UserStorageHandler;
use wcf\system\WCF;
use wcf\util\HeaderUtil;
use wcf\util\JSON;

/**
 * Shows the board permission edit form for users.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Acp\Form
 */
class BoardPermissionUserForm extends AbstractForm {
	/**
	 * list of ACL category names and their readable name
	 * @var	string[]
	 */
	public $aclCategories = [];
	
	/**
	 * list of ACL option values for all user groups the current user is member of
	 * @var	integer[][]
	 */
	public $aclGroupOptionValues = [];
	
	/**
	 * list of ACL options grouped by category name
	 * @var	string[][]
	 */
	public $aclOptions = [];
	
	/**
	 * list of ACL option ids
	 * @var	integer[]
	 */
	public $aclOptionIDs = [];
	
	/**
	 * @inheritDoc
	 */
	public $activeMenuItem = 'wcf.acp.menu.link.user.management';
	
	/**
	 * realtime board list object
	 * @var	RealtimeBoardNodeList
	 */
	public $boardList;
	
	/**
	 * @inheritDoc
	 */
	public $neededPermissions = ['admin.board.canEditBoard'];
	
	/**
	 * ACL object type id
	 * @var	integer
	 */
	public $objectTypeID = 0;
	
	/**
	 * @inheritDoc
	 */
	public $templateName = 'boardPermissionUser';
	
	/**
	 * user object
	 * @var	User
	 */
	public $user;
	
	/**
	 * user id
	 * @var	integer
	 */
	public $userID = 0;
	
	/**
	 * list of option values
	 * 
	 * array(
	 * 	<aclOptionID> => array(
	 * 		<boardID> => <optionValue>
	 * 	)
	 * )
	 * 
	 * @var	integer[][]
	 */
	public $values = [];
	
	/**
	 * @var integer[]
	 */
	public $userGroupOptionValues = [];
	
	/**
	 * @inheritDoc
	 */
	public function readParameters() {
		parent::readParameters();
		
		$this->readAclOptions();
		
		if (isset($_REQUEST['id'])) {
			$this->userID = intval($_REQUEST['id']);
			$this->user = new User($this->userID);
			if (!$this->user->userID) {
				throw new IllegalLinkException();
			}
		}
		else {
			// redirect to own user
			HeaderUtil::redirect(LinkHandler::getInstance()->getLink('BoardPermissionUser', [
				'application' => 'wbb',
				'id' => WCF::getUser()->userID
			]));
			exit;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['values']) && is_array($this->values)) $this->values = $_POST['values'];
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		parent::validate();
		
		if (!empty($this->values)) {
			// exceptions thrown here do not correspond to actual input fields, an exception should
			// not occur unless one befuddles the data, therefore we don't care that much at all
			foreach ($this->values as $aclOptionID => &$optionValues) {
				if (!in_array($aclOptionID, $this->aclOptionIDs)) {
					throw new UserInputException('values', 'invalid');
				}
				
				try {
					$optionValues = JSON::decode($optionValues);
				}
				catch (SystemException $e) {
					throw new UserInputException('values', 'invalid');
				}
				
				foreach ($optionValues as $boardID => $optionValue) {
					if (BoardCache::getInstance()->getBoard($boardID) === null) {
						throw new UserInputException('values', 'invalid');
					}
					
					if ($optionValue != 0 && $optionValue != 1) {
						throw new UserInputException('values', 'invalid');
					}
				}
			}
			unset($optionValues);
		}
	}
	
	/**
	 * Reads associated ACL options.
	 */
	protected function readAclOptions() {
		$this->objectTypeID = ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board');
		foreach (ACLHandler::getInstance()->getOptions($this->objectTypeID) as $aclOption) {
			if (!isset($this->aclCategories[$aclOption->categoryName])) {
				$this->aclCategories[$aclOption->categoryName] = WCF::getLanguage()->get('wcf.acl.option.category.com.woltlab.wbb.board.'.$aclOption->categoryName);
			}
			
			if (!isset($this->aclOptions[$aclOption->categoryName])) {
				$this->aclOptions[$aclOption->categoryName] = [];
			}
			
			$this->aclOptions[$aclOption->categoryName][$aclOption->optionID] = WCF::getLanguage()->get('wcf.acl.option.com.woltlab.wbb.board.'.$aclOption->optionName);
			$this->aclOptionIDs[] = $aclOption->optionID;
		}
	}
	
	/**
	 * Reads group options default values for user's user groups.
	 */
	protected function readUserGroupOptionValues() {
		$profile = new UserProfile($this->user);
		
		foreach (ACLHandler::getInstance()->getOptions($this->objectTypeID) as $aclOption) {
			$optionName = '';
			if (strpos($aclOption->categoryName, 'user.') === 0) {
				$optionName = 'user.board.' . $aclOption->optionName;
			}
			else if (strpos($aclOption->categoryName, 'mod.') === 0) {
				$optionName = 'mod.board.' . $aclOption->optionName;
			}
			
			if ($optionName) {
				$this->userGroupOptionValues[$aclOption->optionID] = intval($profile->getPermission($optionName));
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		$this->boardList = new RealtimeBoardNodeList();
		$this->boardList->readNodeTree();
		
		$this->readUserGroupOptionValues();
		
		// read ACL values for user's user groups
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("optionID IN (?)", [$this->aclOptionIDs]);
		$conditions->add("groupID IN (?)", [$this->user->getGroupIDs(true)]);
		
		$sql = "SELECT	*
			FROM	wcf".WCF_N."_acl_option_to_group
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		while ($row = $statement->fetchArray()) {
			$objectID = $row['objectID'];
			$optionID = $row['optionID'];
			
			if (!isset($this->aclGroupOptionValues[$optionID])) {
				$this->aclGroupOptionValues[$optionID] = [];
			}
			
			// grant is stronger than a deny
			if (!isset($this->aclGroupOptionValues[$optionID][$objectID]) || $this->aclGroupOptionValues[$optionID][$objectID] == 0) {
				$this->aclGroupOptionValues[$optionID][$row['objectID']] = $row['optionValue'];
			}
		}
		
		if (empty($_POST)) {
			// read ACL values
			$conditions = new PreparedStatementConditionBuilder();
			$conditions->add("optionID IN (?)", [array_keys($this->aclOptionIDs)]);
			$conditions->add("userID = ?", [$this->user->userID]);
			
			$sql = "SELECT	*
				FROM	wcf".WCF_N."_acl_option_to_user
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			while ($row = $statement->fetchArray()) {
				$optionID = $row['optionID'];
				
				if (!isset($this->values[$optionID])) {
					$this->values[$optionID] = [];
				}
				
				if (BoardCache::getInstance()->getBoard($row['objectID']) !== null) {
					$this->values[$optionID][$row['objectID']] = $row['optionValue'];
				}
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		parent::save();
		
		// purge values of target user
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("userID = ?", [$this->user->userID]);
		$conditions->add("optionID IN (?)", [$this->aclOptionIDs]);
		
		$sql = "DELETE FROM	wcf".WCF_N."_acl_option_to_user
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		// insert new option values
		if (!empty($this->values)) {
			$sql = "INSERT INTO	wcf".WCF_N."_acl_option_to_user
						(optionID, objectID, userID, optionValue)
				VALUES		(?, ?, ?, ?)";
			$statement = WCF::getDB()->prepareStatement($sql);
			
			WCF::getDB()->beginTransaction();
			foreach ($this->values as $aclOptionID => $optionValues) {
				foreach ($optionValues as $boardID => $optionValue) {
					$statement->execute([
						$aclOptionID,
						$boardID,
						$this->user->userID,
						$optionValue
					]);
				}
			}
			WCF::getDB()->commitTransaction();
		}
		
		// reset cache
		BoardEditor::resetPermissionCache();
		
		// reset affected user's storage value
		UserStorageHandler::getInstance()->reset([$this->user->userID], 'wbbBoardPermissions');
		
		$this->saved();
		
		WCF::getTPL()->assign('success', true);
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'aclCategories' => $this->aclCategories,
			'aclGroupOptionValues' => $this->aclGroupOptionValues,
			'aclOptions' => $this->aclOptions,
			'boardList' => $this->boardList->getNodeList(),
			'user' => $this->user,
			'userID' => $this->userID,
			'values' => $this->values,
			'userGroupOptionValues' => $this->userGroupOptionValues
		]);
	}
}
