<?php
namespace wbb\acp\form;
use wbb\data\board\BoardCache;
use wbb\data\board\BoardEditor;
use wbb\data\board\RealtimeBoardNodeList;
use wcf\form\AbstractForm;
use wcf\system\acl\ACLHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;
use wcf\util\ArrayUtil;

/**
 * Shows the board permission copy form.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Acp\Form
 */
class BoardPermissionCopyForm extends AbstractForm {
	/**
	 * @inheritDoc
	 */
	public $activeMenuItem = 'wbb.acp.menu.link.board.list';
	
	/**
	 * source board id
	 * @var	integer
	 */
	public $boardID = 0;
	
	/**
	 * target board ids
	 * @var	integer[]
	 */
	public $boardIDs = [];
	
	/**
	 * list of boards
	 * @var	RealtimeBoardNodeList
	 */
	public $boardList;
	
	/**
	 * copy moderator permissions
	 * @var	boolean
	 */
	public $copyModerators = false;
	
	/**
	 * copy user permissions
	 * @var	boolean
	 */
	public $copyPermissions = false;
	
	/**
	 * @inheritDoc
	 */
	public $neededPermissions = ['admin.board.canEditBoard'];
	
	/**
	 * @inheritDoc
	 */
	public $templateName = 'boardPermissionCopy';
	
	/**
	 * @inheritDoc
	 */
	public function readFormParameters() {
		parent::readFormParameters();
		
		if (isset($_POST['boardID'])) $this->boardID = intval($_POST['boardID']);
		if (isset($_POST['boardIDs']) && is_array($_POST['boardIDs'])) $this->boardIDs = ArrayUtil::toIntegerArray($_POST['boardIDs']);
		if (isset($_POST['copyModerators'])) $this->copyModerators = true;
		if (isset($_POST['copyPermissions'])) $this->copyPermissions = true;
	}
	
	/**
	 * @inheritDoc
	 */
	public function validate() {
		parent::validate();
		
		// check source board
		if (BoardCache::getInstance()->getBoard($this->boardID) === null) {
			throw new UserInputException('boardID');
		}
		
		// check target board ids
		foreach ($this->boardIDs as $key => $boardID) {
			if ($boardID == $this->boardID) {
				// cannot copy to itself
				unset($this->boardIDs[$key]);
			}
			
			if (BoardCache::getInstance()->getBoard($boardID) === null) {
				unset($this->boardIDs[$key]);
			}
		}
		
		if (empty($this->boardIDs)) {
			throw new UserInputException('boardIDs');
		}
		
		// preserve proper indices
		$this->boardIDs = array_merge($this->boardIDs, []);
		
		if (!$this->copyModerators && !$this->copyPermissions) {
			throw new UserInputException('copyAction');
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		$this->boardList = new RealtimeBoardNodeList();
		$this->boardList->readNodeTree();
		
		/** @noinspection PhpUndefinedMethodInspection */
		if (!$this->boardList->getNodeList()->hasChildren()) {
			throw new IllegalLinkException();
		}
		
		if (empty($_POST)) {
			$this->copyModerators = true;
			$this->copyPermissions = true;
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function save() {
		parent::save();
		
		// fetch ACL option ids
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("objectTypeID = ?", [ACLHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.board')]);
		if (!$this->copyModerators) {
			$conditions->add("categoryName LIKE ?", ['user.%']);
		}
		else if (!$this->copyPermissions) {
			$conditions->add("categoryName LIKE ?", ['mod.%']);
		}
		
		$sql = "SELECT	optionID
			FROM	wcf".WCF_N."_acl_option
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		$optionIDs = $statement->fetchAll(\PDO::FETCH_COLUMN);
		
		// purge permissions of target boards
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("optionID IN (?)", [$optionIDs]);
		$conditions->add("objectID IN (?)", [$this->boardIDs]);
		
		$sql = "DELETE FROM	wcf".WCF_N."_acl_option_to_group
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		$sql = "DELETE FROM	wcf".WCF_N."_acl_option_to_user
			".$conditions;
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute($conditions->getParameters());
		
		// copy permissions
		$conditions = new PreparedStatementConditionBuilder();
		$conditions->add("optionID IN (?)", [$optionIDs]);
		$conditions->add("objectID = ?", [$this->boardID]);
		
		WCF::getDB()->beginTransaction();
		foreach ($this->boardIDs as $boardID) {
			$sql = "INSERT INTO	wcf".WCF_N."_acl_option_to_group
						(optionID, objectID, groupID, optionValue)
				SELECT		optionID, ".$boardID.", groupID, optionValue
				FROM		wcf".WCF_N."_acl_option_to_group
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
			
			$sql = "INSERT INTO	wcf".WCF_N."_acl_option_to_user
						(optionID, objectID, userID, optionValue)
				SELECT		optionID, ".$boardID.", userID, optionValue
				FROM		wcf".WCF_N."_acl_option_to_user
				".$conditions;
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute($conditions->getParameters());
		}
		WCF::getDB()->commitTransaction();
		
		// reset cache
		BoardEditor::resetPermissionCache();
		
		$this->saved();
		
		WCF::getTPL()->assign('success', true);
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		WCF::getTPL()->assign([
			'boardID' => $this->boardID,
			'boardIDs' => $this->boardIDs,
			'boardList' => $this->boardList->getNodeList(),
			'copyModerators' => $this->copyModerators,
			'copyPermissions' => $this->copyPermissions
		]);
	}
}
