/**
 * Class and function collection for WBB ACP
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 */

/**
 * Initialize WBB.ACP namespace
 */
if (!WBB) var WBB = { };
WBB.ACP = { };

/**
 * Namespace for board-related actions
 */
WBB.ACP.Board = { };

/**
 * Collapsible implementation for board list.
 */
WBB.ACP.Board.Collapsible = WCF.Collapsible.SimpleRemote.extend({
	/**
	 * @see	WCF.Collapsible.SimpleRemote._getContainers()
	 */
	_getContainers: function() {
		return $('#boardNodeList .jsCollapsibleCategory');
	},
	
	/**
	 * @see	WCF.Collapsible.SimpleRemote._getTarget()
	 */
	_getTarget: function(containerID) {
		return this._containers[containerID].children('ol').first();
	},
	
	/**
	 * @see	WCF.Collapsible.SimpleRemote._getButtonContainer()
	 */
	_getButtonContainer: function(containerID) {
		return this._containers[containerID].children('span').first();
	},
	
	/**
	 * @see	WCF.Collapsible.SimpleRemote._getObjectID()
	 */
	_getObjectID: function(containerID) {
		return this._containers[containerID].data('objectID');
	}
});

/**
 * Handles copying (cloning) of boards.
 * 
 * @param	integer		boardID
 */
WBB.ACP.Board.Copy = Class.extend({
	/**
	 * board id
	 * @var	integer
	 */
	_boardID: 0,
	
	/**
	 * Initializes the WBB.ACP.Board.Copy class.
	 * 
	 * @param	integer		boardID
	 */
	init: function(boardID) {
		this._boardID = boardID;
		
		$('.jsButtonBoardCopy').click($.proxy(this._click, this));
	},
	
	/**
	 * Handles clicks on the 'Copy Board' button.
	 */
	_click: function() {
		var $template = $('<dl class="wide"><dt /><dd><label><input type="checkbox" id="boardCopyRecursive" value="1" /> ' + WCF.Language.get('wbb.acp.board.copy.recursive') + '</label><small>' + WCF.Language.get('wbb.acp.board.copy.recursive.description') + '</small></dd></dl>');
		
		WCF.System.Confirmation.show(WCF.Language.get('wbb.acp.board.copy.confirmMessage'), $.proxy(function(action) {
			if (action === 'confirm') {
				var $recursive = $('#boardCopyRecursive').is(':checked');
				
				new WCF.Action.Proxy({
					autoSend: true,
					data: {
						actionName: 'copy',
						className: 'wbb\\data\\board\\BoardAction',
						objectIDs: [ this._boardID ],
						parameters: {
							recursive: $recursive
						}
					},
					success: function(data) {
						window.location = data.returnValues.redirectURL;
					}
				});
			}
		}, this), '', $template);
	}
});

/**
 * Handles deleting boards.
 */
WBB.ACP.Board.Delete = WCF.Action.NestedDelete.extend({
	/**
	 * @see	WCF.Action.NestedDelete._didTriggerEffect
	 */
	_didTriggerEffect: function(element) {
		if (!this._containers.length) {
			window.location.reload();
		}
	}
});

/**
 * Namespace for board permission management.
 */
WBB.ACP.Board.Permission = { };

/**
 * Manages board permissions for user groups.
 * 
 * @param	integer		groupID
 * @param	object<object>	values
 */
WBB.ACP.Board.Permission.UserGroup = Class.extend({
	/**
	 * ACL container object
	 * @var	jQuery
	 */
	_aclContainer: null,
	
	/**
	 * active ACL option id
	 * @var	int
	 */
	_aclOptionID: 0,
	
	/**
	 * ACL selection dropdown object
	 * @var	jQuery
	 */
	_aclSelection: null,
	
	/**
	 * list of available board ids
	 * @var	int[]
	 */
	_boardIDs: [],
	
	/**
	 * current user group id
	 * @var	int
	 */
	_groupID: 0,
	
	/**
	 * original values for ACL options
	 * @var	Object<Object>
	 */
	_originalValues: { },
	
	/**
	 * action proxy
	 * @var	WCF.Action.Proxy
	 */
	_proxy: null,
	
	/**
	 * current values for user group options
	 * @var	object
	 */
	_userGroupOptionValues: { },
	
	/**
	 * current values for ACL options
	 * @var	object<object>
	 */
	_values: { },
	
	_elements: {},
	
	_iconClassNames: {},
	
	/**
	 * Initializes the WBB.ACP.Board.Permission.UserGroup class.
	 * 
	 * @param	integer		groupID
	 * @param	object<object>	values
	 * @param	object  	userGroupOptionValues
	 */
	init: function(groupID, values, userGroupOptionValues) {
		this._aclContainer = $('#aclContainer');
		this._boardIDs = [];
		this._elements = {};
		this._iconClassNames = {
			grant: 'icon icon16 fa-check-circle green',
			deny: 'icon icon16 fa-ban red'
		};
		this._groupID = groupID;
		this._userGroupOptionValues = userGroupOptionValues;
		this._values = values;
		this._originalValues = $.extend(true, { }, this._values);
		this._proxy = new WCF.Action.Proxy({
			success: $.proxy(this._success, this)
		});
		
		this._aclSelection = $('#aclSelection').change($.proxy(this._changeACL, this));
		
		elBySelAll('li', this._aclContainer[0], (function (listItem) {
			var boardId = ~~elData(listItem, 'object-id');
			if (boardId !== 0) this._boardIDs.push(boardId);
			
			var grant = elById('grant' + boardId);
			grant.addEventListener('change', (function () {
				if (grant.checked) deny.checked = false;
				
				this._updateMarking(boardId);
			}).bind(this));
			
			var deny = elById('deny' + boardId);
			deny.addEventListener('change', (function () {
				if (deny.checked) grant.checked = false;
				
				this._updateMarking(boardId);
			}).bind(this));
			
			this._elements[boardId] = {
				grant: grant,
				deny: deny
			};
		}).bind(this));
		
		this._updateMarkings();
		
		this._changeACL();
		
		this._aclContainer.parents('form').submit($.proxy(this._submit, this));
		$('#userGroupSwitch').find('a').click($.proxy(this._switch, this));
	},
	
	/**
	 * Changes the ACL option selection.
	 */
	_changeACL: function() {
		this._saveValues();
		
		this._aclOptionID = ~~this._aclSelection.val();
		
		// reset all values
		elBySelAll('input:checked', this._aclContainer[0], function(checkbox) { checkbox.checked = false; });
		
		if (this._aclOptionID === 0) {
			return;
		}
		
		if (this._values[this._aclOptionID]) {
			Object.keys(this._values[this._aclOptionID]).forEach((function (boardId) {
				var optionValue = ~~this._values[this._aclOptionID][boardId];
				
				if (optionValue === 0) {
					this._elements[~~boardId].deny.checked = true;
				}
				else if (optionValue === 1) {
					this._elements[~~boardId].grant.checked = true;
				}
			}).bind(this));
		}
		
		this._updateMarkings();
		
		this._aclContainer.show();
	},
	
	/**
	 * Updates marking of all boards.
	 */
	_updateMarkings: function() {
		for (var $i = 0; $i < this._boardIDs.length; $i++) {
			this._updateMarking(this._boardIDs[$i], true);
		}
	},
	
	/**
	 * Updates icon hinting based upon inheritance rules.
	 * 
	 * @param	integer		boardID
	 * @param	boolean		skipChildren
	 */
	_updateMarking: function(boardID, skipChildren) {
		var boardNode = elById('boardNode' + boardID);
		var icon = elBySel('.sortableNodeLabel > .icon', boardNode);
		
		var defaultValue = null;
		if (this._elements[boardID].deny.checked) {
			defaultValue = false;
			
			icon.className = this._iconClassNames.deny;
		}
		else {
			if (this._elements[boardID].grant.checked) {
				defaultValue = true;
			}
			
			icon.className = this._iconClassNames.grant;
		}
		
		if (defaultValue === null) {
			// check parents
			var parent = boardNode, requireUpdate = true;
			while (parent = parent.parentNode) {
				if (parent.nodeName !== 'LI') continue;
				
				var parentId = ~~elData(parent, 'object-id');
				if (parentId > 0) {
					if (this._elements[parentId].grant.checked) {
						icon.className = this._iconClassNames.grant;
						
						defaultValue = true;
						requireUpdate = false;
					}
					else if (this._elements[parentId].deny.checked) {
						icon.className = this._iconClassNames.deny;
						
						defaultValue = false;
						requireUpdate = false;
					}
					
					if (!requireUpdate) break;
				}
			}
			
			if (requireUpdate) {
				if (this._userGroupOptionValues[this._aclOptionID]) {
					icon.className = this._iconClassNames.grant;
					defaultValue = true;
				}
				else {
					icon.className = this._iconClassNames.deny;
					defaultValue = false;
				}
			}
		}
		
		if (skipChildren !== true) {
			elBySelAll('li', boardNode, (function (node) {
				var nodeId = ~~elData(node, 'object-id');
				if (nodeId > 0) {
					if (!this._elements[nodeId].grant.checked && !this._elements[nodeId].deny.checked) {
						icon = elBySel('.sortableNodeLabel > .icon', node);
						icon.className = this._iconClassNames[(defaultValue ? 'grant' : 'deny')];
					}
				}
			}).bind(this));
		}
	},
	
	/**
	 * Saves changes to current ACL option.
	 */
	_saveValues: function() {
		if (this._aclOptionID == 0) {
			return;
		}
		
		this._values[this._aclOptionID] = { };
		
		for (var $i = 0; $i < this._boardIDs.length; $i++) {
			var $boardID = this._boardIDs[$i];
			
			if ($('#grant' + $boardID).is(':checked')) {
				this._values[this._aclOptionID][$boardID] = 1;
			}
			else if ($('#deny' + $boardID).is(':checked')) {
				this._values[this._aclOptionID][$boardID] = 0;
			}
		}
	},
	
	/**
	 * Stores values as JSON-encoded string prior to submit.
	 * 
	 * Using JSON allows us to compress settings for many forums in one value per ACL option. This
	 * prevents issues with various PHP and Suhosin settings which enforce a low variable limit.
	 * In theory one can configure up to 5k forums without being affected by any default limits.
	 */
	_submit: function() {
		this._saveValues();
		
		var $formSubmit = $('.formSubmit');
		$.each(this._values, function(aclOptionID, values) {
			$('<input type="hidden" name="values[' + aclOptionID + ']" />').val(JSON.stringify(values)).appendTo($formSubmit);
		});
	},
	
	/**
	 * Switches to another user group for editing, will prevent navigation and ask for saving of
	 * data in case something was changed but not yet saved.
	 * 
	 * @param	object		event
	 */
	_switch: function(event) {
		var $link = $(event.currentTarget);
		if ($link.data('groupID') == this._groupID) {
			return;
		}
		
		// compare stored and original values
		this._saveValues();
		
		if (JSON.stringify(this._values) == JSON.stringify(this._originalValues)) {
			window.location = $link.data('link');
			return;
		}
		else {
			WCF.System.Confirmation.show(WCF.Language.get('wbb.acp.board.permission.unsavedChanges'), function(action) {
				if (action === 'confirm') {
					window.location = $link.data('link');
				}
			});
		}
	}
});

/**
 * Manages board permissions for users.
 * 
 * @param	integer		userID
 * @param	object<object>	values
 * @param	object<object>	groupValues
 */
WBB.ACP.Board.Permission.User = WBB.ACP.Board.Permission.UserGroup.extend({
	/**
	 * list of cumulative ACL option values for all groups this user is part of
	 * @var	object<object>
	 */
	_groupValues: { },
	
	/**
	 * @see	WBB.ACP.Board.Permission.UserGroup.init()
	 */
	init: function(userID, values, groupValues, userGroupOptionValues) {
		this._groupValues = groupValues;
		
		this._super(0, values, userGroupOptionValues);
	},
	
	/**
	 * @see	WBB.ACP.Board.Permission.UserGroup._updateMarking()
	 */
	_updateMarking: function(boardID, skipChildren) {
		var boardNode = elById('boardNode' + boardID);
		var icon = elBySel('.sortableNodeLabel > .icon', boardNode);
		
		var defaultValue = null;
		if (this._elements[boardID].deny.checked) {
			defaultValue = false;
			
			icon.className = this._iconClassNames.deny;
		}
		else {
			if (this._elements[boardID].grant.checked) {
				defaultValue = true;
			}
			
			icon.className = this._iconClassNames.grant;
		}
		
		if (defaultValue === null) {
			// check parents
			var parent = boardNode, requireUpdate = true;
			while (parent = parent.parentNode) {
				if (parent.nodeName !== 'LI') continue;
				
				var parentId = ~~elData(parent, 'object-id');
				if (parentId > 0) {
					if (this._elements[parentId].grant.checked) {
						icon.className = this._iconClassNames.grant;
						
						defaultValue = true;
						requireUpdate = false;
					}
					else if (this._elements[parentId].deny.checked) {
						icon.className = this._iconClassNames.deny;
						
						defaultValue = false;
						requireUpdate = false;
					}
					
					if (!requireUpdate) break;
				}
			}
			
			if (requireUpdate) {
				// check group values
				if (this._groupValues[this._aclOptionID] !== undefined && this._groupValues[this._aclOptionID][boardID] !== undefined) {
					if (this._groupValues[this._aclOptionID][boardID]) {
						icon.className = this._iconClassNames.grant;
						
						defaultValue = true;
						requireUpdate = false;
					}
					else {
						icon.className = this._iconClassNames.deny;
						
						defaultValue = true;
						requireUpdate = false;
					}
				}
				
				if (requireUpdate) {
					if (this._userGroupOptionValues[this._aclOptionID]) {
						icon.className = this._iconClassNames.grant;
						defaultValue = true;
					}
					else {
						icon.className = this._iconClassNames.deny;
						defaultValue = false;
					}
				}
			}
		}
		
		if (skipChildren !== true) {
			elBySelAll('li', boardNode, (function (node) {
				var nodeId = ~~elData(node, 'object-id');
				if (nodeId > 0) {
					if (!this._elements[nodeId].grant.checked && !this._elements[nodeId].deny.checked) {
						icon = elBySel('.sortableNodeLabel > .icon', node);
						icon.className = this._iconClassNames[(defaultValue ? 'grant' : 'deny')];
					}
				}
			}).bind(this));
		}
	}
});

/**
 * Namespace for rss feed-related actions
 */
WBB.ACP.RSSFeed = { };

/**
 * Handles the label lists when creating a rss feed.
 */
WBB.ACP.RSSFeed.LabelList = Class.extend({
	/**
	 * Initializes a new WBB.ACP.RSSFeed.LabelList object.
	 */
	init: function(labelIDs) {
		new WCF.Label.Chooser(labelIDs, '#formContainer');
		
		$('#boardID').change($.proxy(this._change, this));
		
		this._change();
	},
	
	/**
	 * Toggles the available label groups when changing the selected boards.
	 */
	_change: function() {
		var $boardID = parseInt($('#boardID').val());
		
		$('.labelList').each(function(index, element) {
			if ($.makeArray($(element).data('boardIds')).indexOf($boardID) === -1) {
				$(element).parents('dl:eq(0)').hide();
			}
			else {
				$(element).parents('dl:eq(0)').show();
			}
		});
	}
});
