<?php
namespace wbb\page;
use wbb\data\board\Board;
use wbb\data\modification\log\ViewableThreadModificationLogList;
use wbb\data\post\Post;
use wbb\data\thread\SimilarThreadList;
use wbb\system\cache\runtime\PostRuntimeCache;
use wbb\system\label\object\ThreadLabelObjectHandler;
use wbb\system\option\ThreadFormOptionHandler;
use wcf\data\attachment\Attachment;
use wcf\data\like\object\LikeObject;
use wcf\data\smiley\SmileyCache;
use wcf\data\user\UserProfile;
use wcf\system\attachment\AttachmentHandler;
use wcf\system\clipboard\ClipboardHandler;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\language\LanguageFactory;
use wcf\system\like\LikeHandler;
use wcf\system\message\embedded\object\MessageEmbeddedObjectManager;
use wcf\system\message\quote\MessageQuoteManager;
use wcf\system\poll\PollManager;
use wcf\system\reaction\ReactionHandler;
use wcf\system\MetaTagHandler;
use wcf\system\user\signature\SignatureCache;
use wcf\system\WCF;
use wcf\util\StringUtil;

/**
 * Shows the thread page.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Forum\Page
 */
class ThreadPage extends AbstractThreadPage {
	/**
	 * like data for posts
	 * @var	LikeObject[]
	 */
	public $likeData = [];
	
	/**
	 * list of similar threads
	 * @var	SimilarThreadList
	 */
	public $similarThreadList;
	
	/**
	 * modification log list object
	 * @var	ViewableThreadModificationLogList
	 */
	public $modificationLogList;
	
	/**
	 * thread form option handler object
	 * @var	ThreadFormOptionHandler
	 */
	public $optionHandler;
	
	/**
	 * True if the best answer in the topic is on the current page.
	 * @var boolean
	 */
	public $bestAnswerOnCurrentPage = false;
	
	/**
	 * @inheritDoc
	 */
	public function readData() {
		parent::readData();
		
		foreach ($this->objectList as $viewablePost) {
			PostRuntimeCache::getInstance()->cachePost($viewablePost->getDecoratedObject());
		}
		
		// fetch likes
		if (MODULE_LIKE) {
			$postIDs = [];
			foreach ($this->objectList as $post) {
				$postIDs[] = $post->postID;
			}
			$objectType = ReactionHandler::getInstance()->getObjectType('com.woltlab.wbb.likeablePost');
			ReactionHandler::getInstance()->loadLikeObjects($objectType, $postIDs);
			$this->likeData = ReactionHandler::getInstance()->getLikeObjects($objectType);
		}
		
		// fetch labels
		if ($this->thread->hasLabels) {
			$assignedLabels = ThreadLabelObjectHandler::getInstance()->getAssignedLabels([$this->thread->threadID]);
			if (isset($assignedLabels[$this->thread->threadID])) {
				foreach ($assignedLabels[$this->thread->threadID] as $label) {
					$this->thread->addLabel($label);
				}
			}
		}
		
		$userIDs = [];
		foreach ($this->objectList as $post) {
			if ($post->userID) {
				$userIDs[] = $post->userID;
			}
		}
		
		// fetch special trophies
		if (MODULE_TROPHY) {
			if (!empty($userIDs)) {
				UserProfile::prepareSpecialTrophies(array_unique($userIDs));
			}
		}
		
		if (MODULE_USER_SIGNATURE) {
			if (!empty($userIDs)) {
				SignatureCache::getInstance()->cacheUserSignature($userIDs);
			}
		}
		
		// init quote objects
		$postIDs = [];
		foreach ($this->objectList as $post) {
			$postIDs[] = $post->postID;
		}
		MessageQuoteManager::getInstance()->initObjects('com.woltlab.wbb.post', $postIDs);
		
		// get similar threads
		if (WBB_THREAD_ENABLE_SIMILAR_THREADS) {
			$boardIDs = Board::filterBoardIDs(Board::getAccessibleBoardIDs(['canViewBoard', 'canEnterBoard', 'canReadThread']));
			if (!empty($boardIDs)) {
				$this->similarThreadList = new SimilarThreadList();
				$this->similarThreadList->getConditionBuilder()->add('thread_similar.threadID = ?', [$this->threadID]);
				$this->similarThreadList->getConditionBuilder()->add('thread.boardID IN (?)', [$boardIDs]);
				$this->similarThreadList->getConditionBuilder()->add('thread.isDisabled = 0 AND thread.isDeleted = 0');
				$this->similarThreadList->readObjects();
			}
		}
		
		// add meta tags
		MetaTagHandler::getInstance()->addTag('og:title', 'og:title', $this->thread->topic . ' - ' . WCF::getLanguage()->get(PAGE_TITLE), true);
		MetaTagHandler::getInstance()->addTag('og:url', 'og:url', $this->thread->getLink(), true);
		MetaTagHandler::getInstance()->addTag('og:type', 'og:type', 'article', true);
		$posts = $this->objectList->getObjects();
		if (isset($posts[$this->thread->firstPostID])) {
			MetaTagHandler::getInstance()->addTag('og:description', 'og:description', StringUtil::decodeHTML(StringUtil::stripHTML($posts[$this->thread->firstPostID]->getExcerpt())), true);
			
			// add image attachments
			$i = 0;
			$attachmentList = $posts[$this->thread->firstPostID]->getAttachments();
			MessageEmbeddedObjectManager::getInstance()->setActiveMessage('com.woltlab.wbb.post', $this->thread->firstPostID);
			
			/** @var Attachment[] $attachments */
			$attachments = array_merge($attachmentList !== null ? $attachmentList->getGroupedObjects($this->thread->firstPostID) : [], MessageEmbeddedObjectManager::getInstance()->getObjects('com.woltlab.wcf.attachment'));
			foreach ($attachments as $attachment) {
				if ($attachment->showAsImage() && $attachment->width >= 200 && $attachment->height >= 200) {
					MetaTagHandler::getInstance()->addTag('og:image' . $i, 'og:image', $attachment->getLink(), true);
					MetaTagHandler::getInstance()->addTag('og:image:width' . $i, 'og:image:width', $attachment->width, true);
					MetaTagHandler::getInstance()->addTag('og:image:height' . $i, 'og:image:height', $attachment->height, true);
					$i++;
				}
			}
		}
		else {
			$post = new Post($this->thread->firstPostID);
			MetaTagHandler::getInstance()->addTag('og:description', 'og:description', StringUtil::decodeHTML(StringUtil::stripHTML($post->getExcerpt())), true);
		}
		
		// add thread tags as keywords
		if (!empty($this->tags)) {
			$keywords = '';
			foreach ($this->tags as $tag) {
				if (!empty($keywords)) $keywords .= ', ';
				$keywords .= $tag->name;
			}
			MetaTagHandler::getInstance()->addTag('keywords', 'keywords', $keywords);
		}
		
		// load modification log entries
		$count = count($this->objectList);
		if (WBB_THREAD_ENABLE_MODERATION_THREAD_LOG && $count > 0) {
			$endTime = TIME_NOW;
			if ($count > 1) {
				$this->objectList->seek($count - 1);
				if ($this->objectList->current()->time < $this->thread->lastPostTime) {
					$conditionBuilder = new PreparedStatementConditionBuilder();
					$conditionBuilder->add("threadID = ?", [$this->threadID]);
					$conditionBuilder->add("time > ?", [$this->objectList->current()->time]);
					
					if (!$this->board->getModeratorPermission('canEnablePost')) {
						$conditionBuilder->add("isDisabled = ?", [0]);
					}
					
					if (!WBB_THREAD_ENABLE_DELETED_POST_NOTE && !$this->board->getModeratorPermission('canReadDeletedPost')) {
						$conditionBuilder->add("isDeleted = ?", [0]);
					}
					
					$sql = "SELECT  MIN(time)
						FROM    wbb".WCF_N."_post
						". $conditionBuilder;
					$statement = WCF::getDB()->prepareStatement($sql, 1);
					$statement->execute($conditionBuilder->getParameters());
					$endTime = $statement->fetchSingleColumn() - 1;
				}
			}
			$this->objectList->rewind();
			
			$this->modificationLogList = new ViewableThreadModificationLogList($this->thread->threadID);
			if ($count > 0) {
				$this->modificationLogList->getConditionBuilder()->add("modification_log.time BETWEEN ? AND ?", [$this->objectList->current()->time, $endTime]);
			}
			else {
				// The list may be empty for broken threads or threads that only contain
				// posts from ignored users.
				$this->modificationLogList->getConditionBuilder()->add("1=0");
			}
			$this->modificationLogList->readObjects();
		}
		
		if ($count > 0 && $this->thread->getFirstPost() !== null) {
			$this->optionHandler = new ThreadFormOptionHandler(false);
			$this->optionHandler->setPost($this->thread->getFirstPost());
		}
		
		if ($this->board->enableBestAnswer && $this->thread->hasBestAnswer()) {
			if (isset($posts[$this->thread->bestAnswerPostID])) {
				$this->bestAnswerOnCurrentPage = true;
			}
		}
	}
	
	/**
	 * @inheritDoc
	 */
	public function assignVariables() {
		parent::assignVariables();
		
		MessageQuoteManager::getInstance()->assignVariables();
		
		WCF::getTPL()->assign([
			'defaultSmilies' => SmileyCache::getInstance()->getCategorySmilies(),
			'hasMarkedItems' => ClipboardHandler::getInstance()->hasMarkedItems(ClipboardHandler::getInstance()->getObjectTypeID('com.woltlab.wbb.post')),
			'likeData' => $this->likeData,
			'similarThreadList' => $this->similarThreadList,
			'permissionCanUseSmilies' => 'user.message.canUseSmilies',
			'searchThread' => true,
			'modificationLogList' => $this->modificationLogList,
			'threadFormOptions' => $this->optionHandler ? $this->optionHandler->getOptions() : [],
			'bestAnswerOnCurrentPage' => $this->bestAnswerOnCurrentPage,
		]);
		
		if ($this->thread->canReply()) {
			// attachments within quick reply
			$tmpHash = StringUtil::getRandomID();
			
			WCF::getTPL()->assign([
				'attachmentHandler' => new AttachmentHandler('com.woltlab.wbb.post', 0, $tmpHash, $this->board->boardID),
				'attachmentObjectID' => 0,
				'attachmentObjectType' => 'com.woltlab.wbb.post',
				'attachmentParentObjectID' => $this->board->boardID,
				'tmpHash' => $tmpHash
			]);
			
			// poll within quick reply
			if ($this->thread->canUsePoll(true)) {
				PollManager::getInstance()->setObject('com.woltlab.wbb.post', 0);
				PollManager::getInstance()->assignVariables();
			}
		}
	}
}
