/**
 * Loads new or collapsed posts.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Ui/Post/Loader
 */
define(['Ajax', 'Dom/ChangeListener', 'Dom/Util', 'WoltLabSuite/Core/Controller/Clipboard'], function(Ajax, DomChangeListener, DomUtil, ControllerClipboard) {
	"use strict";
	
	var _postInlineEditor = null;
	var _postManager = null;
	
	/**
	 * @exports     WoltLabSuite/Forum/Ui/Post/Loader
	 */
	return {
		/**
		 * Initializes the post loader.
		 * 
		 * @param       {WoltLabSuite/Forum/Ui/Post/Manager}           postManager             post manager instance
		 * @param       {WoltLabSuite/Forum/Ui/Post/InlineEditor=}     postInlineEditor        post inline editor instance
		 */
		init: function(postManager, postInlineEditor) {
			_postManager = postManager;
			_postInlineEditor = postInlineEditor;
			
			var callback = this._click.bind(this);
			elBySelAll('.wbbPostDeleted', null, function(post) {
				post.addEventListener(WCF_CLICK_EVENT, callback);
			});
		},
		
		/**
		 * Handles clicks on collapsed posts.
		 * 
		 * @param       {Event}         event           event object
		 * @protected
		 */
		_click: function(event) {
			var nodeName = event.target.nodeName;
			if (nodeName === 'A' || nodeName === 'INPUT') {
				return;
			}
			
			var postId = elData(event.currentTarget, 'object-id');
			this.load([postId]);
		},
		
		/**
		 * Loads a list of posts by id.
		 * 
		 * @param       {int[]}         postIds         list of post ids
		 */
		load: function (postIds) {
			Ajax.api(this, {
				objectIDs: postIds
			});
		},
		
		_ajaxSuccess: function(data) {
			var posts = elCreate('div');
			DomUtil.setInnerHtml(posts, data.returnValues.template);
			
			var lastPost, objectId, placeholder, post;
			while (posts.childElementCount) {
				post = posts.children[0];
				if (post.nodeName === 'LI') {
					lastPost = post;
				}
				else if (post.nodeName === 'SCRIPT' && lastPost) {
					lastPost.appendChild(post);
					continue;
				}
				
				objectId = elData(elBySel('.wbbPost', post), 'object-id');
				placeholder = elById('post' + objectId);
				placeholder.parentNode.insertBefore(post, placeholder);
				elRemove(placeholder);
			}
			
			_postManager.rebuild();
			if (_postInlineEditor !== null) {
				_postInlineEditor.rebuild();
			}
			
			ControllerClipboard.reload();
			DomChangeListener.trigger();
		},
		
		_ajaxSetup: function() {
			return {
				data: {
					actionName: 'getPosts',
					className: 'wbb\\data\\post\\PostAction',
					parameters: {
						showCollapsedPosts: true
					}
				}
			};
		}
	};
});
