/**
 * Inline editor for posts.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Ui/Post/InlineEditor
 */
define(['Core', 'Language', 'Ui/Confirmation', 'WoltLabSuite/Core/Ui/Message/InlineEditor', 'WoltLabSuite/Forum/Ui/Post/Enable', 'WoltLabSuite/Core/Dom/Util'], function(Core, Language, UiConfirmation, UiMessageInlineEditor, UiPostEnable, DomUtil) {
	"use strict";
	
	if (!COMPILER_TARGET_DEFAULT) {
		var Fake = function() {};
		Fake.prototype = {
			init: function() {},
			setPostManager: function() {},
			_showMessage: function() {},
			_dropdownGetItems: function() {},
			_dropdownOpen: function() {},
			_dropdownSelect: function() {},
			_getHash: function() {},
			rebuild: function() {},
			_click: function() {},
			_clickDropdown: function() {},
			_dropdownBuild: function() {},
			_dropdownToggle: function() {},
			_clickDropdownItem: function() {},
			_prepare: function() {},
			_showEditor: function() {},
			_restoreMessage: function() {},
			_save: function() {},
			_validate: function() {},
			throwError: function() {},
			_hideEditor: function() {},
			_restoreEditor: function() {},
			_destroyEditor: function() {},
			_updateHistory: function() {},
			_getEditorId: function() {},
			_getObjectId: function() {},
			_ajaxFailure: function() {},
			_ajaxSuccess: function() {},
			_ajaxSetup: function() {},
			legacyEdit: function() {}
		};
		return Fake;
	}
	
	var _postManager = null;
	
	/**
	 * @param       {int}           threadId        thread id
	 * @param       {Object}        options         list of inline editor options
	 * @constructor
	 */
	function UiPostInlineEditor(threadId, options) {
		this.init(threadId, options);
	}
	Core.inherit(UiPostInlineEditor, UiMessageInlineEditor, {
		/**
		 * @see WoltLabSuite/Core/Ui/Message/InlineEditor.init()
		 */
		init: function(threadId, options) {
			this._threadId = threadId;
			
			options = Core.extend({
				canEditInline: false,
				disableEdit: false
			}, options);
			
			this._disableEdit = options.disableEdit;
			
			UiPostInlineEditor._super.prototype.init.call(this, {
				canEditInline: options.canEditInline,
				className: 'wbb\\data\\post\\PostAction',
				containerId: this._threadId,
				dropdownIdentifier: 'wbbPost',
				editorPrefix: 'postEditor',
				quoteManager: options.quoteManager || null
			});
		},
		
		/**
		 * Sets the post manager instance.
		 * 
		 * @param       {WoltLabSuite/Forum/Ui/Post/Manager}   postManager
		 */
		setPostManager: function(postManager) {
			_postManager = postManager;
		},
		
		_showMessage: function(data) {
			var activeElement = this._activeElement;
			
			UiPostInlineEditor._super.prototype._showMessage.call(this, data);
			
			for (var className in data.returnValues.notes) {
				if (data.returnValues.notes.hasOwnProperty(className)) {
					_postManager.setNote(this._getObjectId(activeElement), className, data.returnValues.notes[className]);
				}
			}
			
			var elementData = this._elements.get(activeElement);
			if (data.returnValues.threadFormOptions) {
				var threadFormOptions = elBySel('.threadFormOptions', elementData.messageBody);
				
				if (threadFormOptions !== null) {
					threadFormOptions.innerHTML = data.returnValues.threadFormOptions;
				}
				else {
					var div = elCreate('div');
					div.classList = 'threadFormOptions';
					div.innerHTML = data.returnValues.threadFormOptions;
					DomUtil.prepend(div, elementData.messageBody);
				}
			}
			else {
				var threadFormOptions = elBySel('.threadFormOptions', elementData.messageBody);
				if (threadFormOptions !== null) {
					threadFormOptions.remove();
				}
			}
		},
		
		/**
		 * @see WoltLabSuite/Core/Ui/Message/InlineEditor._dropdownGetItems()
		 */
		_dropdownGetItems: function() {
			var prefix = 'wbb.post.edit.';
			
			return [
				{ item: 'enable', label: prefix + 'enable' },
				{ item: 'disable', label: prefix + 'disable' },
				{ item: 'close', label: prefix + 'close' },
				{ item: 'open', label: prefix + 'open' },
				{ item: 'trash', label: prefix + 'delete' },
				{ item: 'restore', label: prefix + 'restore' },
				{ item: 'deleteCompletely', label: prefix + 'deleteCompletely' },
				{ item: 'divider' },
				{ item: 'editItem', label: 'wbb.post.edit' }
			];
		},
		
		/**
		 * @see WoltLabSuite/Core/Ui/Message/InlineEditor._dropdownOpen()
		 */
		_dropdownOpen: function() {
			var objectId = this._getObjectId(this._activeDropdownElement);
			
			var canClose = _postManager.getPermission(objectId, 'close');
			var canEnable = _postManager.getPermission(objectId, 'enable');
			
			var isClosed = _postManager.getPropertyValue(objectId, 'is-closed', true);
			var isDeleted = _postManager.getPropertyValue(objectId, 'is-deleted', true);
			var isDisabled = _postManager.getPropertyValue(objectId, 'is-disabled', true);
			
			return {
				close: (canClose && !isClosed),
				trash: (_postManager.getPermission(objectId, 'delete') && !isDeleted),
				deleteCompletely: (_postManager.getPermission(objectId, 'delete-completely') && isDeleted),
				disable: (canEnable && !isDisabled && !isDeleted),
				enable: (canEnable && isDisabled && !isDeleted ),
				open: (canClose && isClosed),
				restore: (_postManager.getPermission(objectId, 'restore') && isDeleted),
				editItem: (this._disableEdit !== true)
			};
		},
		
		/**
		 * @see WoltLabSuite/Core/Ui/Message/InlineEditor._dropdownSelect()
		 */
		_dropdownSelect: function(item) {
			var objectId = this._getObjectId(this._activeDropdownElement);
			var callback = function (parameters) {
				_postManager.update(objectId, item, parameters || {});
			};
			
			switch (item) {
				case 'deleteCompletely':
					UiConfirmation.show({
						confirm: callback,
						message: Language.get('wbb.post.edit.delete.confirmMessage')
					});
					break;
				
				case 'enable':
					UiPostEnable.init(objectId, callback);
					break;
				
				case 'trash':
					UiConfirmation.show({
						confirm: function () {
							callback({
								reason: elById('wbbPostTrashReason').value.trim()
							});
						},
						message: Language.get('wbb.post.edit.trash.confirmMessage'),
						template: '<dl>'
							+ '<dt><label for="wbbPostTrashReason">' + Language.get('wbb.post.edit.trash.reason') + '</label></dt>'
							+ '<dd><textarea id="wbbPostTrashReason" cols="40" rows="3"></textarea></dd>'
						+ '</dl>'
					});
					break;
				
				default:
					callback();
					break;
			}
		},
		
		/**
		 * @see WoltLabSuite/Core/Ui/Message/InlineEditor._getHash()
		 */
		_getHash: function(objectId) {
			return '#post' + objectId;
		}
	});
	
	return UiPostInlineEditor;
});
