/**
 * Handles toggling label groups based on a selected board.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Controller/Thread/LabelCondition
 */
define(['Dom/Traverse'], function(DomTraverse) {
	"use strict";
	
	if (!COMPILER_TARGET_DEFAULT) {
		var Fake = function() {};
		Fake.prototype = {
			_change: function() {},
			_submit: function() {}
		};
		return Fake;
	}
	
	/**
	 * @constructor
	 */
	function ControllerThreadLabelCondition(boardSelectorID, labelGroupMapping) {
		this._boardSelector = elById(boardSelectorID);
		this._boardSelector.addEventListener('change', this._change.bind(this));
		
		this._labelGroupMapping = labelGroupMapping;
		
		this._labelLists = elBySelAll('.labelList');
		if (this._labelLists.length === 0) return;
		
		this._change();
		
		DomTraverse.parentByTag(this._labelLists[0], 'FORM').addEventListener('submit', this._submit.bind(this));
	}
	ControllerThreadLabelCondition.prototype = {
		/**
		 * Toggles the visible label groups based on the selected board.
		 */
		_change: function() {
			var i, length;
			var boardIDs = [];
			if (this._boardSelector.hasAttribute('multiple')) {
				for (i = 0, length = this._boardSelector.options.length; i < length; i++) {
					var option = this._boardSelector.options[i];
					if (option.selected) {
						boardIDs.push(option.value);
					}
				}
			}
			else {
				boardIDs.push(~~this._boardSelector.value);
			}
			
			var labelGroupIDs = boardIDs.length ? this._labelGroupMapping[boardIDs[0]] || [] : [];
			if (labelGroupIDs.length) {
				for (i = 1, length = boardIDs.length; i < length; i++) {
					var localLabelGroupIDs = this._labelGroupMapping[boardIDs[i]] || [ ];
					if (!localLabelGroupIDs.length) {
						labelGroupIDs = [ ];
						break;
					}
					else {
						for (var j = 0, length2 = labelGroupIDs.length; j < length2; j++) {
							if (localLabelGroupIDs.indexOf(labelGroupIDs[j]) === -1) {
								delete labelGroupIDs[j];
							}
						}
					}
				}
			}
			
			for (i = 0, length = this._labelLists.length; i < length; i++) {
				var labelList = this._labelLists[i];
				var dl = DomTraverse.parentBySel(labelList, 'dl');
				if (labelGroupIDs.indexOf(~~elAttr(labelList, 'data-object-id')) === -1) {
					dl.style.setProperty('display', 'none', '');
				}
				else {
					dl.style.removeProperty('display');
				}
			}
		},
		
		/**
		 * Unsets selected labels for invisible label choosers.
		 */
		_submit: function() {
			for (var i = 0, length = this._labelLists.length; i < length; i++) {
				var labelList = this._labelLists[i];
				if (DomTraverse.parentBySel(labelList, 'dl').style.getPropertyValue('display') === 'none') {
					$(elByClass('labelChooser', labelList)[0]).data('labelID', 0);
				}
			}
		}
	};
	
	return ControllerThreadLabelCondition;
});
