/**
 * Initializes modules required for thread view.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Controller/Thread
 */
define(['WoltLabSuite/Core/Controller/Clipboard', 'WoltLabSuite/Core/Ui/Reaction/Handler', '../Ui/Post/InlineEditor', '../Ui/Post/Loader', '../Ui/Post/Manager'], function(ControllerClipboard, UiReactionHandler, UiPostInlineEditor, UiPostLoader, UiPostManager) {
	"use strict";
	
	var _likeHandler = null;
	var _postInlineEditor = null;
	var _postManager = null;
	var _threadId = 0;
	
	/**
	 * @exports	WoltLabSuite/Forum/Controller/Thread
	 */
	return {
		/**
		 * Initializes the thread view.
		 * 
		 * @param       {int}           threadId        thread id
		 * @param       {Object}        modules         boolean list of modules
		 * @param       {Object}        options         initialization options for required modules
		 */
		init: function(threadId, modules, options) {
			_threadId = threadId;
			
			// clipboard
			if (modules.clipboard) {
				ControllerClipboard.setup({
					hasMarkedItems: options.clipboard.hasMarkedItems,
					pageClassName: 'wbb\\page\\ThreadPage',
					pageObjectId: _threadId
				});
			}
			
			// like handler
			if (modules.postLikeHandler) this._initPostLikeHandler();
			
			if (modules.postManager) {
				// post manager
				_postManager = new UiPostManager(threadId);
				
				if (modules.postInlineEditor) {
					// post inline editor
					_postInlineEditor = new UiPostInlineEditor(_threadId, options.postInlineEditor);
					_postInlineEditor.setPostManager(_postManager);
				}
				
				if (modules.postLoader) {
					UiPostLoader.init(_postManager, _postInlineEditor);
				}
			}
		},
		
		/**
		 * Initializes the like handler for posts.
		 * 
		 * @private
		 */
		_initPostLikeHandler: function() {
			_likeHandler = new UiReactionHandler('com.woltlab.wbb.likeablePost', {
				// selectors
				containerSelector: '.wbbPost:not(.messageCollapsed)'
			});
		},
		
		/**
		 * Returns the like handler for posts or null if disabled.
		 * 
		 * @return      {(WoltLabSuite/Core/Ui/Reaction/Handler|null)}    like handler for posts or null if disabled
		 */
		getPostLikeHandler: function() {
			return _likeHandler;
		},
		
		/**
		 * Returns the InlineEditor instance for posts or null if disabled.
		 * 
		 * @return      {(WoltLabSuite/Forum/Ui/Post/InlineEditor|null)}       InlineEditor instance for posts
		 */
		getPostInlineEditor: function() {
			return _postInlineEditor;
		},
		
		/**
		 * Returns the Manager instance for posts or null if disabled.
		 * 
		 * @return      {(WoltLabSuite/Forum/Ui/Post/Manager|null)}    Manager instance for posts
		 */
		getPostManager: function() {
			return _postManager;
		}
	};
});
