/**
 * Handles toggling watched boards in the list of all watched boards.
 * 
 * @author	Matthias Schmidt
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Controller/Board/WatchedList
 */
define(['Ajax', 'Language', 'Dom/Traverse', 'Dom/Util', 'Ui/Confirmation'], function(Ajax, Language, DomTraverse, DomUtil, UiConfirmation) {
	"use strict";
	
	if (!COMPILER_TARGET_DEFAULT) {
		var Fake = function() {};
		Fake.prototype = {
			setup: function() {},
			_getSelectedBoardIDs: function() {},
			_mark: function() {},
			_markAll: function() {},
			_stopWatching: function() {},
			_updateActionButtonLabel: function() {},
			_updateMarkAllButton: function() {}
		};
		return Fake;
	}
	
	var _button = null;
	var _checkboxes = [];
	var _markAllButton = null;
	
	/**
	 * @exports	WoltLabSuite/Forum/Controller/Board/WatchedList
	 */
	return {
		/**
		 * Initializes the watched board list.
		 */
		setup: function() {
			var changeCallback = this._mark.bind(this);
			var boardTitles = elBySelAll('.wbbBoard .containerHeadline > h3');
			for (var i = 0, length = boardTitles.length; i < length; i++) {
				var boardTitle = boardTitles[i];
				
				var checkbox = elCreate('input');
				elAttr(checkbox, 'type', 'checkbox');
				elAttr(checkbox, 'data-object-id', DomTraverse.parentByClass(boardTitle, 'wbbBoardContainer').getAttribute('data-board-id'));
				checkbox.addEventListener('change', changeCallback);
				checkbox.style.setProperty('margin-right', '7px', '');
				
				_checkboxes.push(checkbox);
				
				DomUtil.prepend(checkbox, boardTitle);
			}
			
			_button = elById('stopWatchingButton');
			_button.addEventListener('click', this._stopWatching.bind(this));
			
			_markAllButton = elById('markAllWatchedBoards');
			_markAllButton.addEventListener('click', this._markAll.bind(this));
			elAttr(_markAllButton, 'data-has-unmarked-boards', 1);
		},
		
		/**
		 * Returns the ids of the selected watched boards.
		 * 
		 * @return	{int[]}         ids of selected boards
		 */
		_getSelectedBoardIDs: function() {
			var boardIDs = [];
			for (var i = 0, length = _checkboxes.length; i < length; i++) {
				var checkbox = _checkboxes[i];
				if (checkbox.checked) {
					boardIDs.push(checkbox.getAttribute('data-object-id'));
				}
			}
			
			return boardIDs;
		},
		
		/**
		 * Handles changing a mark checkbox.
		 * 
		 * @param	{object}	event	event object
		 */
		_mark: function(event) {
			this._updateActionButtonLabel();
			this._updateMarkAllButton(!event.currentTarget.checked ? true : undefined);
		},
		
		/**
		 * Handles clicking on the 'mark all' button.
		 */
		_markAll: function() {
			var check = elDataBool(_markAllButton, 'has-unmarked-boards');
			
			for (var i = 0, length = _checkboxes.length; i < length; i++) {
				_checkboxes[i].checked = check;
			}
			
			this._updateActionButtonLabel();
			this._updateMarkAllButton(!check);
		},
		
		/**
		 * Handles a click on the stop watching button.
		 */
		_stopWatching: function() {
			var boardIDs = this._getSelectedBoardIDs();
			var languageItem = 'wbb.board.watchedBoards.stopWatching' + (this._getSelectedBoardIDs().length ? 'Marked' : 'All') + '.confirmMessage';
			
			UiConfirmation.show({
				confirm: function() {
					Ajax.apiOnce({
						data: {
							actionName: 'stopWatching',
							className: 'wbb\\data\\board\\BoardAction',
							objectIDs: boardIDs
						},
						success: function() {
							window.location.reload();
						}
					});
				},
				message: Language.get(languageItem)
			});
		},
		
		/**
		 * Updates the label of the action button.
		 */
		_updateActionButtonLabel: function() {
			var boardIDs = this._getSelectedBoardIDs();
			
			if (boardIDs.length) {
				_button.textContent = Language.get('wbb.board.watchedBoards.stopWatchingMarked', {
					count: boardIDs.length
				});
			}
			else {
				_button.textContent = Language.get('wbb.board.watchedBoards.stopWatchingAll');
			}
		},
		
		/**
		 * Updates the label of mark all button.
		 * 
		 * @param	{boolean}	hasUnmarkedBoards	if undefined, the value will be calculated
		 */
		_updateMarkAllButton: function(hasUnmarkedBoards) {
			if (hasUnmarkedBoards === undefined) {
				hasUnmarkedBoards = false;
				for (var i = 0, length = _checkboxes.length; i < length; i++) {
					if (!_checkboxes[i].checked) {
						hasUnmarkedBoards = true;
						break;
					}
				}
			}
			
			elAttr(_markAllButton, 'data-has-unmarked-boards', ~~hasUnmarkedBoards);
			elBySel('span:not(.icon)', _markAllButton).textContent = Language.get('wbb.board.watchedBoards.' + (hasUnmarkedBoards ? '' : 'un') + 'markAll');
		}
	};
});
