/**
 * Provides the thread form option dialog.
 *
 * @author	Joshua Ruesweg
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Acp/Ui/Board/Thread/Form/Option/DialogHandler
 * @since       5.2
 */
define(['Ajax', 'EventHandler', 'Dictionary', 'Dom/Util', 'Language', 'Ui/Dialog', 'WoltLabSuite/Core/Language/Input'], 
function (Ajax, EventHandler, Dictionary, DomUtil, Language, UiDialog, LanguageInput) {
	"use strict";
	
	var _dialogLoaded, _hasI18nValues = false;
	var _saveCallback;
	var _data = null;
	var _title = '';
	
	/**
	 * @exports     WoltLabSuite/Forum/Acp/Ui/Board/Thread/Form/Option/DialogHandler
	 */
	return {
		/**
		 * Sets an internal flag variable whether the installation has multiple languages and therefore the i18n fields need to be loaded or not.
		 * 
		 * @param       {boolean}       i18n
		 */
		setI18n: function (i18n) {
			_hasI18nValues = i18n;
		},
		
		/**
		 * Returns true, if the form has i18n fields. 
		 * 
		 * @return      {boolean}
		 */
		hasI18n: function () {
			return _hasI18nValues;
		},
		
		/**
		 * Open the edit/add dialog for options.
		 * 
		 * @param       {Function}      saveCallback
		 * @param       {Object}        data
		 */
		loadDialog: function (saveCallback, data, title) {
			_saveCallback = saveCallback; 
			
			if (!_dialogLoaded) {
				Ajax.api(this);
				_data = data;
				_title = title;
			}
			else {
				if (data !== null && data !== undefined) {
					this.setData(data);
				}
				UiDialog.open(this);
				UiDialog.setTitle(this, title);
			}
		},
		
		/**
		 * Closes the current dialog. 
		 */
		closeDialog: function () {
			UiDialog.close(this);
		},
		
		/**
		 * Submit the current dialog and trigger the saveCallback, if there are no validation errors. 
		 * 
		 * @param       {Event}         event
		 */
		_submitDialog: function (event) {
			event.preventDefault();
			
			elBySel('input[type="submit"]', this.getDialogContent()).disabled = true;
			
			this._resetErrors();
			
			if (this._validateDialog()) {
				this._saveDialog();
			}
			else {
				elBySel('input[type="submit"]', this.getDialogContent()).disabled = false;
			}
		},
		
		/**
		 * Validates the current dialog. 
		 * 
		 * @returns     {boolean}
		 */
		_validateDialog: function () {
			var validationFailed = false;
			
			EventHandler.fire('com.woltlab.wbb.threadFormOptionDialogHandler', 'validateDialog', {
				dialogData: this._getDialogData(),
				validationFailed: validationFailed
			});
			
			if (!(this._getDialogData()['optionTitle'] !== undefined && this._getDialogData()['optionTitle'].trim() !== '') && (!_hasI18nValues || !LanguageInput.validate('optionTitle', false))) {
				this.throwError('optionTitle', Language.get('wcf.global.form.error.empty'));
				
				validationFailed = true;
			}
			
			if (_hasI18nValues && !LanguageInput.validate('optionDescription', true)) {
				this.throwError('optionDescription', Language.get('wcf.global.form.error.empty'));
				
				validationFailed = true;
			}
			
			return !validationFailed;
		},
		
		/**
		 * Throw an error in the current dialog. 
		 * 
		 * @param       {string}        errorFieldName
		 * @param       {string}        errorMessageText
		 */
		throwError: function (errorFieldName, errorMessageText) {
			var inputElement = elById(errorFieldName, this.getDialogContent());
			
			if (!inputElement) {
				throw new Error('Unknown errorFieldName `' + errorFieldName + '`');
			}
			
			var dlElement = inputElement.closest('dl');
			var ddElement = inputElement.closest('dd');
			dlElement.classList.add('formError');
			
			var errorMessage = elCreate('small');
			errorMessage.classList = 'innerError';
			errorMessage.innerHTML = errorMessageText;
			ddElement.appendChild(errorMessage);
		},
		
		/**
		 * Reset the errors in the current dialog.
		 */
		_resetErrors: function () {
			var element;
			var innerErrorElements = elBySelAll('.innerError', this.getDialogContent());
			var formErrorElements = elBySelAll('.formError', this.getDialogContent());
			
			for (var i = 0, length = innerErrorElements.length; i < length; i++) {
				element = innerErrorElements[i];
				element.remove();
			}
			
			for (var i = 0, length = formErrorElements.length; i < length; i++) {
				element = formErrorElements[i];
				element.classList.remove('formError');
			}
		},
		
		/**
		 * Save the current dialog und call the saveCallback given on the dialog open function. 
		 */
		_saveDialog: function () {
			if (typeof _saveCallback === 'function') {
				_saveCallback(this._getDialogData(), this.getDialogContent());
			}
		},
		
		/**
		 * Sets up the submit button in the current dialog.
		 */
		_setupSubmitButton: function () {
			var submitDiv = elCreate('div');
			submitDiv.className = 'formSubmit';
			var submitButton = elCreate('input');
			submitButton.type = 'submit';
			submitButton.value = Language.get('wcf.global.button.submit');
			submitButton.addEventListener('click', this._submitDialog.bind(this));
			submitDiv.appendChild(submitButton);
			
			var dialogContent = this.getDialogContent();
			dialogContent.appendChild(submitDiv)
		},
		
		/**
		 * Delete all dialog values in the current dialog. 
		 */
		cleanUpDialog: function () {
			var dialogContent = this.getDialogContent();
			
			EventHandler.fire('com.woltlab.wbb.threadFormOptionDialogHandler', 'cleanUpDialog', {
				dialogContent: dialogContent
			});
			
			elBySel('#showOrder', dialogContent).value = 0;
			elBySel('#defaultValue', dialogContent).value = '';
			elBySel('#validationPattern', dialogContent).value = '';
			elBySel('#required', dialogContent).checked = false;
			var optionType = elBySel('#optionType', dialogContent);
			elBySel('option[value="text"]', optionType).selected = true;
			elBySel('#selectOptions', dialogContent).value = '';
			elHide(elBySel('#selectOptionsDL', dialogContent));
			
			// reset i18n
			if (_hasI18nValues) {
				var optionTitleValues = LanguageInput.getValues('optionTitle');
				
				var dummyObject = {};
				optionTitleValues.forEach(function (value, key) {
					dummyObject[key] = '';
				});
				
				LanguageInput.setValues('optionTitle', dummyObject);
				LanguageInput.setValues('optionDescription', dummyObject);
			}
			else {
				elBySel('#optionTitle', dialogContent).value = '';
				elBySel('#optionDescription', dialogContent).value = '';
			}
			
			this._resetErrors();
			elBySel('input[type="submit"]', this.getDialogContent()).disabled = false;
		},
		
		/**
		 * Returns the current dialog content.
		 * 
		 * @returns     {*}
		 */
		getDialogContent: function () {
			return UiDialog.getDialog('threadFormOptionDialog').content;
		},
		
		/**
		 * Set the data for the current dialog. 
		 * 
		 * @param       {Array}         data
		 */
		setData: function (data) {
			EventHandler.fire('com.woltlab.wbb.threadFormOptionDialogHandler', 'setData', {
				data: data
			});
			
			Object.keys(data['simpleValues']).forEach(function (key) {
				elBySel('#'+key, this.getDialogContent()).value = data['simpleValues'][key];
			}.bind(this));
			
			Object.keys(data['checkboxValues']).forEach(function (key) {
				if (data['checkboxValues'][key] == 1) {
					elBySel('#'+key, this.getDialogContent()).checked = true;
				}
				else {
					elBySel('#'+key, this.getDialogContent()).checked = false;
				}
			}.bind(this));
			
			Object.keys(data['selectOption']).forEach(function (key) {
				elBySel('option[value="'+data['selectOption'][key]+'"]', elBySel('#'+key, this.getDialogContent())).selected = true; 
			}.bind(this));
			
			Object.keys(data['i18nValues']).forEach(function (key) {
				if (_hasI18nValues) {
					var dObject = Dictionary.fromObject(data['i18nValues'][key]);
					
					// in case the dialog has been loaded and the i18n fields are not yet initialized
					function setValue() {
						try {
							LanguageInput.setValues(key, dObject);
							clearTimeout(setValue);
						}
						catch(e) {
							setTimeout(setValue, 100);
						}
					}
					
					setValue();
				}
				else {
					elBySel('#'+key, this.getDialogContent()).value = data['i18nValues'][key];
				}
			}.bind(this));
			
			elBySel('#optionType', this.getDialogContent()).dispatchEvent(new Event('change'));
		},
		
		/**
		 * Returns the dialog data as object.
		 * 
		 * @return      {Object}
		 */
		_getDialogData: function () {
			var dialogData = {};
			var dialogContent = this.getDialogContent();
			var simpleValues = ['showOrder', 'defaultValue', 'validationPattern', 'selectOptions', 'optionType'];
			var checkboxValues = ['required', 'isDisabled'];
			var i18nValues = ['optionTitle', 'optionDescription'];
			
			EventHandler.fire('com.woltlab.wbb.threadFormOptionDialogHandler', 'getDialogData', {
				dialogData: dialogData,
				dialogContent: dialogContent,
				simpleValues: simpleValues,
				checkboxValues: checkboxValues,
				i18nValues: i18nValues
			});
			
			simpleValues.forEach(function (value) {
				dialogData[value] = elBySel('#'+value, dialogContent).value;
			});
			
			checkboxValues.forEach(function (value) {
				dialogData[value] = elBySel('#'+value, dialogContent).checked ? 1 : 0;
			});
			
			i18nValues.forEach(function (value) {
				if (_hasI18nValues) {
					var i18nData = LanguageInput.getValues(value);
					
					if (i18nData.get(0) !== undefined) {
						dialogData[value] = i18nData.get(0);
					}
					else {
						dialogData[value+'_i18n'] = {};
						i18nData.forEach(function (languageValue, key) {
							if (key === 0) return;
							
							dialogData[value+'_i18n'][key] = languageValue;
						})
					}
				}
				else {
					dialogData[value] = elBySel('#'+value, dialogContent).value;
				}
			});
			
			return dialogData;
		},
		
		_ajaxSuccess: function (data) {
			if (data.actionName === 'getDialogAddForm') {
				_dialogLoaded = true; 
				var fragment = DomUtil.createFragmentFromHtml(data.returnValues.template);
				
				// Hide `isDisabled` button, because disabling form elements are 
				// not supported by thread forms.
				elHide(elBySel('#isDisabled', fragment).closest('dl'));
				
				UiDialog.open(this, fragment);
				UiDialog.setTitle(this, _title);
				this._setupSubmitButton();
				if (_data !== null && data !== undefined) {
					this.setData(_data);
					_data = null;
				}
				UiDialog.rebuild(this);
			}
		},
		
		_ajaxSetup: function () {
			return {
				data: {
					className: 'wbb\\data\\thread\\form\\option\\ThreadFormOptionAction',
					actionName: 'getDialogAddForm'
				}
			};
		},
		
		_dialogSetup: function () {
			return {
				id: 'threadFormOptionDialog',
				options: {
					onClose: this.cleanUpDialog.bind(this)
				},
				source: null
			};
		}
	};
});
