/**
 * Provides the custom board icon designer.
 * 
 * @author	Alexander Ebert
 * @copyright	2001-2019 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @module	WoltLabSuite/Forum/Acp/Ui/Board/Icon
 */
define(['Core', 'Dictionary', 'Language', 'Ui/Dialog', 'WoltLabSuite/Core/Ui/Color/Picker', 'WoltLabSuite/Core/Ui/Style/FontAwesome'], function (Core, Dictionary, Language, UiDialog, UiColorPicker, UiStyleFontAwesome) {
	"use strict";
	
	var _color, _colorContainer, _content, _icon, _useColor;
	var _icons = new Dictionary();
	var _type = '';
	
	/**
	 * @exports     WoltLabSuite/Forum/Acp/Ui/Board/Icon
	 */
	return {
		/**
		 * Initializes the board icon designer.
		 */
		init: function () {
			var iconContainer = elById('iconContainer'), icon, type;
			elBySelAll('dl[data-type]', iconContainer, (function(container) {
				type = elData(container, 'type');
				
				elBySel('.button', container).addEventListener(WCF_CLICK_EVENT, this._click.bind(this, type));
				
				_icons.set(type, {
					container: container,
					icon: elBySel('.icon', container)
				});
			}).bind(this));
		},
		
		/**
		 * Opens the icon designer for selected type.
		 * 
		 * @param       {string}        type            type identifier
		 * @param       {Event}         event           event object
		 * @protected
		 */
		_click: function (type, event) {
			event.preventDefault();
			
			_type = type;
			
			UiDialog.open(this);
			UiDialog.setTitle(this, event.currentTarget.closest('dd').previousElementSibling.textContent.trim());
		},
		
		/**
		 * Sets the icon name.
		 * 
		 * @param       {string}        iconName        icon name
		 * @protected
		 */
		_setIcon: function (iconName) {
			_icon.textContent = iconName;
			
			this._renderIcon();
		},
		
		/**
		 * Sets the color, can be either a string or an object holding the
		 * individual r, g, b and a values.
		 * 
		 * @param       {(string|Object)}       color           color data
		 * @protected
		 */
		_setColor: function (color) {
			if (typeof color !== "string") {
				color = 'rgba(' + color.r + ', ' + color.g + ', ' + color.b + ', ' + color.a + ')';
			}
			
			elData(_color, 'color', color);
			_color.style.setProperty('background-color', color, '');
			
			this._toggleColor(undefined, true);
			
			this._renderIcon();
		},
		
		/**
		 * Toggles the custom color on or off.
		 * 
		 * @param       {Event}         event           event object
		 * @param       {boolean=}      toggleOn        force toggle the custom color usage
		 * @protected
		 */
		_toggleColor: function(event, toggleOn) {
			if (toggleOn !== undefined) {
				_useColor.checked = toggleOn;
			}
			
			this._renderIcon();
		},
		
		/**
		 * Renders the custom icon preview.
		 * 
		 * @protected
		 */
		_renderIcon: function () {
			var color = elData(_color, 'color');
			
			var icon = elBySel('.jsBoardIcon', _content);
			
			// set icon
			icon.className = icon.className.replace(/\b(fa-[a-z0-9\-]+)\b/, '');
			icon.classList.add('fa-' + _icon.textContent);
			
			if (_useColor.checked) {
				icon.style.setProperty('color', color, '');
				
				elShow(_colorContainer);
			}
			else {
				icon.style.removeProperty('color');
				
				elHide(_colorContainer);
			}
			
			var colorBoxValue = elBySel('.colorBoxValue', _content);
			colorBoxValue.style.setProperty('background-color', color, '');
			elData(colorBoxValue, 'color', color);
		},
		
		/**
		 * Reads or writes an icon property.
		 * 
		 * @param       {string}                property        property name
		 * @param       {(string|number)}       newValue        property value, anything that can be meaningful cast to a string
		 * @return      {string}                property value or undefined on read
		 * @protected
		 */
		_value: function (property, newValue) {
			var input = elBySel('input[name="iconData[' + _type + '][' + property + ']"]');
			if (newValue === undefined) {
				return (property === 'useColor') ? (~~input.value === 1) : input.value;
			}
			
			input.value = newValue;
		},
		
		/**
		 * Saves the custom icon design.
		 * 
		 * @param       {Event}         event           event object
		 * @protected
		 */
		_save: function(event) {
			event.preventDefault();
			
			var color = elData(_color, 'color');
			var icon = _icon.textContent;
			var useColor = _useColor.checked ? 1 : 0;
			
			this._value('color', color);
			this._value('icon', icon);
			this._value('useColor', useColor);
			
			var boardIcon = elBySel('.jsBoardIcon', _icons.get(_type).container);
			
			// set icon
			boardIcon.className = boardIcon.className.replace(/\b(fa-[a-z0-9\-]+)\b/, '');
			boardIcon.classList.add('fa-' + icon);
			
			if (useColor === 1) {
				boardIcon.style.setProperty('color', color, '');
			}
			else {
				boardIcon.style.removeProperty('color');
			}
			
			UiDialog.close(this);
		},
		
		_dialogSetup: function () {
			return {
				id: 'boardIconEditor',
				options: {
					onSetup: (function (context) {
						_content = context;
						
						_color = elBySel('.colorBoxValue', context);
						_icon = elBySel('.jsBoardIconName', context);
						_useColor = elBySel('.jsBoardIconUseColor', context);
						_colorContainer = elBySel('.jsBoardIconColorContainer', context);
						
						_useColor.addEventListener('change', this._toggleColor.bind(this));
						
						elBySel('.jsBoardIconName + .button', context).addEventListener(WCF_CLICK_EVENT, (function (event) {
							event.preventDefault();
							
							UiStyleFontAwesome.open(this._setIcon.bind(this));
						}).bind(this));
						
						var colorPicker = elBySel('.jsColorPicker', context);
						elBySel('.jsButtonOpenColorPicker', context).addEventListener(WCF_CLICK_EVENT, function (event) {
							event.preventDefault();
							
							Core.triggerEvent(colorPicker, WCF_CLICK_EVENT);
						});
						
						new UiColorPicker(colorPicker, {
							callbackSubmit: this._setColor.bind(this)
						});
						
						elBySel('.formSubmit > .buttonPrimary', context).addEventListener(WCF_CLICK_EVENT, this._save.bind(this));
					}).bind(this),
					onShow: (function () {
						this._setIcon(this._value('icon'));
						this._setColor(this._value('color'));
						this._toggleColor(undefined, this._value('useColor'));
					}).bind(this),
					title: Language.get('wbb.acp.board.icon')
				}
			};
		}
	};
});
