<?php
namespace calendar\system\event\listener;
use calendar\data\event\Event;
use calendar\data\event\EventAction;
use calendar\data\event\EventEditor;
use wbb\data\board\BoardCache;
use wbb\data\post\PostAction;
use wbb\data\thread\Thread;
use wbb\data\thread\ThreadAction;
use wcf\system\event\listener\IParameterizedEventListener;
use wcf\system\html\input\HtmlInputProcessor;
use wcf\system\language\LanguageFactory;
use wcf\system\tagging\TagEngine;
use wcf\system\WCF;

/**
 * Creates the event thread.
 * 
 * @author	Marcel Werk
 * @copyright	2001-2018 WoltLab GmbH
 * @license	WoltLab License <http://www.woltlab.com/license-agreement.html>
 * @package	WoltLabSuite\Calendar\System\Event\Listener
 */
class EventActionEventThreadListener implements IParameterizedEventListener {
	/**
	 * @inheritDoc
	 */
	public function execute($eventObj, $className, $eventName, array &$parameters) {
		/** @var EventAction $eventObj */
		
		switch ($eventObj->getActionName()) {
			case 'triggerPublication':
				// check board ids
				$board = null;
				$categoryIDs = [];
				if (CALENDAR_EVENT_THREAD_SINGLE_BOARD) {
					$board = BoardCache::getInstance()->getBoard(CALENDAR_EVENT_THREAD_BOARD_ID);
					if ($board === null || !$board->isBoard()) return;
				
					if (CALENDAR_EVENT_THREAD_CATEGORIES) {
						$categoryIDs = explode("\n", CALENDAR_EVENT_THREAD_CATEGORIES);
					}
				}
				
				foreach ($eventObj->getObjects() as $event) {
					$event = new EventEditor(new Event($event->eventID));
					/** @noinspection PhpUndefinedFieldInspection */
					if ($event->eventThreadID) continue;
					
					// check categories
					if (!empty($categoryIDs) || !CALENDAR_EVENT_THREAD_SINGLE_BOARD) {
						$result = false;
						
						if (CALENDAR_EVENT_THREAD_SINGLE_BOARD) {
							if (in_array($event->categoryID, $categoryIDs)) {
								$result = true;
							}
						}
						else {
							/** @noinspection PhpUndefinedFieldInspection */
							if ($event->getCategory()->eventThreadBoardID) {
								/** @noinspection PhpUndefinedFieldInspection */
								$board = BoardCache::getInstance()->getBoard($event->getCategory()->eventThreadBoardID);
								if ($board === null || !$board->isBoard()) {
									$board = null;
								}
								else {
									$result = true;
								}
							}
						}
						
						if (!$result) continue;
					}
					
					$language = WCF::getLanguage();
					if ($event->languageID) $language = LanguageFactory::getInstance()->getLanguage($event->languageID);
					// get tags
					$tags = [];
					if (MODULE_TAGGING) {
						$tagObjects = TagEngine::getInstance()->getObjectTags(
							'com.woltlab.calendar.event',
							$event->eventID,
							[$event->languageID === null ? LanguageFactory::getInstance()->getDefaultLanguageID() : ""]
						);
						foreach ($tagObjects as $tagObject) {
							$tags[] = $tagObject->getTitle();
						}
					}
					// save event thread
					$htmlInputProcessor = new HtmlInputProcessor();
					$htmlInputProcessor->process($language->getDynamicVariable('calendar.event.eventThread.message', ['event' => $event->getDecoratedObject()]), 'com.woltlab.wbb.post');
					$threadData = [
						'data' => [
							'boardID' => $board->boardID,
							'languageID' => $event->languageID,
							'topic' => $language->getDynamicVariable('calendar.event.eventThread.subject', ['event' => $event->getDecoratedObject()]),
							'time' => $event->time,
							'userID' => $event->userID,
							'username' => $event->username
						],
						'board' => $board,
						'postData' => [],
						'tags' => $tags,
						'htmlInputProcessor' => $htmlInputProcessor
					];
					$objectAction = new ThreadAction([], 'create', $threadData);
					$resultValues = $objectAction->executeAction();
					
					// update event thread id
					$event->update([
						'eventThreadID' => $resultValues['returnValues']->threadID
					]);
					
					// mark thread as read
					$threadAction = new ThreadAction([$resultValues['returnValues']], 'markAsRead');
					$threadAction->executeAction();
				}
				break;
				
			case 'update':
				foreach ($eventObj->getObjects() as $event) {
					if ($event->eventThreadID) {
						$thread = new Thread($event->eventThreadID);
						$post = $thread->getFirstPost();
						$event = new EventEditor(new Event($event->eventID));
						
						$language = WCF::getLanguage();
						if ($event->languageID) $language = LanguageFactory::getInstance()->getLanguage($event->languageID);
						
						// get tags
						$tags = [];
						if (MODULE_TAGGING) {
							$tagObjects = TagEngine::getInstance()->getObjectTags(
								'com.woltlab.calendar.event',
								$event->eventID,
								[$event->languageID === null ? LanguageFactory::getInstance()->getDefaultLanguageID() : ""]
							);
							foreach ($tagObjects as $tagObject) {
								$tags[] = $tagObject->getTitle();
							}
						}
						$threadAction = new ThreadAction([$thread], 'update', [
							'data' => [
								'tags' => $tags,
								'languageID' => $event->languageID,
								'topic' => $language->getDynamicVariable('calendar.event.eventThread.subject', ['event' => $event->getDecoratedObject()])
							]
						]);
						$threadAction->executeAction();
						$htmlInputProcessor = new HtmlInputProcessor();
						$htmlInputProcessor->process($language->getDynamicVariable('calendar.event.eventThread.message', ['event' => $event->getDecoratedObject()]), 'com.woltlab.wbb.post');
						$postAction = new PostAction([$post], 'update', [
							'htmlInputProcessor' => $htmlInputProcessor
						]);
						$postAction->executeAction();
					}
				}
				break;
				
			case 'disable':
				foreach ($eventObj->getObjects() as $event) {
					if ($event->eventThreadID) {
						$thread = new Thread($event->eventThreadID);
						$threadAction = new ThreadAction([$thread], 'disable');
						$threadAction->executeAction();
					}
				}
				break;
				
			case 'enable':
				foreach ($eventObj->getObjects() as $event) {
					if ($event->eventThreadID) {
						$thread = new Thread($event->eventThreadID);
						$threadAction = new ThreadAction([$thread], 'enable');
						$threadAction->executeAction();
					}
				}
				break;
				
			case 'trash':
				foreach ($eventObj->getObjects() as $event) {
					if ($event->eventThreadID) {
						$thread = new Thread($event->eventThreadID);
						$threadAction = new ThreadAction([$thread], 'trash');
						$threadAction->executeAction();
					}
				}
				break;
				
			case 'restore':
				foreach ($eventObj->getObjects() as $event) {
					if ($event->eventThreadID) {
						$thread = new Thread($event->eventThreadID);
						$threadAction = new ThreadAction([$thread], 'restore');
						$threadAction->executeAction();
					}
				}
				break;
				
			case 'delete':
				foreach ($eventObj->getObjects() as $event) {
					if ($event->eventThreadID) {
						$thread = new Thread($event->eventThreadID);
						$threadAction = new ThreadAction([$thread], 'delete');
						$threadAction->executeAction();
					}
				}
				break;
		}
	}
}
