<?php
namespace wcf\system\cronjob;
use wcf\system\bot\Bot;
//use wcf\system\bot\BotSystem;
use wcf\data\bot\action\log\BotActionLogEditor;
use wcf\data\bot\action\BotActionEditor;
use wcf\data\user\User;
use wcf\data\user\UserList;
use wcf\util\DateUtil;
use wcf\data\cronjob\Cronjob;
use wcf\system\WCF;
use wcf\util\StringUtil;
use wcf\system\event\EventHandler;
use wcf\data\option\OptionEditor;
use wcf\system\feed\reader\FeedReader;
use wcf\system\request\LinkHandler;


/**
 * execute bot events: birthdayOfUser, dateX, monthly, yearly, userHasOverXLikesReceived,
 * userHasOverXActivityPoints, userHasOverXProfileHits, userIsMemberXDays, feedreader
 *
 * @author	Kawas
 * @copyright	2013-2014 Kawas
 * @license	creative commons <http://creativecommons.org/licenses/by-sa/4.0/deed.de>
 * @package	com.kawas.wcf.bot
 * @category	Community Framework
 */
class BotCronjob extends AbstractCronjob {

  /**
   * activates multiply execution of events: birthdayOfUser, dateX, monthly, yearly, userHasOverXLikesReceived,
   * userHasOverXActivityPoints, userHasOverXProfileHits, userIsMemberXDays
   * @var boolean
   */
  const DEBUG_MODE = WCF_BOT_DEBUG_MODE;

  /**
   * execution start time of cronjob
   * @var integer
   */
  public $startExecutionTime;

  /**
   * max execution time
   * @var integer
   */
  public $maxExecutionTime = WCF_BOT_MAX_EXECUTION_TIME;


	/**
	 * @see	\wcf\system\cronjob\ICronjob::execute()
	 */
	public function execute(Cronjob $cronjob) {
		if (Bot::getInstance()->isEnabled()) {
			$this->startExecutionTime = microtime(true);
	      	$this->maxExecutionTime = $this->maxExecutionTime/1000;

			if(date('Y-m-d', TIME_NOW) != gmdate('Y-m-d', $cronjob->lastExec) OR self::DEBUG_MODE){

				// birthdayOfUser event
				$userOptionID = User::getUserOptionID('birthday');

				$sql = "SELECT	user.*, option_value.userOption" . $userOptionID . " as userage
					FROM	wcf".WCF_N."_user_option_value option_value
					  LEFT JOIN wcf".WCF_N."_user user
						ON user.userID = option_value.userID
					WHERE 	DAY(option_value.userOption" . $userOptionID . ") = DAY(NOW())
						AND MONTH(option_value.userOption" . $userOptionID . ") = MONTH(NOW())";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute();

				while ($row = $statement->fetchArray()) {
          			$parameters = Bot::createUserParameters(new User(null, $row));
					$parameters['userage'] = DateUtil::getAge($row['userage']);
					Bot::getInstance()->fireActions('birthdayOfUser', $parameters);
				}

				// dateX event
				$actions = Bot::getInstance()->getActions('dateX');
				foreach ($actions AS $action) {
					if ($action->getOption('execution_date') == gmdate('Y-m-d', TIME_NOW)) {
						$action->execute();
					}
				}


				$day = gmdate('d', TIME_NOW);
				$month = gmdate('m', TIME_NOW);

				// monthly event
				$actions = Bot::getInstance()->getActions('monthly');
				foreach ($actions AS $action) {
					if ($action->getOption('execution_day') == $day) $action->execute();
				}

				// yearly event
				$actions = Bot::getInstance()->getActions('yearly');
				foreach ($actions AS $action) {
					if ($action->getOption('execution_day') == $day AND $action->getOption('execution_month') == $month ) $action->execute();
				}
			}

			// userHasOverXLikesReceived event
			$this->userHasOverX('likes_received', 'likesReceived');

			// userHasOverXActivityPoints event
			$this->userHasOverX('activity_points', 'activityPoints');

			// userHasOverXProfileHits event
			$this->userHasOverX('profile_hits', 'profileHits');


			// userIsMemberXDays event
			$actions = Bot::getInstance()->getActions('userIsMemberXDays');

			foreach($actions as $action) {
				$action->registerEventParameters(array(
					'registration_date' => TIME_NOW - ($action->getOption('days') * 86400)
				));
			}

			$this->userHasOverX('registration_date', 'registrationDate', true, $actions);

			// userIsInactiveXDays event
			$actions = Bot::getInstance()->getActions('userIsInactiveXDays');

			foreach($actions as $action) {
				$action->registerEventParameters(array(
					'last_activity_time' => TIME_NOW - ($action->getOption('days') * 86400)
				));
			}

			$this->userHasOverX('last_activity_time', 'lastActivityTime', true, $actions);

		    // call execute event for addinational bot events
		    parent::execute($cronjob);

		    // feedReader event
		    $actions = Bot::getInstance()->getActions('feedReader');

		    foreach($actions as $action) {
		        $feed = FeedReader::getInstanceByUrl($action->getOption('feed_url'));
		        $items = $feed->getItems();
		        $items = array_reverse($items);

		        $sql = "SELECT hash FROM wcf".WCF_N."_bot_action_log
		                     WHERE actionID = ?
		                       ORDER BY actionLogID DESC";
		        $statement = WCF::getDB()->prepareStatement($sql, count($items) + 5);
		        $statement->execute(array($action->getObjectID()));

		        $hashDB = array();
		        while ($row = $statement->fetchArray()) {
		        	$hashDB[$row['hash']] = true;
		        }

		        foreach($items as $item){
		          $hash = $item->get_id(true);
		         	if(!isset($hashDB[$hash])) {
		            	if(!$action->getOption('ignore_old')) {

			              	// generate source link if bbcodes or html allowed
			              	$link = $item->get_link();
			             	if(!empty($link)) {
			                	if($action->getOption('enable_html', null, false)) {
			                  		$source = '<br /><br /><a href="' . $link . '">' . $feed->get_title() . '</a>';
			                	} elseif($action->getOption('enable_bbcodes', null, false)) {
			                  		$source = '\n\n[url=\'' . $link . '\']' . $feed->get_title() . '[/url]';
			                	}
			              	}

			              	$feedTitle = $feed->get_title();
			              	$parameters = array(
				                'subject' => (!empty($feedTitle) ? $feedTitle : '') . ': ' . $item->get_title(),
				                'message' =>  $item->get_content() . (isset($source) ? $source : ''),
				                'title' => $item->get_title(),
				                'content' => $item->get_content(),
				                'date' => $item->get_date(),
				                'link' => $item->get_link(),
				                'author' => (is_object($item->get_author()) ? $item->get_author()->get_name() : ''),
				                'channel' => $feed->get_title(),
			              	);

			              	$action->registerEventParameters($parameters);
			              	$action->execute();
		          		}

		           		BotActionLogEditor::create(array(
		  						'actionID' => $action->getObjectID(),
		  						'hash' => $hash
		  					));

		         	}

		          	if($this->isExecutionTimeExceeded()) return;
		        }

		        if($action->getOption('ignore_old')) {
		        	$actionEditor = new BotActionEditor($action);
		        	$actionEditor->updateOption('ignore_old', 0);
		        }
		    }

			// send messages to admins - community bot system (bot event)
			/*if (WCF_BOT_INSTALL_MESSAGE) {
				if (WCF_BOT_INSTALL_MESSAGE == 1) BotSystem::createInstallMessage();
				if (WCF_BOT_INSTALL_MESSAGE == 2) BotSystem::createUpdateMessage();

				// disable wcf_bot_install_message
				$sql = "UPDATE  wcf".WCF_N."_option
							SET optionValue = ?
							WHERE optionName = ?";

				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute(array(0, 'wcf_bot_install_message'));
				OptionEditor::resetCache();
			}*/ 

		}
	}

	/**
	 * general function for userHasOverX events
	 *
	 * @param string	$optionName
	 * @param string 	$subject
	 * @param boolean 	$reverseOperator
	 * @param object	array</wcf/system/bot/action/type/IBotActionType> $actions
	 */
	public function userHasOverX($optionName, $subject, $reverseOperator = false, $actions = null){
		if($this->isExecutionTimeExceeded()) return;

		if($actions == null) {
			$actions = Bot::getInstance()->getActions('userHasOverX' . StringUtil::firstCharToUpperCase($subject));
		}

		if(count($actions) == 0) return;

		foreach ($actions AS $action) {
			$sql = "SELECT user.*, action_log.actionID
						FROM wcf".WCF_N."_user user
							LEFT JOIN wcf".WCF_N."_bot_action_log action_log
						ON action_log.userID = user.userID AND action_log.actionID = ?
							WHERE user." .$subject. " " .($reverseOperator  ? "<=" : " >="). " ? ".
								(self::DEBUG_MODE  ? "" : " AND actionID IS NULL");


			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute(array($action->actionID, intval($action->getOption($optionName))));

			while ($row = $statement->fetchArray()) {
        $user = new User(null, $row);

				if($user->userID == Bot::getInstance()->userID) continue;

				if(!self::DEBUG_MODE) {
					BotActionLogEditor::create(array(
						'actionID' => $action->actionID,
						'userID' => $user->userID
					));
				}

        if(!$action->getOption('ignore_old')) {
  				$action->registerUserEventParameters($user);
  				$action->execute();
        }

				if($this->isExecutionTimeExceeded()) return;
			}

      if($action->getOption('ignore_old')) {
        $actionEditor = new BotActionEditor($action);
        $actionEditor->updateOption('ignore_old', 0);
      }
		}
	}

	/**
	 * return true if execution time is exceeded and force a new cronjob execution
	 *
	 * @return boolean
	 */
	public function isExecutionTimeExceeded() {
		if((microtime(true) - $this->startExecutionTime) > $this->maxExecutionTime){
			Bot::getInstance()->forcingCronjob();
			return true;
		} else {
			return false;
		}
	}

}
