<?php
namespace wcf\system\bot;
use wcf\system\WCF;
use wcf\system\SingletonFactory;
use wcf\system\cache\builder\BotCacheBuilder;
use wcf\data\user\User;
use wcf\system\request\LinkHandler;
use wcf\data\cronjob\Cronjob;
use wcf\data\cronjob\CronjobEditor;

/**
 * Represents the community bot
 *
 * @author	Kawas
 * @copyright	2013-2014 Kawas
 * @license	creative commons <http://creativecommons.org/licenses/by-sa/4.0/deed.de>
 * @package	com.kawas.wcf.bot
 * @category	Community Framework
 */
class Bot extends SingletonFactory{

	/**
	 * actions
	 * @var	array<\wcf\data\bot\action\BotAction>
	 */
	public $actionList;

	/**
	 * actions types
	 * @var	array<\wcf\data\bot\action\type\BotActionType>
	 */
	public $actionTypeList;

	/**
	 * initialized actions types
	 * @var	array<boolean>
	 */
	public $initActionTypes;

	/**
	 * status of the bot
	 * @var	boolean
	 */
	protected $isEnabled = WCF_MODULE_BOT;

	/**
	 * install date of community bot
	 * @var	integer
	 */
	public $installDate = WCF_BOT_INSTALL_DATE;

	/**
	 * options of the actions
	 * @var	array<array>
	 */
	public $options;

	/**
	 * user object of bot
	 * @var	wcf\data\user\User
	 */
	public $userObj;

	/**
	 * user object of current user for data restore
	 * @var	wcf\data\user\User
	 */
	public $currentUserObj;

	/**
	 * if true force a new cronjob excution
	 * @var	boolean
	 */
	protected $forcingCronjob = false;


	const USERNAME = 'Community Bot';

	/**
	 * @see	\wcf\system\SingletonFactory::init()
	 */
	protected function init() {
		$this->userObj = new User(WCF_BOT_USERID);

		if($this->userID == 0) $this->isEnabled = false;

		if($this->isEnabled()){
			// the bot needs a clone of the current user object to restore the user
			// after manipulation of the user object (bugfix 1.0.6)
			$this->currentUserObj = clone WCF::getUser();

			$cache = BotCacheBuilder::getInstance();
			$this->actionList = $cache->getData(array(), 'actionList');
			$parameters = array(
				'page_title' => PAGE_TITLE,
				'bot_user_id' => $this->userID,
				'bot_username' => $this->username,
				'bot_user_link' => LinkHandler::getInstance()->getLink('User', array(
					'object' => $this->userObj,
					'isEmail' => true
				))
			);

			foreach($this->actionList->objects as $action) {
				$action->registerEventParameters($parameters);
			}

			$this->actionTypeList = $cache->getData(array(), 'actionTypeList');
			$objects = $this->actionTypeList->objects;
			$this->actionTypeList->objects = array();
			foreach($objects as $object) {
				$this->actionTypeList->objects[$object->actionTypeName] = new $object->className(null, null, $object);
			}

			$this->options = $cache->getData(array(), 'options');
			$this->initActionTypes = array();
		}
	}

	/**
	 * Returns vars of user object
	 *
	 * @param	string		$varName
	 * @return	mixed
	 */
	public function __get($varName) {
		return $this->userObj->__get($varName);
	}

	/**
	 * manipulates user data for action classes with user specific user content
	 */
	public function manipulateUser() {
		WCF::getUser()->__construct(null, null, $this->userObj);
	}

	/**
	 * Restores user data
	 */
	public function restoreUser() {
		WCF::getUser()->__construct(null, null, $this->currentUserObj);
	}

	/**
	 * Returns true if bot is enabled.
	 *
	 * @param	string		$eventName
	 * @return	boolean
	 */
	public function isEnabled($eventName = '') {
		if (!$this->isEnabled) return false;
		if (!empty($eventName) AND count($this->getActions($eventName)) == 0) return false;

		return true;
	}

	/**
	 * Initialize action type.
	 *
	 * @param	string		$actionTypeName
	 */
	public function initActionType($actionTypeName) {
		if(!isset($this->initActionTypes[$actionTypeName])){
			$this->initActionTypes[$actionTypeName] = true;
			$this->actionTypeList->objects[$actionTypeName]->botObj = $this;
			$this->actionTypeList->objects[$actionTypeName]->init();
		}
	}

	/**
	 * Executes all actions with the given eventname.
	 *
	 * @param	array		$eventName
	 * @param	array		$parameters
	 */
	public function fireActions($eventName, array $parameters = null) {
		$this->manipulateUser();

		foreach($this->actionList->objects as $action) {
			if($action->eventName == $eventName AND !$action->isDisabled) {
				$this->initActionType($action->actionTypeName);
				if($parameters !== null) $action->registerEventParameters($parameters);
				$action->execute(false);
			}
		}

		$this->restoreUser();
	}

	/**
	 * Returns all actions with the given eventname.
	 *
	 * @param	string		$actionTypeName
	 * @return 	array<\wcf\data\bot\action\BotAction>
	 */
	public function getActions($eventName) {
		$actions = array();

		foreach($this->actionList->objects as $action) {
			if($action->eventName == $eventName AND !$action->isDisabled AND (!$action->isRisky OR WCF_BOT_SECURITY_MODE_DISABLED)) {
				$this->initActionType($action->actionTypeName);
				$actions[] = $action;
			}
		}

		return $actions;
	}

	/**
	 * Execute shutdown methode of all initialized actions and force a new cronjob excution now
	 */
	public function __destruct(){
		if($this->isEnabled()) {
			// execute shutdown methods of loaded actions
			$this->manipulateUser();

			foreach($this->actionTypeList->objects AS $actionTypeName => $action) {
				if (isset($this->initActionTypes[$actionTypeName]) AND !$action->isDisabled){
					$action->shutdownExecute();
				}
			}

			$this->restoreUser();

			// Force a new cronjob excution now if forcingCronjob is true
			if($this->forcingCronjob) {
				$sql = "SELECT	*
					FROM	wcf".WCF_N."_cronjob
					WHERE	className = ?";
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute(array('wcf\system\cronjob\BotCronjob'));

				$row = $statement->fetchArray();
				$cronjob = new Cronjob(null, $row);
				$cronjobEditor = new CronjobEditor($cronjob);
				$cronjobEditor->update(array(
					'nextExec' => TIME_NOW,
					'failCount' => 0
				));

				CronjobEditor::resetCache();
			}
		}
	}

	/**
	 * Creates the parameters user_id, username and user_link with a user pobject
	 *
	 * @param	\wcf\data\user\User		$user
	 * @return array<string>
	 */
	public static function createUserParameters(User $user){
		return array(
			'user_id' => $user->userID,
			'username' => $user->username,
			'user_link' => LinkHandler::getInstance()->getLink('User', array(
				'object' => $user,
				'isEmail' => true
			))
		);
	}

	/**
	 * Force a new cronjob excution
	 */
	public function forcingCronjob() {
		$this->forcingCronjob = true;
	}

}
